/*
 * Decompiled with CFR 0.152.
 */
package ai.digital.deploy.gitops.util;

import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class YamlHandler {
    private static final Logger logger = LoggerFactory.getLogger(YamlHandler.class);

    public static List<YamlContent> scanAndReadYamlFiles(String repositoryPath) {
        logger.info("Scanning for YAML files in repository: {}", (Object)repositoryPath);
        ArrayList<YamlContent> yamlContents = new ArrayList<YamlContent>();
        try {
            List<String> yamlFiles = YamlHandler.scanForYamlFiles(repositoryPath);
            logger.info("Found {} YAML files to process", (Object)yamlFiles.size());
            for (String yamlFilePath : yamlFiles) {
                YamlContent yamlContent = YamlHandler.readYamlFile(yamlFilePath);
                yamlContents.add(yamlContent);
                if (yamlContent.isSuccess()) {
                    logger.debug("Successfully read YAML file: {}", (Object)yamlFilePath);
                    continue;
                }
                logger.warn("Failed to read YAML file: {} - {}", (Object)yamlFilePath, (Object)yamlContent.getErrorMessage());
            }
        }
        catch (Exception e) {
            logger.error("Error during YAML scanning and reading: {}", (Object)e.getMessage(), (Object)e);
            yamlContents.add(YamlContent.failure(repositoryPath, "Failed to scan repository: " + e.getMessage()));
        }
        long successCount = yamlContents.stream().filter(YamlContent::isSuccess).count();
        logger.info("YAML scanning complete: {} successful, {} failed out of {} total files", new Object[]{successCount, (long)yamlContents.size() - successCount, yamlContents.size()});
        return yamlContents;
    }

    private static YamlContent readYamlFile(String yamlFilePath) {
        try {
            String content = Files.readString(Paths.get(yamlFilePath, new String[0]));
            if (content == null || content.trim().isEmpty()) {
                logger.debug("YAML file is empty: {}", (Object)yamlFilePath);
                return YamlContent.success(yamlFilePath, "");
            }
            return YamlContent.success(yamlFilePath, content);
        }
        catch (IOException e) {
            logger.error("Failed to read YAML file: {} - {}", (Object)yamlFilePath, (Object)e.getMessage());
            return YamlContent.failure(yamlFilePath, "IO Error: " + e.getMessage());
        }
        catch (Exception e) {
            logger.error("Unexpected error reading YAML file: {} - {}", (Object)yamlFilePath, (Object)e.getMessage());
            return YamlContent.failure(yamlFilePath, "Unexpected error: " + e.getMessage());
        }
    }

    public static List<String> scanForYamlFiles(String directoryPath) throws IOException {
        logger.info("Scanning directory for YAML files: {}", (Object)directoryPath);
        Path startPath = Paths.get(directoryPath, new String[0]);
        if (!Files.exists(startPath, new LinkOption[0]) || !Files.isDirectory(startPath, new LinkOption[0])) {
            logger.warn("Directory does not exist or is not a directory: {}", (Object)directoryPath);
            return new ArrayList<String>();
        }
        List<Object> yamlFiles = new ArrayList();
        try (Stream<Path> paths = Files.walk(startPath, new FileVisitOption[0]);){
            yamlFiles = paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> {
                String fileName = path.getFileName().toString().toLowerCase();
                return fileName.endsWith(".yml") || fileName.endsWith(".yaml");
            }).map(Path::toAbsolutePath).map(Path::toString).collect(Collectors.toList());
        }
        catch (IOException e) {
            logger.error("Error scanning directory for YAML files: {}", (Object)directoryPath, (Object)e);
            throw e;
        }
        logger.info("Found {} YAML files in directory: {}", (Object)yamlFiles.size(), (Object)directoryPath);
        if (logger.isDebugEnabled()) {
            yamlFiles.forEach(file -> logger.debug("Found YAML file: {}", file));
        }
        return yamlFiles;
    }

    public static String getRelativePathForDisplay(String repositoryPath, String filePath) {
        if (repositoryPath != null && filePath != null && filePath.startsWith(repositoryPath)) {
            return filePath.substring(repositoryPath.length()).replaceFirst("^[/\\\\]", "");
        }
        return filePath;
    }

    public static String getRelativePath(String basePath, String filePath) {
        try {
            Path base = Paths.get(basePath, new String[0]).toAbsolutePath();
            Path file = Paths.get(filePath, new String[0]).toAbsolutePath();
            return base.relativize(file).toString();
        }
        catch (Exception e) {
            logger.debug("Could not get relative path for {} from {}", (Object)filePath, (Object)basePath);
            return filePath;
        }
    }

    public static String extractRepositoryName(String url) {
        try {
            String cleanUrl;
            String string = cleanUrl = url.endsWith(".git") ? url.substring(0, url.length() - 4) : url;
            String repoName = cleanUrl.contains("/") ? cleanUrl.substring(cleanUrl.lastIndexOf("/") + 1) : (cleanUrl.contains(":") ? cleanUrl.substring(cleanUrl.lastIndexOf(":") + 1) : cleanUrl);
            return repoName.replaceAll("[^a-zA-Z0-9_-]", "_");
        }
        catch (Exception e) {
            logger.warn("Failed to extract repository name from URL: {}", (Object)url, (Object)e);
            return "unknown-repo";
        }
    }

    public static String extractOrganizationName(String url) {
        try {
            String pathPart;
            String[] parts;
            String cleanUrl = url.toLowerCase().trim();
            cleanUrl = cleanUrl.replaceFirst("^https?://", "");
            cleanUrl = cleanUrl.replaceFirst("^git@", "");
            cleanUrl = cleanUrl.replaceFirst("^[^@]*@", "");
            if (cleanUrl.contains(":") && !cleanUrl.startsWith("http") && (parts = cleanUrl.split(":")).length > 1 && (pathPart = parts[1]).contains("/")) {
                String orgName = pathPart.substring(0, pathPart.indexOf("/"));
                return YamlHandler.sanitizeDirectoryName(orgName);
            }
            if (cleanUrl.contains("/")) {
                String[] segments = cleanUrl.split("/");
                for (int i = 0; i < segments.length - 1; ++i) {
                    String segment = segments[i];
                    if (segment.contains(".") || segment.isEmpty()) continue;
                    return YamlHandler.sanitizeDirectoryName(segment);
                }
            }
            return null;
        }
        catch (Exception e) {
            logger.debug("Failed to extract organization name from URL: {}", (Object)url, (Object)e);
            return null;
        }
    }

    public static String sanitizeDirectoryName(String name) {
        if (name == null || name.trim().isEmpty()) {
            return "unknown";
        }
        return name.replaceAll("[^a-zA-Z0-9_-]", "_");
    }

    public static String createOrgPrefixedRepositoryName(String url) {
        try {
            String repoName = YamlHandler.extractRepositoryName(url);
            String orgName = YamlHandler.extractOrganizationName(url);
            if (orgName != null && !orgName.trim().isEmpty() && !orgName.equals("unknown")) {
                return repoName + "_" + orgName;
            }
            return repoName;
        }
        catch (Exception e) {
            logger.warn("Failed to create org-prefixed repository name for URL: {}", (Object)url, (Object)e);
            return "unknown-repo";
        }
    }

    public static String createGitRepoFolderName(String gitDirectoryId, String username, String url, String branch) {
        String repoName = YamlHandler.extractRepositoryName(url);
        String sanitizedUsername = YamlHandler.sanitizeDirectoryName(username != null ? username : "unknown");
        String sanitizedBranch = YamlHandler.sanitizeDirectoryName(branch != null ? branch : "main");
        String idHash = YamlHandler.generateShortHash(gitDirectoryId);
        return String.format("Folder_%s_%s_%s_%s", idHash, sanitizedUsername, repoName, sanitizedBranch);
    }

    public static String generateShortHash(String input) {
        if (input == null || input.isEmpty()) {
            return "00000000";
        }
        int hash = input.hashCode();
        return String.format("%08x", (long)hash & 0xFFFFFFFFL);
    }

    public static class YamlContent {
        private final String filePath;
        private final String content;
        private final boolean success;
        private final String errorMessage;

        public YamlContent(String filePath, String content, boolean success, String errorMessage) {
            this.filePath = filePath;
            this.content = content;
            this.success = success;
            this.errorMessage = errorMessage;
        }

        public String getFilePath() {
            return this.filePath;
        }

        public String getContent() {
            return this.content;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public static YamlContent success(String filePath, String content) {
            return new YamlContent(filePath, content, true, null);
        }

        public static YamlContent failure(String filePath, String errorMessage) {
            return new YamlContent(filePath, null, false, errorMessage);
        }
    }
}

