package com.xebialabs.deployit.plugins.lock;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.RetryableStep;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.sql.lock.CiLockRepositoryHolder;
import org.springframework.dao.DeadlockLoserDataAccessException;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;

public class AcquireAllLocksStep implements RetryableStep {

    private static final int ACQUIRE_LOCKS_ORDER = 2;
    private final Set<ConfigurationItem> cisToBeLocked;
    private final boolean enableRetry;
    private final int retryInSeconds;
    private final int retryLimit;
    private final boolean asyncRetry;

    public AcquireAllLocksStep(Set<ConfigurationItem> cisToBeLocked, boolean enableRetry, int retryInSeconds,
                               int retryLimit, boolean asyncRetry) {
        this.cisToBeLocked = cisToBeLocked;
        this.enableRetry = enableRetry;
        this.retryInSeconds = retryInSeconds;
        this.retryLimit = retryLimit;
        this.asyncRetry = asyncRetry;
    }

    @Override
    public StepExitCode execute(ExecutionContext context) throws Exception {
        ReentrantLock taskLock = TaskLockManager.getLock(context.getTask().getId());
        try {
            taskLock.lock();
            context.logOutput("Attempting to acquire locks on CIs " + cisToBeLocked);
            boolean locked = CiLockRepositoryHolder.getCiLockRepository().lock(cisToBeLocked, context.getTask().getId());
            if (!locked && enableRetry) {
                if (asyncRetry) {
                    return StepExitCode.RETRY;
                } else {
                    locked = blockingRetry(context);
                }
            }
            if (locked) {
                context.logOutput("All locks acquired");
                context.setAttribute("lockCleanupListener", new LockCleanupListener(cisToBeLocked));
                return StepExitCode.SUCCESS;
            } else {
                context.logError("Failed to acquire one or more locks");
                return StepExitCode.PAUSE;
            }
        } catch (DeadlockLoserDataAccessException deadlockLoserDataAccessException) {
            context.logError("DeadlockLoserDataAccessException occurred while acquiring locks on CIs " +
                    cisToBeLocked + ". Will retry in " + retryInSeconds + " seconds.",
                    deadlockLoserDataAccessException);
            Thread.sleep(retryInSeconds * 1000L);
            return StepExitCode.RETRY;
        } finally {
            taskLock.unlock();
        }
    }

    private boolean blockingRetry(ExecutionContext context) throws Exception {
        int retryCount = 0;
        boolean locked = false;
        while (!locked && retryCount < retryLimit) {
            context.logOutput("Will retry in " + retryInSeconds + " seconds");
            Thread.sleep(retryInSeconds * 1000L);
            context.logOutput("Attempting to acquire locks on CIs " + cisToBeLocked);
            locked = CiLockRepositoryHolder.getCiLockRepository().lock(cisToBeLocked, context.getTask().getId());
            retryCount++;
        }
        return locked;
    }

    @Override
    public String getDescription() {
        return "Acquiring locks for the following CIs: " + cisToBeLocked.toString();
    }

    @Override
    public int getOrder() {
        return ACQUIRE_LOCKS_ORDER;
    }

    @Override
    public Optional<Long> getStepRetryDelay(ExecutionContext context) {
        if (asyncRetry) {
            return Optional.of(retryInSeconds * 1000L);
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<Integer> getStepRetryCount(ExecutionContext context) {
        if (asyncRetry) {
            return Optional.of(retryLimit);
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Optional<StepExitCode> stepRetryFailed(ExecutionContext context) {
        if (asyncRetry) {
            context.logError("Failed to acquire one or more locks");
            return Optional.of(StepExitCode.PAUSE);
        } else {
            return Optional.empty();
        }
    }
}
