package com.xebialabs.deployit.plugin.powershell;

import java.util.Collection;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.NOOP;
import static java.lang.String.format;

public final class PowerShellBatchContributor {

    private PowerShellBatchContributor() {}

    public static final int MAX_DESCRIPTION_LENGTH = 50;

    @Contributor
    public static void generateCombinedSteps(Deltas deltas, DeploymentPlanningContext ctx) {
        logger.debug("Begin");
        Multimap<HostContainer, Delta> deltasPerContainer = getDeltasPerContainer(deltas);
        generateStepsPerContainer(deltasPerContainer, ctx);
        logger.debug("End");
    }

    private static Multimap<HostContainer, Delta> getDeltasPerContainer(Deltas deltas) {
        logger.debug("Finding deltas to batch");

        Multimap<HostContainer, Delta> deltasPerContainer = ArrayListMultimap.create();

        for (Delta d : deltas.getDeltas()) {
            Deployed<?, ?> deployed = getActiveDeployed(d);
            if (deployed instanceof BaseExtensiblePowerShellDeployed<?>) {
                BaseExtensiblePowerShellDeployed<?> pd = (BaseExtensiblePowerShellDeployed<?>) deployed;
                if (pd.shouldBatch()) {
                    logger.debug("Adding BaseExtensiblePowerShellDeployed [{}] to batch", pd.getId());
                    deltasPerContainer.put(pd.getContainer(), d);
                } else {
                    logger.debug("NOT adding BaseExtensiblePowerShellDeployed [{}] to batch", pd.getId());
                }
            }
        }
        logger.debug("Found " + deltasPerContainer.size() + " deltas to batch");
        return deltasPerContainer;
    }

    private static void generateStepsPerContainer(Multimap<HostContainer, Delta> deltasPerContainer, DeploymentPlanningContext ctx) {
        logger.debug("Generating batch steps");
        for (HostContainer container : deltasPerContainer.keySet()) {
            Collection<Delta> deltas = deltasPerContainer.get(container);
            generateStepsForContainer(container, deltas, ctx);
        }
    }

    private static void generateStepsForContainer(HostContainer container, Collection<Delta> deltas, DeploymentPlanningContext ctx) {
        logger.debug("Generating batch steps for container [{}]", container.getId());
        ListMultimap<BatchGroup, BatchElement> batchGroups = ArrayListMultimap.create();
        int maxBatchSize = generateSingleSteps(deltas, ctx, batchGroups);
        generateBatchSteps(container, ctx, batchGroups, maxBatchSize);
    }

    private static int generateSingleSteps(Collection<Delta> deltas, DeploymentPlanningContext ctx, ListMultimap<BatchGroup, BatchElement> batchGroups) {
        logger.debug("Generating component steps");
        int maxBatchSize = -1;
        for (Delta d : deltas) {
            BaseExtensiblePowerShellDeployed<?> pd = (BaseExtensiblePowerShellDeployed<?>) getActiveDeployed(d);
            if (maxBatchSize == -1 || pd.maxBatchSize < maxBatchSize) {
                maxBatchSize = pd.maxBatchSize;
                logger.debug("Setting maxBatchSize={}", maxBatchSize);
            }

            if ((d.getOperation() == DESTROY && pd.stopOnDestroy) || (d.getOperation() == MODIFY && pd.stopStartOnModify) || (d.getOperation() == NOOP && pd.stopStartOnNoop)) {
                if (!isNullOrEmpty(pd.stopScript)) {
                    BatchGroup b = new BatchGroup(pd.stopOrder, pd.stopVerb);
                    BatchElement e = new BatchElement(pd.createStep(ctx, d, pd.stopOrder, pd.stopScript, pd.stopVerb, pd.stopOptions));
                    logger.debug("Adding {} to {} for stop", e, b);
                    batchGroups.put(b, e);
                }
            }
            if (d.getOperation() == DESTROY || (d.getOperation() == MODIFY && isNullOrEmpty(pd.modifyScript))) {
                if (!isNullOrEmpty(pd.destroyScript)) {
                    BatchGroup b = new BatchGroup(pd.destroyOrder, pd.destroyVerb);
                    BatchElement e = new BatchElement(pd.createStep(ctx, d, pd.destroyOrder, pd.destroyScript, pd.destroyVerb, pd.destroyOptions), d, DESTROY);
                    logger.debug("Adding {} to {} for destroy", e, b);
                    batchGroups.put(b, e);
                }
            }
            if (d.getOperation() == CREATE || (d.getOperation() == MODIFY && isNullOrEmpty(pd.modifyScript))) {
                if (!isNullOrEmpty(pd.createScript)) {
                    BatchGroup b = new BatchGroup(pd.createOrder, pd.createVerb);
                    BatchElement e = new BatchElement(pd.createStep(ctx, d, pd.createOrder, pd.createScript, pd.createVerb, pd.createOptions), d, CREATE);
                    logger.debug("Adding {} to {} for create", e, b);
                    batchGroups.put(b, e);
                }
            }
            if (d.getOperation() == MODIFY) {
                if (!isNullOrEmpty(pd.modifyScript)) {
                    BatchGroup b = new BatchGroup(pd.modifyOrder, pd.modifyVerb);
                    BatchElement e = new BatchElement(pd.createStep(ctx, d, pd.modifyOrder, pd.modifyScript, pd.modifyVerb, pd.modifyOptions), d, MODIFY);
                    logger.debug("Adding {} to {} for modify", e, b);
                    batchGroups.put(b, e);
                }
            }
            if ((d.getOperation() == CREATE && pd.startOnCreate) || (d.getOperation() == MODIFY && pd.stopStartOnModify) || (d.getOperation() == NOOP && pd.stopStartOnNoop)) {
                if (!isNullOrEmpty(pd.startScript)) {
                    BatchGroup b = new BatchGroup(pd.startOrder, pd.startVerb);
                    BatchElement e = new BatchElement(pd.createStep(ctx, d, pd.startOrder, pd.startScript, pd.startVerb, pd.startOptions));
                    logger.debug("Adding {} to {} for start", e, b);
                    batchGroups.put(b, e);
                }
            }
        }
        return maxBatchSize;
    }

    private static void generateBatchSteps(HostContainer container, DeploymentPlanningContext ctx, ListMultimap<BatchGroup, BatchElement> batchGroups, int maxBatchSize) {
        logger.debug("Grouping component steps into batch steps (with maxBatchSize = {})", maxBatchSize);
        for (BatchGroup batchGroup : batchGroups.keySet()) {
            List<BatchElement> elements = batchGroups.get(batchGroup);
            logger.debug("Grouping component steps for batch group {} into batches", batchGroup);
            for (int batchStartIndex = 0; batchStartIndex < elements.size(); batchStartIndex += maxBatchSize) {
                int batchEndIndex = Math.min(batchStartIndex + maxBatchSize, elements.size());
                logger.debug("Grouping {} component steps for batch group {} into one batch", batchEndIndex - batchStartIndex, batchGroup);
                List<BatchElement> batch = elements.subList(batchStartIndex, batchEndIndex);
                logger.debug("Converting batch elements {} into steps", batch);
                List<PowerShellDeploymentStep> batchSteps = Lists.transform(batch,
                    new Function<BatchElement, PowerShellDeploymentStep>() {
                        @Override
                        public PowerShellDeploymentStep apply(BatchElement input) {
                            return input.getStep();
                        }
                    });
                logger.debug("Generatting description");
                String description = generateBatchDescription(batchGroup.getVerb(), container, batchSteps);
                logger.debug("Creating batch step with description [{}]", description);
                PowerShellBatchDeploymentStep batchStep = new PowerShellBatchDeploymentStep(batchGroup.getOrder(), container, batchSteps, description);
                logger.debug("Adding batch step");
                ctx.addStep(batchStep);
                logger.debug("Adding checkpoints");
                for(BatchElement e: batch) {
                    if(e.getDelta() != null) {
                        if(e.getOperation() != null) {
                            ctx.addCheckpoint(batchStep, e.getDelta(), e.getOperation());
                        } else {
                            ctx.addCheckpoint(batchStep, e.getDelta());
                        }
                    }
                }
            }
        }
    }

    private static String generateBatchDescription(String verb, HostContainer container, List<PowerShellDeploymentStep> steps) {
        StringBuilder descriptionBuilder = new StringBuilder();
        descriptionBuilder.append(verb);
        for (int i = 0; i < steps.size(); i++) {
            if(descriptionBuilder.length() > MAX_DESCRIPTION_LENGTH) {
                descriptionBuilder.append(", ...");
                break;
            }
            descriptionBuilder.append((i > 0 ? ", ": " "));
            descriptionBuilder.append(steps.get(i).deployed.getName());
        }
        descriptionBuilder.append(" on ");
        descriptionBuilder.append(container.getName());
        return descriptionBuilder.toString();
    }

    private static Deployed<?, ?> getActiveDeployed(Delta d) {
        if (d.getDeployed() != null) {
            return d.getDeployed();
        } else {
            return d.getPrevious();
        }
    }

    static class BatchGroup {
        private final int order;
        private final String verb;
        
        public BatchGroup(int order, String verb) {
            this.order = order;
            this.verb = verb;
        }
        
        public int getOrder() {
            return order;
        }
        
        public String getVerb() {
            return verb;
        }

        public String toString() {
            return format("BatchGroup[order = %d, verb = %s]", order, verb);
        }
        
        @Override
        public int hashCode() {
            return Objects.hashCode(order, verb);
        }
        
        @Override
        public boolean equals(Object o) {
            if(o == null) {
                return false;
            }
            if(this.getClass() != o.getClass()) {
                return false;
            }
            BatchGroup that = (BatchGroup) o;
            return Objects.equal(this.verb, that.verb) && Objects.equal(this.order, that.order);
        }
    }
    
    static class BatchElement {
        private final PowerShellDeploymentStep step;
        private final Delta delta;
        private final Operation operation;

        public BatchElement(PowerShellDeploymentStep step) {
            this(step, null, null);
        }

        public BatchElement(PowerShellDeploymentStep step, Delta delta, Operation operation) {
            this.step = step;
            this.delta = delta;
            this.operation = operation;
        }

        public PowerShellDeploymentStep getStep() {
            return step;
        }

        public Delta getDelta() {
            return delta;
        }

        public Operation getOperation() {
            return operation;
        }

        public String toString() {
            if(delta == null && operation == null) {
                return format("BatchElement[step = \"%s\"]", step.getDescription());
            } else {
                return format("BatchElement[step = \"%s\", delta = %s, operation = %s]", step.getDescription(), delta, operation);
            }
        }
    }

    private static Logger logger = LoggerFactory.getLogger(PowerShellBatchContributor.class);

}
