package com.xebialabs.deployit.plugin.powershell;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.overthere.OverthereConnection;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;

public class PowerShellVarsConverter extends BasePowershellVarsConverter {

    public static String javaToPowerShell(OverthereConnection connection, Map<String, Object> powershellVars, boolean maskPasswords,
                                          boolean exposePasswordPropertiesOnHosts) {
        return javaToPowerShell(connection, powershellVars, true, null, maskPasswords, exposePasswordPropertiesOnHosts);
    }

    public static String javaToPowerShell(OverthereConnection connection, Map<String, Object> powershellVars, boolean uploadArtifactData,
                                          ExecutionContext ctx, boolean maskPasswords, boolean exposePasswordPropertiesOnHosts) {
        List<String> powerShellList = javaToPowerShellList(connection, powershellVars, uploadArtifactData, ctx, maskPasswords, exposePasswordPropertiesOnHosts);
        return Joiner.on("\n").join(powerShellList) + "\n";
    }

    static List<String> javaToPowerShellList(OverthereConnection connection, Map<String, Object> powershellVars, boolean uploadArtifactData,
                                             ExecutionContext ctx, boolean maskPasswords, boolean exposePasswordPropertiesOnHosts) {
        PowerShellVarsConverter converter = new PowerShellVarsConverter(connection, powershellVars, maskPasswords, exposePasswordPropertiesOnHosts);
        converter.setUploadArtifactData(uploadArtifactData);
        converter.setExecutionContext(ctx);
        return converter.convert();
    }

    PowerShellVarsConverter(OverthereConnection connection, Map<String, Object> powershellVars, boolean maskPasswords, boolean exposePasswordPropertiesOnHosts) {
        super(connection, powershellVars, maskPasswords, exposePasswordPropertiesOnHosts);
    }

    @Override
    protected void setNullVariable(String variableName) {
        add("$" + variableName + " = $null");
    }

    @Override
    protected void setBooleanVariable(String variableName, boolean variableValue) {
        add("$" + variableName + " = " + toPowerShellBoolean(variableValue));
    }

    @Override
    protected void setIntegerVariable(String variableName, int variableValue) {
        add("$" + variableName + " = " + variableValue);
    }

    @Override
    protected void setLongVariable(String variableName, long variableValue) {
        add("$" + variableName + " = " + variableValue);
    }

    @Override
    protected void setStringVariable(String variableName, String variableValue) {
        add("$" + variableName + " = " + toPowerShellString(variableValue));
    }

    @Override
    protected void setCollectionOfStringsVariable(String variableName, Collection<?> variableValue) {
        add("$" + variableName + " = " + toPowerShellCollectionOfStrings(variableValue));
    }

    @Override
    protected void startCreateObject(String variableName) {
        add("$" + variableName + " = New-Object Object");
    }

    @Override
    protected void setBooleanProperty(String objectName, String propertyName, boolean propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellBoolean(propertyValue));
    }

    @Override
    protected void setIntegerProperty(String objectName, String propertyName, int propertyValue) {
        setNoteProperty(objectName, propertyName, Integer.toString(propertyValue));
    }

    @Override
    protected void setStringProperty(String objectName, String propertyName, String propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellString(propertyValue));
    }

    @Override
    protected void setPasswordProperty(String objectName, String propertyName, String propertyValue) {
        setNullProperty(objectName, propertyName);
        this.add("$" + objectName + "." + toPowerShellString(propertyName) + " = " +
                (maskPasswords ?
                        toPowerShellString("********") :
                        "([System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String(" + toPowerShellString(encodeBase64String(propertyValue.getBytes(UTF_8))) + ")))"
                )
        );
    }

    @Override
    protected void setCollectionOfStringsProperty(String objectName, String propertyName, Collection<?> propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellCollectionOfStrings(propertyValue));
    }

    @Override
    protected void setMapOfStringToStringReferencesProperty(String objectName, String propertyName, Map<String, String> propertyValue) {
        List<String> entries = newArrayList();
        for (String key : propertyValue.keySet()) {
            String value = propertyValue.get(key);
            if (value == null) {
                entries.add(toPowerShellString(key) + " = $null");
            } else {
                entries.add(toPowerShellString(key) + " = " + toPowerShellString(value));
            }
        }
        setNoteProperty(objectName, propertyName, "@{" + Joiner.on("; ").join(entries) + "}");
    }

    public static String toPowerShellBoolean(Boolean bool) {
        return bool ? "$true" : "$false";
    }

    public static String toPowerShellCollectionOfStrings(Collection<?> variableValue) {
        Collection<String> encodedStrings = Collections2.transform(variableValue, new Function<Object, String>() {
            public String apply(Object input) {
                return toPowerShellString(input.toString());
            }
        });
        return "@(" + Joiner.on(", ").join(encodedStrings) + ")";
    }

}
