package com.xebialabs.deployit.plugin.powershell;

import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.flow.*;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.PowerShellScriptCallback;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.executePowerShellScript;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.previewPowerShellScript;
import static com.xebialabs.deployit.plugin.powershell.PowerShellVarsConverter.toPowerShellString;

@SuppressWarnings("serial")
public class PowerShellBatchDeploymentStep implements Step, PreviewStep, StageableStep, SatelliteAware {
    private static Logger logger = LoggerFactory.getLogger(PowerShellBatchDeploymentStep.class);
    private int order;
    private HostContainer container;
    private List<PowerShellDeploymentStep> steps;
    private String description;

    public PowerShellBatchDeploymentStep(int order, HostContainer container, List<PowerShellDeploymentStep> steps, String description) {
        this.order = order;
        this.container = container;
        this.steps = newArrayList(steps);
        this.description = description;
    }

    @Override
    public Preview getPreview() {
        return previewPowerShellScript(getPowerShellScriptCallback(null, true, false));
    }

    @Override
    public void requestStaging(StagingContext ctx) {
        for (PowerShellDeploymentStep step : steps) {
            step.requestStaging(ctx);
        }
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        return executePowerShellScript(container, ctx, getPowerShellScriptCallback(ctx, false, true), getPowerShellScriptCallback(ctx, true, false));
    }

    private PowerShellScriptCallback getPowerShellScriptCallback(final ExecutionContext ctx, final boolean maskPasswords, final boolean uploadArtifacts) {
        return new PowerShellStepUtils.PowerShellScriptCallback() {
            @Override
            public String getScriptPath() {
                return "combined.ps1";
            }

            @Override
            public String generateScript(OverthereConnection conn) {
                return doGenerateScript(conn, ctx, maskPasswords, uploadArtifacts);
            }

            @Override
            public void uploadAdditionalResources(HostContainer container, ExecutionContext ctx, OverthereConnection conn) {
                //
            }
        };
    }

    public String doGenerateScript(OverthereConnection conn, ExecutionContext ctx, boolean maskPasswords, boolean uploadArtifacts) {
        logger.debug("Generating PowerShell batch script for step [{}]", getDescription());
        StringBuilder combinedScriptBuffer = new StringBuilder();
        for (PowerShellDeploymentStep s : steps) {
            logger.debug("Appending contents of step [{}]", s.getDescription());
            combinedScriptBuffer.append("Write-Host '===', ")
                    .append(toPowerShellString(s.getDescription()))
                    .append(", '==='\r\n");
            combinedScriptBuffer.append("& {\r\n");
            combinedScriptBuffer.append(s.doGenerateScript(new PowerShellVarsConverter(conn, s.powershellVars, maskPasswords, uploadArtifacts)));
            combinedScriptBuffer.append("}\r\n");
        }
        return combinedScriptBuffer.toString();
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Satellite getSatellite() {
        return container.getHost().getSatellite();
    }
}
