package com.xebialabs.deployit.core.upgrade.service

import java.sql.{PreparedStatement, ResultSet}

import com.xebialabs.deployit.core.sql.spring.{DeployJdbcTemplate, transactional}
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.core.upgrade._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional
import java.sql.{PreparedStatement, ResultSet}

import javax.sql.DataSource

import scala.jdk.CollectionConverters._

trait ArchiveSecurityUpgraderService extends Queries {
  def updateEnvironments(cisEnvs: List[ConfigurationItemData]): Unit

  def updateApplications(cisApps: List[ConfigurationItemData]): Unit

  def updateTargetCis(otherCis: List[ConfigurationItemData]): Unit

  def insertArchiveRoles(roles: List[RoleEntry]): Unit

  def insertArchiveRoleRoles(roleRoles: List[RoleRole]): Unit

  def insertArchiveRolePrincipals(rolePrincipals: List[RolePrincipal]): Unit

  def insertArchiveRolePermissions(rolePermissions: List[PermissionEntry]): Unit
}

@Component
class DefaultArchiveSecurityUpgraderService(
                                      @Autowired @Qualifier("reportingSchema") val schemaInfo: SchemaInfo,
                                      @Autowired @Qualifier("reportingDataSource") val reportingDataSource: DataSource,
                                      @Autowired @Qualifier("reportingTransactionManager") val reportingTransactionManager: PlatformTransactionManager
                                    ) extends ArchiveSecurityUpgraderService with Logging {
  private[this] val reportingJdbcTemplate = new DeployJdbcTemplate(reportingDataSource, false)

  private val CHUNK_SIZE = 1000

  @Transactional("reportingTransactionManager")
  def fetchArchiveRoles(): List[RoleEntry] = {
    import RolesSchema.Roles._

    reportingJdbcTemplate.query(sqlb"SELECT $ID, $NAME, $CI_ID from $archivedTableName",
      new RowMapper[RoleEntry]() {
        override def mapRow(rs: ResultSet, rowNum: Int): RoleEntry = {
          RoleEntry(rs.getString(1), rs.getString(2), rs.getInt(3))
        }
      }).asScala.toList
  }

  @Transactional("reportingTransactionManager")
  def fetchArchiveRoleRoles(): List[RoleRole] = {
    import RolesSchema.RoleRoles._

    reportingJdbcTemplate.query(sqlb"SELECT $ROLE_ID, $MEMBER_ROLE_ID from $archivedTableName",
      new RowMapper[RoleRole]() {
        override def mapRow(rs: ResultSet, rowNum: Int): RoleRole = {
          RoleRole(rs.getString(1), rs.getString(2))
        }
      }).asScala.toList
  }

  @Transactional("reportingTransactionManager")
  def fetchArchiveRolePrincipals(): List[RolePrincipal] = {
    import RolesSchema.RolePrincipals._

    reportingJdbcTemplate.query(sqlb"SELECT $ROLE_ID, $PRINCIPAL_NAME from $archivedTableName",
      new RowMapper[RolePrincipal]() {
        override def mapRow(rs: ResultSet, rowNum: Int): RolePrincipal = {
          RolePrincipal(rs.getString(1), rs.getString(2))
        }
      }).asScala.toList
  }

  @Transactional("reportingTransactionManager")
  def fetchArchiveRolePermissions(): List[PermissionEntry] = {
    import PermissionsSchema._

    reportingJdbcTemplate.query(sqlb"SELECT $ROLE_ID, $PERMISSION_NAME, $CI_ID from $archivedTableName",
      new RowMapper[PermissionEntry]() {
        override def mapRow(rs: ResultSet, rowNum: Int): PermissionEntry = {
          PermissionEntry(rs.getString(1), rs.getString(2), rs.getInt(3))
        }
      }).asScala.toList
  }

  def updateApplications(cisApps: List[ConfigurationItemData]): Unit = {
    updateCisByChunks(cisApps, "applications", updateApplicationChunk)
  }

  def updateEnvironments(cisEnvs: List[ConfigurationItemData]): Unit = {
    updateCisByChunks(cisEnvs, "environments", updateEnvironmentChunk)
  }

  def updateTargetCis(otherCis: List[ConfigurationItemData]): Unit = {
    updateCisByChunks(otherCis, "all other CIS", updateTargetCiChunk)
  }

  private def updateCisByChunks(cis: List[ConfigurationItemData], ciTypeName: String, updater: List[ConfigurationItemData] => Unit): Unit = {
    var chunkCounter = 0
    logger.debug(s"Updating archive data for existing $ciTypeName")
    cis.grouped(CHUNK_SIZE).foreach(l => {
      transactional(reportingTransactionManager) {
        chunkCounter += 1
        logger.debug(s"Updating archive data for existing $ciTypeName chunk $chunkCounter")
        updater(l)
      }
    })
  }

  def insertArchiveRoles(roles: List[RoleEntry]): Unit = {
    insertIntoArchiveTable[RoleEntry](roles, "roles", fetchArchiveRoles _, insertRoleChunk)
  }

  def insertArchiveRoleRoles(roleRoles: List[RoleRole]): Unit = {
    insertIntoArchiveTable[RoleRole](roleRoles, "roleRoles", fetchArchiveRoleRoles _, insertRoleRoleChunk)
  }

  def insertArchiveRolePrincipals(rolePrincipals: List[RolePrincipal]): Unit = {
    insertIntoArchiveTable[RolePrincipal](rolePrincipals, "rolePrincipals", fetchArchiveRolePrincipals _, insertRolePrincipalChunk)
  }

  def insertArchiveRolePermissions(rolePermissions: List[PermissionEntry]): Unit = {
    insertIntoArchiveTable[PermissionEntry](rolePermissions, "rolePermissions", fetchArchiveRolePermissions _, insertPermissionChunk)
  }

  private def insertIntoArchiveTable[E](mainEntites: List[E], ciTypeName: String, fetchArchived: () => List[E], insertChunk: List[E] => Unit): Unit = {
    var chunkCounter = 0
    val archiveEntities = fetchArchived()
    val diff = mainEntites diff archiveEntities
    logger.debug(s"Inserting $ciTypeName (total/existing: ${mainEntites.size}/${archiveEntities.size}) into archive security tables")
    diff.grouped(CHUNK_SIZE).foreach(l => {
      transactional(reportingTransactionManager) {
        chunkCounter += 1
        logger.debug(s"Inserting $ciTypeName chunk $chunkCounter")
        insertChunk(l)
      }
    })
  }

  private def updateApplicationChunk(cisApps: List[ConfigurationItemData]): Unit = {
    import ArchivedDeploymentTasks._
    val archivedApps =
      reportingJdbcTemplate.queryForList(sqlb"SELECT DISTINCT ${Applications.application} from ${Applications.archiveApplicationTable}",
        classOf[String]).asScala.toSet
    cisApps.filter(item => archivedApps contains item.name).foreach(
      item =>
        reportingJdbcTemplate.update(sqlb"UPDATE ${Applications.archiveApplicationTable} SET " +
          sqlb"${Applications.applicationInternalId} = ?, ${Applications.applicationSecuredCi} = ? WHERE ${Applications.application} = ?",
          item.internalId, item.securedCi, item.name)
    )
  }

  private def updateEnvironmentChunk(cisEnvs: List[ConfigurationItemData]): Unit = {
    import ArchivedDeploymentTasks._
    val archivedEnvs =
      reportingJdbcTemplate.queryForList(sqlb"SELECT DISTINCT $environment from $archiveTasksTable", classOf[String]).asScala.toSet

    cisEnvs.filter(item => archivedEnvs contains item.id).foreach(
      item =>
        reportingJdbcTemplate.update(sqlb"UPDATE $archiveTasksTable SET $environmentInternalId = ?, $environmentSecuredCi = ? WHERE $environment = ?",
          item.internalId, item.securedCi, item.id)
    )
  }

  private def updateTargetCiChunk(otherCis: List[ConfigurationItemData]): Unit = {
    import ArchivedControlTasks._

    val archivedControlTasks =
      reportingJdbcTemplate.queryForList(sqlb"SELECT DISTINCT $targetCi from $archivedControlTasksTable",
        classOf[String]).asScala.toSet
    otherCis.filter(item => archivedControlTasks contains item.id).foreach(
      item =>
        reportingJdbcTemplate.update(sqlb"UPDATE $archivedControlTasksTable SET " +
          sqlb"$targetInternalId = ?, $targetSecuredCi = ? WHERE $targetCi = ?",
          item.internalId, item.securedCi, item.id)
    )
  }

  def insertRoleChunk(roles: List[RoleEntry]): Unit = {
    import RolesSchema.Roles._

    reportingJdbcTemplate.batchUpdate(sqlb"INSERT INTO $archivedTableName  ($ID , $NAME, $CI_ID) VALUES (?,?,?)",
      new BatchPreparedStatementSetter {
        override def getBatchSize: Int = roles.size

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, roles(i).id)
          Setter.setStringForReporting(ps, 2, roles(i).name)
          ps.setInt(3, roles(i).ciId)
        }
      })
  }

  def insertRoleRoleChunk(roleRoles: List[RoleRole]): Unit = {
    import RolesSchema.RoleRoles._

    reportingJdbcTemplate.batchUpdate(sqlb"INSERT INTO $archivedTableName  ($ROLE_ID , $MEMBER_ROLE_ID) VALUES (?,?)",
      new BatchPreparedStatementSetter {
        override def getBatchSize: Int = roleRoles.size

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, roleRoles(i).roleId)
          ps.setString(2, roleRoles(i).memberRoleId)
        }
      })
  }

  def insertRolePrincipalChunk(rolePrincipals: List[RolePrincipal]): Unit = {
    import RolesSchema.RolePrincipals._

    reportingJdbcTemplate.batchUpdate(sqlb"INSERT INTO $archivedTableName  ($ROLE_ID , $PRINCIPAL_NAME) VALUES (?,?)",
      new BatchPreparedStatementSetter {
        override def getBatchSize: Int = rolePrincipals.size

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, rolePrincipals(i).roleId)
          Setter.setStringForReporting(ps, 2, rolePrincipals(i).principalName)
        }
      })
  }

  def insertPermissionChunk(rolePermissions: List[PermissionEntry]): Unit = {
    import PermissionsSchema._

    reportingJdbcTemplate.batchUpdate(sqlb"INSERT INTO $archivedTableName  ($ROLE_ID , $PERMISSION_NAME, $CI_ID) VALUES (?,?,?)",
      new BatchPreparedStatementSetter {
        override def getBatchSize: Int = rolePermissions.size

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, rolePermissions(i).roleId)
          Setter.setStringForReporting(ps,2, rolePermissions(i).permissionName)
          ps.setInt(3, rolePermissions(i).ciId)
        }
      })
  }

}
