package com.xebialabs.deployit.core.upgrade

import ai.digital.deploy.core.common.security.permission.{DeployitPermissions, StitchPermissions}
import com.xebialabs.deployit.core.sql.spring.transactional
import com.xebialabs.deployit.core.upgrade.Deployit1000ReadOnlyRoleUpgrader.READ_ONLY_ROLE_NAME
import com.xebialabs.deployit.core.upgrade.service.{ArchiveSecurityUpgraderService, SecurityUpgraderService}
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.permission.PlatformPermissions.{LOGIN, REPORT_VIEW, VIEW_SECURITY}
import com.xebialabs.deployit.security.service.UserProfileService
import com.xebialabs.deployit.security.{GLOBAL_ID, UserService}
import com.xebialabs.deployit.server.api.upgrade.{RepositoryInitialization, Upgrade, Version}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.transaction.PlatformTransactionManager

import java.util.UUID
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class Deployit1000ReadOnlyRoleUpgrader(

                                        @Autowired val securityUpgraderService: SecurityUpgraderService,
                                        @Autowired val archiveSecurityUpgraderService: ArchiveSecurityUpgraderService,
                                        @Autowired xldUserProfileService: UserProfileService,
                                        @Autowired userService: UserService,
                                        @Autowired @Qualifier("mainTransactionManager") transactionManager: PlatformTransactionManager
                                      ) extends Upgrade with RepositoryInitialization with Logging {

  override def doUpgrade(): Boolean = doCreateRoleAndProfiles()

  override def upgradeVersion(): Version = Version.valueOf("deployit", "10.0.0")

  override def doInitialize(): Unit = doCreateRoleAndProfiles()

  override def getComponent: String = upgradeVersion().getComponent

  override def getPriority: Integer = Integer.MAX_VALUE

  private def doCreateUserProfiles(): Boolean = {
    Try({
      transactional(transactionManager) {
        createProfiles()
        logger.debug("User profiles created.")
      }
    }) match {
      case Success(_) =>
        true
      case Failure(exception: Exception) =>
        logger.error(s"Unable to complete ${getClass.getName}.", exception)
        false
    }
  }

  private def createProfiles(): Unit = {
    userService.listUsernames().asScala foreach (username => {
      Try(xldUserProfileService.findOne(username)) match {
        case Success(_) => identity()
        case Failure(_: NotFoundException) =>
          xldUserProfileService.createOrUpdate(username)
        case Failure(e) => throw e
      }
    })
  }

  private def doCreateRole(): Boolean = {
    Try({
      if (!roleAlreadyExists()) {
        transactional(transactionManager) {
          createReadOnlyRole()
          logger.debug(s"""Read only role "$READ_ONLY_ROLE_NAME" created.""")
        }
      }
      assignPermissions()
    }) match {
      case Success(_) =>
        true
      case Failure(exception: Exception) =>
        logger.error("Unable to complete Deployit1000ReadOnlyRoleUpgrader.", exception)
        false
    }
  }

  private def createReadOnlyRole(): Unit = {
    val roleEntry = List(RoleEntry(UUID.randomUUID().toString, READ_ONLY_ROLE_NAME, GLOBAL_ID))
    securityUpgraderService.insertRoles(roleEntry)
    archiveSecurityUpgraderService.insertArchiveRoles(roleEntry)
  }

  private def assignPermissions(): Unit = {
    implicit val role: RoleEntry = securityUpgraderService.fetchRoleByName(READ_ONLY_ROLE_NAME)
      .getOrElse(throw new DeployitException(s"Could not find $READ_ONLY_ROLE_NAME role - something went wrong with creation of it."))
    val permissions = List(
      VIEW_SECURITY,
      LOGIN,
      REPORT_VIEW,
      DeployitPermissions.TASK_VIEW,
      DeployitPermissions.TASK_PREVIEWSTEP,
      StitchPermissions.VIEW
    ).map(permission => PermissionEntry(role.id, permission.getPermissionName, GLOBAL_ID))

    securityUpgraderService.insertRolePermissions(permissions)
    archiveSecurityUpgraderService.insertArchiveRolePermissions(permissions)
  }

  private def roleAlreadyExists(): Boolean =
    securityUpgraderService.fetchRoleByName(READ_ONLY_ROLE_NAME).map(
      role => {
        logger.warn(s"A role with name $READ_ONLY_ROLE_NAME already exists. Skipping role creation in ${this.getClass.getSimpleName}. " +
          "Recommend manually creating an admin read only role with all functional read permissions.")
        role
      }
    ).isDefined

  private def doCreateRoleAndProfiles(): Boolean = doCreateUserProfiles && doCreateRole
}

object Deployit1000ReadOnlyRoleUpgrader {
  val READ_ONLY_ROLE_NAME = "deploy_admin_read_only"
}
