package com.xebialabs.deployit.core.upgrade

import com.xebialabs.deployit.core.sql.spring.transactional
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.repository.sql.base.asCiPKType
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.server.api.upgrade.Upgrade
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.transaction.PlatformTransactionManager

/**
 * This upgrader fixes corrupted data after a CI was moved and the security configuration ("secured_ci") was not updated.
 * This fix should be applied in XLD 8.0.10 and XLD 8.2.1 upgrades.
 */
abstract class AbstractFixSecurityAfterMoveUpgrader(schemaInfo: SchemaInfo, jdbcTemplate: JdbcTemplate, transactionManager: PlatformTransactionManager) extends Upgrade with Logging {

  override def doUpgrade(): Boolean = {
    val queries = new UpgraderQueries(schemaInfo)
    val incorrectRoots = jdbcTemplate.queryForList(queries.SELECT_MOVED_CIS)
    info(s"Found ${incorrectRoots.size()} moved CIS with incorrect security configuration.")
    incorrectRoots.forEach { map =>
      transactional(transactionManager) {
        val (movedPath, oldSecuredId, newSecuredId) = (
          map.get("moved_path").asInstanceOf[String],
          asCiPKType(map.get("old_secured_id")),
          asCiPKType(map.get("new_secured_id"))
        )
        val updated = jdbcTemplate.update(queries.UPDATE_MOVED_CIS, newSecuredId, movedPath, movedPath + "/%", oldSecuredId)
        info(s"Fix $updated CIS with incorrect security configuration.")
      }
    }
    true
  }

  private class UpgraderQueries(val schemaInfo: SchemaInfo) extends Queries {

    lazy val SELECT_MOVED_CIS: String = {
      import CIS._
      sqlb"select c.$path as moved_path, c.$secured_ci as old_secured_id, p.$secured_ci as new_secured_id from $tableName c inner join $tableName p on (c.$parent_id = p.$ID) where c.$secured_ci <> c.$ID and c.$secured_ci <> p.$secured_ci"
    }

    lazy val UPDATE_MOVED_CIS: String = {
      import CIS._
      sqlb"update $tableName set $secured_ci = ? where ($path = ? or $path like ?) and $secured_ci = ?"
    }
  }

}
