package com.xebialabs.deployit.core.upgrade

import java.util
import java.util.{Map => JMap}
import com.xebialabs.deployit.core.sql.spring.transactional
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.core.upgrade.service.BatchUpgraderService
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.CiRepository
import com.xebialabs.deployit.repository.{RepositoryService, SearchParameters}
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.deployit.util.PasswordEncrypter
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.transaction.PlatformTransactionManager

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class Deployit975ReEncryptDictionaryValuesUpgrader(@Autowired @Qualifier("mainSchema") val schemaInfo: SchemaInfo,
                                                   @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                   @Autowired ciRepository: CiRepository,
                                                   @Autowired repositoryService: RepositoryService,
                                                   @Autowired @Qualifier("mainTransactionManager") transactionManager: PlatformTransactionManager,
                                                   @Autowired batchUpgraderService: BatchUpgraderService)
  extends Upgrade with Logging with Queries {

  val passwordEncrypter: PasswordEncrypter = PasswordEncrypter.getInstance()

  override def doUpgrade(): Boolean = {
    Try({
      transactional(transactionManager) {
        updateEncryptedDictionary()
      }
    }) match {
      case Success(_) =>
        logger.debug("Updated legacy password values in XL Deploy successfully.")
        true
      case Failure(exception: Exception) =>
        logger.error("Unable to complete Deployit975ReEncryptDictionaryValuesUpgrader.", exception)
        false
    }
  }

  override def upgradeVersion(): Version = Version.valueOf("deployit", "9.7.5")

  def updateEncryptedDictionary(): Unit = {

    val query = new SearchParameters().setType(Type.valueOf("udm.Dictionary"))
    batchUpgraderService.getBatchCis(query, params => {
      ciRepository.listEntities(params).asScala.foreach { ci: ConfigurationItem =>
        val encryptedDictionaryEntries: JMap[String, String] = ci.getProperty("encryptedEntries")
        logger.info(s"Found encrypted dictionary ${encryptedDictionaryEntries.keySet()}")
        val newDictionaryValues = new util.HashMap[String, String]()
        newDictionaryValues.putAll(encryptedDictionaryEntries)
        encryptedDictionaryEntries.asScala.foreach(dictionaryEntry => {
          val key = dictionaryEntry._1
          val value = dictionaryEntry._2
          if (!passwordEncrypter.isEncrypted(value)) {
            val reEncryptedValue = passwordEncrypter.encrypt(value)
            logger.info(s"Dictionary $key is going to be reEncrypted.")
            newDictionaryValues.put(key, reEncryptedValue)
          }
        })

        if (!newDictionaryValues.equals(encryptedDictionaryEntries.asScala)) {
          ci.setProperty("encryptedEntries", newDictionaryValues)
          repositoryService.update(ci)
        }
      }
    })
  }
}
