package com.xebialabs.deployit.core.upgrade.service

import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchExecutorRepository}
import com.xebialabs.deployit.core.sql.spring.{Setter, XlSingleColumnRowMapper}
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo, SelectBuilder, SqlCondition => cond}
import com.xebialabs.deployit.core.upgrade.service.CISSchema._
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.repository.sql.base.CiPKType
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper, SingleColumnRowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util.UUID
import scala.jdk.CollectionConverters._

trait DirectoryRefUpgraderService extends Queries {
  val directoryType = "core.Directory"
  val rootType = "internal.Root"
  def updateDirectoryUUIDs()
  def updateDirectoryRefs()
  def getCiIdDirectoryReferencesMap(): Map[CiPKType, String]
  def updateSecuredDirectoryRefs()
  def getCiIdSecuredDirectoryReferencesMap(): Map[CiPKType, String]
}

@Component
@Transactional("mainTransactionManager")
class DefaultDirectoryRefUpgraderService(
                                      @Autowired @Qualifier("mainSchema") val schemaInfo: SchemaInfo,
                                      @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                      @Autowired @Qualifier("mainTransactionManager") val transactionManager: PlatformTransactionManager,
                                      @Autowired @Qualifier("mainBatchExecutorRepository") val batchExecutorRepository: BatchExecutorRepository
) extends DirectoryRefUpgraderService {
  override def updateDirectoryUUIDs(): Unit = {
    val directories: List[CiPKType] = jdbcTemplate.query(
      sqlb"SELECT $idColumn from $tableName where $ciTypeColumn = ? or $ciTypeColumn = ?",
      new XlSingleColumnRowMapper(classOf[CiPKType]),
      rootType, directoryType).asScala.toList
    val commands = directories.map(
      id => BatchCommand(sqlb"UPDATE $tableName SET $directoryUUIDColumn = ? WHERE $idColumn = ?", UUID.randomUUID().toString, id)
    )
    batchExecutorRepository.execute(commands)
  }

  override def updateDirectoryRefs(): Unit = {
    val selectRootsAndDirectories = new SelectBuilder(tableName)(schemaInfo)
      .select(idColumn).select(directoryUUIDColumn)
      .where(cond.in(ciTypeColumn, Seq(rootType, directoryType)))
    val directoriesPkUUIDMap: Map[CiPKType, String] = jdbcTemplate.query(
      selectRootsAndDirectories.query, Setter(selectRootsAndDirectories.parameters),
      new RowMapper[(CiPKType, String)]() {
        override def mapRow(rs: ResultSet, rowNum: Int): (CiPKType, String) = {
          (rs.getInt(idColumn.name), rs.getString(directoryUUIDColumn.name))
        }
      }).asScala.toMap

    val selectCIs = new SelectBuilder(tableName)(schemaInfo)
      .select(idColumn).select(parentIdColumn)
      .where(cond.not(cond.in(ciTypeColumn, Seq(rootType))))
    val cisIdParentIdMap = jdbcTemplate.query(
      selectCIs.query, Setter(selectCIs.parameters),
      new RowMapper[(CiPKType, CiPKType)]() {
        override def mapRow(rs: ResultSet, rowNum: Int): (CiPKType, CiPKType) = {
          (rs.getInt(idColumn.name), rs.getInt(parentIdColumn.name))
        }
      }).asScala.toMap

    def getCiDirectoryId(id: CiPKType): String = {
      cisIdParentIdMap.get(id) match {
        case None => throw new DeployitException("Could not find directory UUID for CI")
        case Some(parentId) =>
          directoriesPkUUIDMap.getOrElse(parentId, getCiDirectoryId(parentId))
      }
    }

    val commands = cisIdParentIdMap.map {
      case (id: CiPKType, parentId: CiPKType) =>
        val ciDirectoryUUID = directoriesPkUUIDMap.getOrElse(parentId, getCiDirectoryId(parentId))
        BatchCommand(sqlb"UPDATE $tableName SET $directoryRefColumn = ? WHERE $idColumn = ?", ciDirectoryUUID, id)
    }
    batchExecutorRepository.execute(commands)
  }

  override def getCiIdDirectoryReferencesMap(): Map[CiPKType, String] = {
    val selectCisDirectoryData = new SelectBuilder(tableName)(schemaInfo)
      .select(idColumn).select(directoryRefColumn)
      .where(cond.not(cond.in(ciTypeColumn, Seq(rootType, directoryType))))
    jdbcTemplate.query(
      selectCisDirectoryData.query, Setter(selectCisDirectoryData.parameters),
      new RowMapper[(CiPKType, String)]() {
        override def mapRow(rs: ResultSet, rowNum: Int): (CiPKType, String) = {
          (rs.getInt(idColumn.name), rs.getString(directoryRefColumn.name))
        }
      }
    ).asScala.toMap
  }

  override def updateSecuredDirectoryRefs(): Unit = {
    jdbcTemplate.update(
      sqlb"UPDATE $tableName SET $securedDirectoryRefColumn = $directoryUUIDColumn WHERE $ciTypeColumn = ?",
      rootType)
    val rootSelect = new SelectBuilder(tableName)(schemaInfo)
      .select(pathColumn)
      .select(directoryUUIDColumn)
      .where(cond.equals(ciTypeColumn, rootType))
    val roots = jdbcTemplate.query(
      rootSelect.query,
      Setter(rootSelect.parameters),
      new RowMapper[(String, String)]() {
        override def mapRow(rs: ResultSet, rowNum: Int): (String, String) = {
          (rs.getString(pathColumn.name), rs.getString(directoryUUIDColumn.name))
        }
      }
    ).asScala.toMap
    val selectCis = new SelectBuilder(tableName)(schemaInfo)
      .select(idColumn)
      .select(pathColumn)
      .select(securedCiColumn)
      .select(directoryUUIDColumn)
      .where(cond.not(cond.equals(ciTypeColumn, rootType)))
    val cis = jdbcTemplate.query(
      selectCis.query,
      Setter(selectCis.parameters),
      new RowMapper[(CiPKType, (CiPKType, String, String))]() {
        override def mapRow(rs: ResultSet, rowNum: Int): (CiPKType, (CiPKType, String, String)) = {
          (rs.getInt(idColumn.name), (rs.getInt(securedCiColumn.name), rs.getString(directoryUUIDColumn.name), rs.getString(pathColumn.name)))
        }
      }
    ).asScala.toMap

    def getRootPath(path: String): String = {
        path.substring(0, path.substring(1).indexOf("/") + 1)
    }

    def getRootDirectoryUUIDForPath(path: String): String = {
      roots.getOrElse(getRootPath(path), throw new DeployitException(s"Could not find Root CI for $path"))
    }

    val commands = cis.map {
      case (id, (securedId, directoryUUID, _)) if id == securedId =>
        BatchCommand(sqlb"UPDATE $tableName SET $securedDirectoryRefColumn = ? WHERE $idColumn = ?", directoryUUID, id)
      case (id, (securedId, _, path)) if securedId == 0 =>
        BatchCommand(sqlb"UPDATE $tableName SET $securedDirectoryRefColumn = ? WHERE $idColumn = ?", getRootDirectoryUUIDForPath(path), id)
      case (id, (securedId, _, path)) =>
        val securedParentDirectoryUUID = cis.getOrElse(securedId, (securedId, getRootDirectoryUUIDForPath(path), path))._2
        BatchCommand(sqlb"UPDATE $tableName SET $securedDirectoryRefColumn = ? WHERE $idColumn = ?", securedParentDirectoryUUID, id)
    }
    batchExecutorRepository.execute(commands)
  }

  override def getCiIdSecuredDirectoryReferencesMap(): Map[CiPKType, String] = {
    val selectCisDirectoryData = new SelectBuilder(tableName)(schemaInfo)
      .select(idColumn).select(securedDirectoryRefColumn)
      .where(cond.not(cond.in(ciTypeColumn, Seq(rootType, directoryType))))
    jdbcTemplate.query(
      selectCisDirectoryData.query, Setter(selectCisDirectoryData.parameters),
      new RowMapper[(CiPKType, String)]() {
        override def mapRow(rs: ResultSet, rowNum: Int): (CiPKType, String) = {
          (rs.getInt(idColumn.name), rs.getString(securedDirectoryRefColumn.name))
        }
      }
    ).asScala.toMap

  }
}
