package com.xebialabs.deployit.core.upgrade.service

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchExecutorRepository}
import com.xebialabs.deployit.core.sql.spring.{DeployJdbcTemplate, Setter, XlSingleColumnRowMapper}
import com.xebialabs.deployit.core.upgrade.service.ArchivedControlTasks._
import com.xebialabs.deployit.core.upgrade.service.ArchivedDeploymentTasks.Applications._
import com.xebialabs.deployit.core.upgrade.service.ArchivedDeploymentTasks._
import com.xebialabs.deployit.repository.sql.base.CiPKType
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional

import javax.sql.DataSource
import scala.jdk.CollectionConverters._

trait ArchiveDirectoryRefUpgraderService extends Queries {
  def updateDirectoryRefsForArchivedTables(ciIdDirectoryRefMap: Map[CiPKType, String])
}

@Component
@Transactional("reportingTransactionManager")
class DefaultArchiveDirectoryRefUpgraderService(@Autowired @Qualifier("reportingSchema") val schemaInfo: SchemaInfo,
                                                @Autowired @Qualifier("reportingDataSource") val reportingDataSource: DataSource,
                                                @Autowired @Qualifier("reportingTransactionManager") val transactionManager: PlatformTransactionManager,
                                                @Autowired @Qualifier("reportingBatchExecutorRepository") val batchExecutorRepository: BatchExecutorRepository
                                               ) extends ArchiveDirectoryRefUpgraderService {
  private[this] val jdbcTemplate = new DeployJdbcTemplate(reportingDataSource, false)

  private def collectInternalIds(tableName: TableName, internalIdColumn: ColumnName): List[CiPKType] = {
    val selectColumn = new SelectBuilder(tableName)(schemaInfo)
      .select(internalIdColumn)
      .where(SqlCondition.equalsNotNull(internalIdColumn))
      .distinct()
    jdbcTemplate.query(
      selectColumn.query, Setter(selectColumn.parameters),
      new XlSingleColumnRowMapper(classOf[CiPKType]),
    ).asScala.toList
  }

  private def createAndExecuteCommands(ciIdDirectoryRefMap: Map[CiPKType, String],
                                       tableToUpdate: TableName,
                                       directoryRefColumn: ColumnName,
                                       internalIdColumn: ColumnName
                                      ): Unit = {
    val ciInternalIds = collectInternalIds(tableToUpdate, internalIdColumn)
    val commands = ciInternalIds.map(id => {
      val directoryRef = ciIdDirectoryRefMap.getOrElse(id, null)
      BatchCommand(sqlb"UPDATE $tableToUpdate SET $directoryRefColumn = ? WHERE $internalIdColumn = ?", directoryRef, id)
    })
    batchExecutorRepository.execute(commands)
  }

  override def updateDirectoryRefsForArchivedTables(ciIdDirectoryRefMap: Map[CiPKType, String]): Unit = {
    createAndExecuteCommands(ciIdDirectoryRefMap, archiveTasksTable, environmentDirRefColumn, environmentInternalId)
    createAndExecuteCommands(ciIdDirectoryRefMap, archiveApplicationTable, applicationDirRefColumn, applicationInternalId)
    createAndExecuteCommands(ciIdDirectoryRefMap, archivedControlTasksTable, targetDirRefColumn, targetInternalId)

  }
}
