package com.xebialabs.deployit.core.upgrade

import com.xebialabs.deployit.core.sql.spring.transactional
import com.xebialabs.deployit.core.sql.{MssqlDialect, MysqlDialect, Queries, SchemaInfo}
import com.xebialabs.deployit.core.upgrade.service.PermissionsSchema
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.deployit.sql.base.schema.CIS
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.transaction.PlatformTransactionManager

import scala.util.control.Breaks._

class Deployit802SecurityPerformanceUpgrader(
                                              @Autowired @Qualifier("mainSchema") val schemaInfo: SchemaInfo,
                                              @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                              @Autowired @Qualifier("mainTransactionManager") val transactionManager: PlatformTransactionManager
                                            ) extends Upgrade with Logging {

  override def doUpgrade(): Boolean = {
    val queries = new UpgraderQueries(schemaInfo)
    var level = 0
    logger.info("Updating secured parent for all CIs.")
    transactional(transactionManager) {
      jdbcTemplate.update(queries.INITIAL_UPDATE)
      logger.info(s"Updated level $level")
    }
    breakable {
      while (true) {
        transactional(transactionManager) {
          if (jdbcTemplate.update(queries.UPDATE) == 0) break()
          level += 1
          logger.info(s"Updated level $level")
        }
      }
    }
    logger.info("Done.")
    true
  }

  override def upgradeVersion(): Version = Version.valueOf("deployit", "8.0.2")

  private class UpgraderQueries(val schemaInfo: SchemaInfo) extends Queries {

    val INITIAL_UPDATE = sqlb"update ${CIS.tableName} set ${CIS.secured_ci} = ${CIS.ID} where ${CIS.ID} in (select ${PermissionsSchema.CI_ID} from ${PermissionsSchema.tableName})"

    val UPDATE: String = {
      import CIS._
      schemaInfo.sqlDialect match {
        case MysqlDialect =>
          sqlb"update $tableName c inner join $tableName p on (p.$ID = c.$parent_id and p.$secured_ci is not null) set c.$secured_ci = p.$secured_ci where c.$secured_ci is null"
        case MssqlDialect =>
          sqlb"update c set c.$secured_ci = p.$secured_ci from $tableName c inner join $tableName p on (p.$ID = c.$parent_id and p.$secured_ci is not null) where c.$secured_ci is null"
        case _ =>
          sqlb"update $tableName set $secured_ci = (select $secured_ci from $tableName p where p.$ID = $tableName.$parent_id) where $secured_ci is null and exists(select * from $tableName p where p.$ID = $tableName.$parent_id and p.$secured_ci is not null)"
      }
    }
  }
}
