package com.xebialabs.deployit.repository.sql.placeholders

import com.xebialabs.deployit.core.sql.{SelectBuilder, SqlCondition => cond, _}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.repository.sql.persisters.{DictionaryEncryptedEntriesSchema, DictionaryEntriesSchema, EnvironmentDictionariesSchema}
import com.xebialabs.deployit.sql.base.schema.CIS

object PlaceholdersSearchQueryBuilder {

  val aliasCI = "ci"
  val aliasDictionaryEntries = "de"
  val aliasDictionaryEncryptedEntries = "dee"
  val aliasEnvironmentDictionaries = "ed"
  val aliasDictionariesUnion = "duni"
  val aliasCIForEnvironment = "cienv"
  val aliasPlaceholder = "ph"
  val aliasDictEntriesUnion = "deu"
  val aliasPhJoin = "ph1"
  val aliasDeJoin = "de1"
  val aliasDeeJoin = "dee1"

  def dictionaryUnion()(implicit schemaInfo: SchemaInfo): UnionBuilder = {
    new UnionBuilder(
      new SelectBuilder(DictionaryEntriesSchema.tableName)
        .select(DictionaryEntriesSchema.key)
        .select(DictionaryEntriesSchema.dictionary_id),
      new SelectBuilder(DictionaryEncryptedEntriesSchema.tableName)
        .select(DictionaryEncryptedEntriesSchema.key)
        .select(DictionaryEncryptedEntriesSchema.dictionary_id))
      .as(aliasDictionariesUnion)
  }

  def leftJoinOnKeyWithUnion(joinBuilder: JoinBuilder)(implicit schemaInfo: SchemaInfo): JoinBuilder = {
    joinBuilder.join(
      new SelectBuilder(DictionaryEntriesSchema.tableName).as(aliasDictionaryEntries),
      cond.and(
        Seq(
          cond.equals(
            PlaceholderSearchSchema.key.tableAlias(aliasDictionariesUnion),
            DictionaryEntriesSchema.key.tableAlias(aliasDictionaryEntries)),
          cond.equals(
            PlaceholderSearchSchema.dictionaryId.tableAlias(aliasDictionariesUnion),
            DictionaryEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEntries)
          )
        )
      ),
      JoinType.Left)
      .join(
        new SelectBuilder(DictionaryEncryptedEntriesSchema.tableName).as(aliasDictionaryEncryptedEntries),
        cond.and(
          Seq(
            cond.equals(
              PlaceholderSearchSchema.key.tableAlias(aliasDictionariesUnion),
              DictionaryEncryptedEntriesSchema.key.tableAlias(aliasDictionaryEncryptedEntries)),
            cond.equals(
              PlaceholderSearchSchema.dictionaryId.tableAlias(aliasDictionariesUnion),
              DictionaryEncryptedEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEncryptedEntries)
            )
          )
        ),
        JoinType.Left)
  }

  def joinOnEnvironmentDictionariesSchema(joinBuilder: JoinBuilder, ciEnvironmentSelectBuilder: SelectBuilder, joinType: JoinType)
                                         (implicit schemaInfo: SchemaInfo): JoinBuilder = {
    joinBuilder.join(
      new SelectBuilder(EnvironmentDictionariesSchema.tableName).as(aliasEnvironmentDictionaries),
      cond.or(
        Seq(
          cond.equals(EnvironmentDictionariesSchema.dictionary_id.tableAlias(aliasEnvironmentDictionaries),
            DictionaryEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEntries)),
          cond.equals(EnvironmentDictionariesSchema.dictionary_id.tableAlias(aliasEnvironmentDictionaries),
            DictionaryEncryptedEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEncryptedEntries))
        )
      ),
      JoinType.Left
    ).join(
      ciEnvironmentSelectBuilder,
      cond.equals(CIS.ID.tableAlias(aliasCIForEnvironment), EnvironmentDictionariesSchema.environment_id.tableAlias(aliasEnvironmentDictionaries)),
      joinType
    )
  }

  def joinOnDictionaryIdsWithCi(joinBuilder: JoinBuilder, ciSelectBuilder: SelectBuilder)(implicit schemaInfo: SchemaInfo): JoinBuilder = {
    joinBuilder.join(
      ciSelectBuilder,
      cond.or(
        Seq(
          cond.equals(CIS.ID.tableAlias(aliasCI), DictionaryEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEntries)),
          cond.equals(CIS.ID.tableAlias(aliasCI), DictionaryEncryptedEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEncryptedEntries))
        )
      )
    )
  }

  def forKey(selectBuilder: SelectBuilder, key: Option[String])(implicit schemaInfo: SchemaInfo): SelectBuilder = {
    key.map(k => selectBuilder.where(
      cond.equals(PlaceholderSearchSchema.key.tableAlias(aliasDictionariesUnion), k)
    ))
    selectBuilder
  }

  def forEnvironment(selectBuilder: SelectBuilder, environmentId: Option[String], environmentName: Option[String]): SelectBuilder = {
    environmentId.map(e => selectBuilder.where(cond.likeEscaped(SqlFunction.lower(CIS.path.tableAlias(aliasCIForEnvironment)), createLike(e).toLowerCase)))
    environmentName.map(e => selectBuilder.where(cond.likeEscaped(SqlFunction.lower(CIS.name.tableAlias(aliasCIForEnvironment)), createLike(e).toLowerCase)))
    selectBuilder
  }


  def forDictionary(selectBuilder: SelectBuilder, dictionaryId: Option[String], dictionaryName: Option[String])(implicit schemaInfo: SchemaInfo): SelectBuilder = {
    dictionaryId.map(d => selectBuilder.where(cond.likeEscaped(SqlFunction.lower(CIS.path.tableAlias(aliasCI)), createLike(d).toLowerCase)))
    dictionaryName.map(d => selectBuilder.where(cond.likeEscaped(SqlFunction.lower(CIS.name.tableAlias(aliasCI)), createLike(d).toLowerCase)))
    selectBuilder
  }

  def forValue(selectBuilder: SelectBuilder, value: Option[String])(implicit schemaInfo: SchemaInfo): SelectBuilder = {
    value.map(v => selectBuilder.where(
      cond.likeEscaped(SqlFunction.lower(DictionaryEntriesSchema.value.tableAlias(aliasDictionaryEntries)), createValueLike(v).toLowerCase)
    ))
    selectBuilder
  }

  def forPage(selectBuilder: JoinBuilder, page: Int, resultsPerPage: Int)(implicit schemaInfo: SchemaInfo): Pageable = {
    selectBuilder.showPage(page, resultsPerPage)
  }

  def forOrdering(selectBuilder: SelectBuilder, order: Option[Ordering]): SelectBuilder = {
    evaluateOrderBy(order.filter(o => o.field.equals(CIS.name.name))).foreach(o => selectBuilder.orderBy(o))
    selectBuilder.orderBy(OrderBy.asc(CIS.path))
  }

  def forPlaceholderPage(selectBuilder: Pageable, page: Int, resultsPerPage: Int)(implicit schemaInfo: SchemaInfo): Pageable = {
    selectBuilder.showPage(page, resultsPerPage)
  }

  def forPlaceholderOrdering(selectBuilder: SelectBuilder, ordering: Option[Ordering]): SelectBuilder = {
    ordering match {
      case Some(order) => selectBuilder.orderBy(evaluatePlaceholderOrderBy(order))
      case None => selectBuilder.orderBy(OrderBy.asc(PlaceholderSchema.key))

    }
  }

  private def evaluatePlaceholderOrderBy(o: Ordering): OrderBy = {
    if (o.isAscending)
      OrderBy.asc(SqlPlaceholderSelectQuery.orderFieldMapping(o.field))
    else
      OrderBy.desc(SqlPlaceholderSelectQuery.orderFieldMapping(o.field))
  }


  def buildEqualsKeyAndLikeCiPath(key: Option[String], ciPath: Option[String]): cond = {
    val keyCondition = key.map(k =>
      cond.equals(PlaceholderSchema.key, k)
    )
    val pathCondition = buildLikeCiPathCondition(ciPath)

    keyCondition.map(kc => cond.and(Seq(kc, pathCondition))).getOrElse(pathCondition)
  }

  def buildLikeKeyAndCiPath(key: Option[String], ciPath: Option[String]): cond = {
    val keyCondition = key.map(k =>
      cond.likeEscaped(SqlFunction.lower(PlaceholderSchema.key), createLike(k).toLowerCase)
    ).getOrElse(cond.like(PlaceholderSchema.key, createLike("")))
    val pathCondition = buildLikeCiPathCondition(ciPath)

    cond.and(Seq(keyCondition, pathCondition))
  }

  def buildLikeKeyWithAlias(key: Option[String], columnName: Selectable): cond = {
    key.map(k =>
      cond.likeEscaped(SqlFunction.lower(columnName), createLike(k).toLowerCase)
    ).getOrElse(cond.like(columnName, createLike("")))
  }

  def buildLikeCiPathCondition(ciPath: Option[String]) = {
    ciPath.map(cp =>
      cond.likeEscaped(SqlFunction.lower(PlaceholderSchema.ciPath), createLike(cp).toLowerCase)
    ).getOrElse(cond.like(PlaceholderSchema.ciPath, createLike("")))
  }

  def withPermissionRolesForDictionaries(selectBuilder: SelectBuilder, references: Option[List[String]])
                                        (implicit schemaInfo: SchemaInfo): SelectBuilder = {
    references.map(ref => selectBuilder.where(
      cond.or(Seq(cond.subselect(
        CIS.ID.tableAlias(aliasCI),
        new SelectBuilder(EnvironmentDictionariesSchema.tableName)
          .select(EnvironmentDictionariesSchema.dictionary_id)
          .where(
            cond.subselect(
              EnvironmentDictionariesSchema.environment_id,
              getPermissionReferencesSelectBuilder(ref, CIS.ID)
            )
          )
      ),
        cond.not(
          cond.subselect(CIS.ID.tableAlias(aliasCI),
            new SelectBuilder(EnvironmentDictionariesSchema.tableName).select(EnvironmentDictionariesSchema.dictionary_id))
        )
      ))
    ))
    selectBuilder
  }

  def withPermissionRolesForEnvironments(selectBuilder: SelectBuilder, references: Option[List[String]])
                                        (implicit schemaInfo: SchemaInfo): SelectBuilder = {
    references.map(ref => selectBuilder.where(cond.subselect(
      CIS.path.tableAlias(aliasCIForEnvironment),
      getPermissionReferencesSelectBuilder(ref, CIS.path))))
    selectBuilder
  }


  def withPermissionRolesForPlaceholders(selectBuilder: SelectBuilder, references: Option[List[String]])
                                        (implicit schemaInfo: SchemaInfo): SelectBuilder = {
    references.map(ref => selectBuilder.where(cond.subselect(
      PlaceholderSchema.ciPath,
      getPermissionReferencesSelectBuilder(ref, CIS.path))))
    selectBuilder
  }

  def withPermissionRolesForPlaceholdersId(selectBuilder: SelectBuilder, references: Option[List[String]])
                                          (implicit schemaInfo: SchemaInfo): SelectBuilder = {
    references.map(ref => selectBuilder.where(cond.subselect(
      CIS.ID.tableAlias(aliasDictEntriesUnion),
      getPermissionReferencesSelectBuilder(ref, CIS.ID))))
    selectBuilder
  }

  def withPermissionRolesForDictEntries(selectBuilder: SelectBuilder, column: Selectable, references: Option[List[String]])
                                       (implicit schemaInfo: SchemaInfo): SelectBuilder = {

    references.map(ref => selectBuilder.where(cond.subselect(
      column,
      getPermissionReferencesSelectBuilder(ref, CIS.ID))))
    selectBuilder
  }

  private def getPermissionReferencesSelectBuilder(references: List[String], columnName: Selectable)(implicit schemaInfo: SchemaInfo): SelectBuilder = {
    new SelectBuilder(CIS.tableName).select(columnName).where(cond.in(CIS.secured_directory_ref, references))
  }

  private def createLike(k: String): String = {
    "%" + k + "%"
  }

  private def createValueLike(value: String): String = {
    wrap("%" + value + "%")
  }

  private def evaluateOrderBy(order: Option[Ordering]): Option[OrderBy] = {
    order.map(o =>
      if (o.isAscending)
        OrderBy.asc(PlaceholderSearchQuery.orderFieldMapping(o.field))
      else
        OrderBy.desc(PlaceholderSearchQuery.orderFieldMapping(o.field))
    )
  }

  def wrap(str: String): String = s"u{$str}"

  def unwrap(str: String): String = {
    val Pattern = """u\{(.*)\}""".r
    str match {
      case Pattern(orchName) => orchName
      case s => s
    }
  }
}

object PlaceholderSearchQuery {
  val orderFieldMapping: Map[String, ColumnName] = Map(
    "path" -> CIS.path,
    "name" -> CIS.name
  )
}

object PlaceholderSearchSchema {
  val key = ColumnName("key")
  val dictionaryId = ColumnName("dictionary_id")
}

object QueryUtil {
  def queryEquals(builder: SelectBuilder, column: Selectable, token: Option[String]): Option[builder.type] =
    token.map(queryToken => builder.where(cond.equals(column, queryToken)))

  def queryLike(builder: SelectBuilder, column: Selectable, token: Option[String]): Option[builder.type] =
    token.map(queryToken => builder.where(cond.likeEscaped(SqlFunction.lower(column), s"%$queryToken%".toLowerCase())))
}

class PlaceholdersDictionarySearchQueryBuilder(private val key: Option[String], private val value: Option[String],
                                               private val dictionaryId: Option[String], private val dictionaryName: Option[String],
                                               private val references: Option[List[String]],
                                               private val order: Option[Ordering], private val paging: Paging)
                                              (implicit schemaInfo: SchemaInfo) {

  import PlaceholdersSearchQueryBuilder._

  val selectBuilder: JoinBuilder = joinOnDictionaryIdsWithCi(leftJoinOnKeyWithUnion(new JoinWithUnionBuilder(dictionaryUnion())),
    withPermissionRolesForDictionaries(
      forValue(forKey(forOrdering(forDictionary(new SelectBuilder(CIS.tableName).as(aliasCI)
      .select(DictionaryEntriesSchema.value.tableAlias(aliasDictionaryEntries), "dictValue")
      .select(DictionaryEncryptedEntriesSchema.value.tableAlias(aliasDictionaryEncryptedEntries), "encryptedValue")
      .select(CIS.path.tableAlias(aliasCI))
      .select(CIS.ci_type.tableAlias(aliasCI)), dictionaryId, dictionaryName), order), key), value),
      references)
  )

  Option(paging).map(p => forPage(selectBuilder, p.getPage(), p.getResultsPerPage()))
}

class PlaceholdersEnvironmentSearchQueryBuilder(private val key: Option[String], private val value: Option[String],
                                                private val environmentId: Option[String], private val environmentName: Option[String],
                                                private val references: Option[List[String]],
                                                private val order: Option[Ordering], private val paging: Paging)
                                               (implicit schemaInfo: SchemaInfo) {

  import PlaceholdersSearchQueryBuilder._

  val selectBuilder: JoinBuilder =
    joinOnEnvironmentDictionariesSchema(leftJoinOnKeyWithUnion(new JoinWithUnionBuilder(dictionaryUnion())),
      withPermissionRolesForEnvironments(forValue(forKey(forOrdering(forEnvironment(new SelectBuilder(CIS.tableName).as(aliasCIForEnvironment)
        .select(CIS.path.tableAlias(aliasCIForEnvironment))
        .select(CIS.ci_type.tableAlias(aliasCIForEnvironment))
        .select(CIS.name.tableAlias(aliasCIForEnvironment)), environmentId, environmentName), order), key), value), references)
        .distinct(),
      JoinType.Inner
    )

  Option(paging).map(p => forPage(selectBuilder, p.getPage(), p.getResultsPerPage()))
}

class PlaceholdersDictionarySearchCountQueryBuilder(private val key: Option[String], private val value: Option[String],
                                                    private val dictionaryId: Option[String], private val dictionaryName: Option[String],
                                                    private val references: Option[List[String]])
                                                   (implicit schemaInfo: SchemaInfo) {

  import PlaceholdersSearchQueryBuilder._

  val selectBuilder: JoinBuilder = joinOnDictionaryIdsWithCi(
    leftJoinOnKeyWithUnion(new JoinWithUnionBuilder(dictionaryUnion())),
    withPermissionRolesForDictionaries(forValue(forKey(forDictionary(new SelectBuilder(CIS.tableName).as(aliasCI)
      .select(CIS.path.tableAlias(aliasCI)), dictionaryId, dictionaryName), key), value), references).count()
  )
}

class PlaceholdersEnvironmentSearchCountQueryBuilder(private val key: Option[String], private val value: Option[String],
                                                     private val environmentId: Option[String], private val environmentName: Option[String],
                                                     private val references: Option[List[String]])
                                                    (implicit schemaInfo: SchemaInfo) {

  import PlaceholdersSearchQueryBuilder._

  val selectBuilder: JoinBuilder =
    joinOnEnvironmentDictionariesSchema(leftJoinOnKeyWithUnion(new JoinWithUnionBuilder(dictionaryUnion())),
      withPermissionRolesForEnvironments(forValue(forKey(forEnvironment(new SelectBuilder(CIS.tableName).as(aliasCIForEnvironment)
        .select(CIS.path.tableAlias(aliasCIForEnvironment)), environmentId, environmentName), key), value), references)
        .distinct().count(),
      JoinType.Inner
    )
}

class PlaceholdersSearchQueryBuilder(private val key: Option[String], applicationId: Option[String],
                                     applicationName: Option[String],
                                     references: Option[List[String]], ordering: Option[Ordering], paging: Paging)
                                    (implicit schemaInfo: SchemaInfo) {

  import PlaceholdersSearchQueryBuilder._
  import QueryUtil._

  private val placeholderAlias = "ciPlaceholder"

  private val placeholderBuilder = forPlaceholderOrdering(
    new SelectBuilder(PlaceholderSchema.tableName)
      .select(PlaceholderSchema.id.tableAlias(placeholderAlias))
      .select(PlaceholderSchema.key.tableAlias(placeholderAlias))
      .select(PlaceholderSchema.ciPath.tableAlias(placeholderAlias))
      .select(PlaceholderSchema.ciType.tableAlias(placeholderAlias)).as(placeholderAlias),
    ordering
  )

  queryEquals(placeholderBuilder, PlaceholderSchema.key, key)
  queryLike(placeholderBuilder, PlaceholderSchema.ciPath, applicationId)
  queryLike(placeholderBuilder, PlaceholderSchema.ciName, applicationName)

  Option(paging).map(p => forPlaceholderPage(placeholderBuilder, p.getPage(), p.getResultsPerPage()))



  val selectBuilder: AbstractQueryBuilder =
    references.map {refs =>
      val ciBuilder = new SelectBuilder(CIS.tableName)
        .where(cond.in(CIS.secured_directory_ref, refs))

      new JoinBuilder(placeholderBuilder)
        .join(ciBuilder, cond.equals(PlaceholderSchema.ciPath.tableAlias(placeholderAlias), CIS.path))
    }.getOrElse(placeholderBuilder)
}

class DistinctPlaceholdersSearchQueryBuilder(private val key: Option[String],
                                             references: Option[List[String]], ordering: Option[Ordering], paging: Paging)
                                            (implicit schemaInfo: SchemaInfo) {

  import PlaceholdersSearchQueryBuilder._

  val placeholderBuilder = new SelectBuilder(PlaceholderSchema.tableName).as(aliasPlaceholder).
    select(PlaceholderSchema.key).distinct()

  val dictEntriesBuilder = new SelectBuilder(DictionaryEntriesSchema.tableName).as(aliasDictionaryEntries).
    select(DictionaryEntriesSchema.key).distinct()

  val dictEncEntriesBuilder = new SelectBuilder(DictionaryEncryptedEntriesSchema.tableName).as(aliasDictionaryEncryptedEntries).
    select(DictionaryEncryptedEntriesSchema.key).distinct()


  val keysUnionBuilder = new JoinWithUnionBuilder(
    new UnionBuilder(placeholderBuilder, dictEntriesBuilder, dictEncEntriesBuilder).as(aliasDictEntriesUnion)
  )

  val selectBuilder: JoinBuilder =
    keysUnionBuilder.join(
      new SelectBuilder(PlaceholderSchema.tableName).select(PlaceholderSchema.key.tableAlias(aliasDictEntriesUnion))
        .where(buildLikeKeyWithAlias(key, PlaceholderSchema.key.tableAlias(aliasDictEntriesUnion)))
        .distinct().as(aliasPhJoin),
      cond.equals(PlaceholderSchema.key.tableAlias(aliasPhJoin), PlaceholderSchema.key.tableAlias(aliasDictEntriesUnion)),
      JoinType.Left)
      .join(
        new SelectBuilder(DictionaryEntriesSchema.tableName).as(aliasDeJoin),
        cond.equals(DictionaryEntriesSchema.key.tableAlias(aliasDeJoin), DictionaryEntriesSchema.key.tableAlias(aliasDictEntriesUnion)),
        JoinType.Left)
      .join(
        new SelectBuilder(DictionaryEncryptedEntriesSchema.tableName).as(aliasDeeJoin)
          .orderBy(OrderBy.asc(DictionaryEncryptedEntriesSchema.key.tableAlias(aliasDictEntriesUnion))),
        cond.equals(DictionaryEntriesSchema.key.tableAlias(aliasDeeJoin), DictionaryEncryptedEntriesSchema.key.tableAlias(aliasDictEntriesUnion)),
        JoinType.Left
      )

  val placeholderBuilerWithPermission = withPermissionRolesForPlaceholders(
    new SelectBuilder(PlaceholderSchema.tableName).as(aliasPlaceholder).
      select(PlaceholderSchema.key).where(buildLikeKeyAndCiPath(key, None)).orderBy(OrderBy.asc(PlaceholderSchema.key)),
    references).distinct()

  val dictEntriesBuilderWithPermission = withPermissionRolesForDictEntries(
    new SelectBuilder(DictionaryEntriesSchema.tableName).as(aliasDictionaryEntries).
      select(DictionaryEntriesSchema.key).where(buildLikeKeyWithAlias(key, DictionaryEntriesSchema.key)).orderBy(OrderBy.asc(DictionaryEntriesSchema.key)),
    DictionaryEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEntries), references).distinct()

  val dictEncEntriesBuilderWithPermission = withPermissionRolesForDictEntries(
    new SelectBuilder(DictionaryEncryptedEntriesSchema.tableName).as(aliasDictionaryEncryptedEntries).
      select(DictionaryEncryptedEntriesSchema.key).where(buildLikeKeyWithAlias(key, DictionaryEncryptedEntriesSchema.key)).orderBy(OrderBy.asc(DictionaryEncryptedEntriesSchema.key)),
    DictionaryEncryptedEntriesSchema.dictionary_id.tableAlias(aliasDictionaryEncryptedEntries), references).distinct()

}

class PlaceholdersSearchIdsQueryBuilder(private val ids: Iterable[String])
                                       (implicit schemaInfo: SchemaInfo) {

  val selectBuilder: SelectBuilder = new SelectBuilder(PlaceholderSchema.tableName)
    .select(PlaceholderSchema.key)
    .select(PlaceholderSchema.ciName)
    .select(PlaceholderSchema.ciPath)
    .select(PlaceholderSchema.ciType)
    .where(cond.in(PlaceholderSchema.ciPath, ids))

}

class PlaceholdersSearchCountQueryBuilder(private val key: Option[String], id: Option[String], name: Option[String],
                                          references: Option[List[String]])
                                         (implicit schemaInfo: SchemaInfo) {

  import QueryUtil._

  private val builder: SelectBuilder =
    new SelectBuilder(PlaceholderSchema.tableName)
      .select(SqlFunction.countAll)

  queryEquals(builder, PlaceholderSchema.key, key)
  queryLike(builder, PlaceholderSchema.ciPath, id)
  queryLike(builder, PlaceholderSchema.ciName, name)

  val selectBuilder: AbstractQueryBuilder =
    references.map { refs =>
      val ciBuilder = new SelectBuilder(CIS.tableName)
        .where(cond.in(CIS.secured_directory_ref, refs))
      new JoinBuilder(builder)
        .join(ciBuilder, cond.equals(PlaceholderSchema.ciPath, CIS.path))
    }.getOrElse(builder)
}
