package com.xebialabs.deployit.repository.sql.specific

import com.xebialabs.deployit.core.sql.SelectBuilder
import com.xebialabs.deployit.core.sql.batch.BatchCommandWithSetter
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.reader.properties.CiDataProvider
import com.xebialabs.deployit.repository.sql.specific.configurable.ConfigurableTypeSpecificPersisterFactory.CiFromMapReader
import com.xebialabs.deployit.repository.sql.specific.configurable.PropertyTable

import java.util.{List => JList}

trait TypeSpecificPersisterFactory {
  def createReader(t: Type, pk: CiPKType): Option[TypeSpecificReader]

  def createInserter(t: Type, pk: CiPKType): Option[TypeSpecificInserter]

  def createUpdater(t: Type, pk: CiPKType): Option[TypeSpecificUpdater]

  def createDeleter(t: Type, pk: CiPKType): Option[TypeSpecificDeleter]

  def createReferenceFinder(pks: Seq[CiPKType]): TypeSpecificReferenceFinder
}

trait TypeSpecificReader {
  val tables: Map[String, PropertyTable[_, _]]

  def readProperties(ci: ConfigurationItem,
                     ciResolver: CiPKType => ConfigurationItem,
                     convert: (PropertyDescriptor, Any) => Any)
                    (implicit ciDataProvider: CiDataProvider,
                     readFromMap: CiFromMapReader,
                     typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): Unit
}

trait TypeSpecificInserter {
  def insertProperty(propertyName: String, value: Any): Boolean

  def finish(handler: PropertyErrorHandler = defaultErrorHandler): Unit

  def batchFinish(): List[BatchCommandWithSetter]
}

trait TypeSpecificUpdater {
  def updateProperty(propertyName: String, value: Any): Boolean

  def finish()(implicit ciDataProvider: CiDataProvider): Unit

  def batchFinish()(implicit ciDataProvider: CiDataProvider): List[BatchCommandWithSetter]
}

trait TypeSpecificDeleter {
  def deleteProperties(): Unit
}

trait TypeSpecificReferenceFinder {
  def findReferences(): Seq[(CiPKType, CiPKType)]

  def findReferencesBuilders(): Seq[SelectBuilder]
}

abstract class AbstractTypeSpecificPersisterFactory(myType: Type) extends TypeSpecificPersisterFactory {

  override def createReader(t: Type, pk: CiPKType): Option[TypeSpecificReader] = ifCanHandle(t) {
    createReader(pk)
  }

  def createReader(pk: CiPKType): TypeSpecificReader

  override def createInserter(t: Type, pk: CiPKType): Option[TypeSpecificInserter] = ifCanHandle(t) {
    createInserter(pk)
  }

  def createInserter(pk: CiPKType): TypeSpecificInserter

  override def createUpdater(t: Type, pk: CiPKType): Option[TypeSpecificUpdater] = ifCanHandle(t) {
    createUpdater(pk)
  }

  def createUpdater(pk: CiPKType): TypeSpecificUpdater

  override def createDeleter(t: Type, pk: CiPKType): Option[TypeSpecificDeleter] = ifCanHandle(t) {
    createDeleter(pk)
  }

  def createDeleter(pk: CiPKType): TypeSpecificDeleter

  private def ifCanHandle[T](t: Type)(block: => T) =
    if (canHandle(t)) Option(block) else None

  protected def canHandle[T](t: Type): Boolean = {
    t.instanceOf(myType)
  }
}
