package com.xebialabs.deployit.repository.sql.persisters

import com.xebialabs.deployit.core.sql.batch.BatchCommandWithSetter
import com.xebialabs.deployit.core.sql.{ColumnName, SchemaInfo, TableName}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.repository.sql.base.CiPKType
import com.xebialabs.deployit.repository.sql.persisters.SourceArtifactSchema.checksum
import com.xebialabs.deployit.repository.sql.reader.properties.CiDataProvider
import com.xebialabs.deployit.repository.sql.specific.TypeSpecificUpdater
import com.xebialabs.deployit.repository.sql.specific.configurable.{ConfigurableTypeSpecificPersisterFactory, PropertyColumnsTable, TypeConfiguration}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component

@Component
class SourceArtifactPersisterFactory (@Autowired @Qualifier("mainJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                      @Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ConfigurableTypeSpecificPersisterFactory(Type.valueOf(classOf[SourceArtifact])) {
  import SourceArtifact._
  import SourceArtifactSchema._
  override val config: TypeConfiguration = new TypeConfiguration(
    Some(new PropertyColumnsTable(tableName, ID, Map(CHECKSUM_PROPERTY_NAME -> checksum, FILE_URI_PROPERTY_NAME -> file_uri)))
  )

  override def createUpdater(pk: CiPKType): TypeSpecificUpdater =  new ConditionalOnChecksumUpdater(super.createUpdater(pk))
}

object SourceArtifactSchema {
  val tableName: TableName = TableName("XLD_SOURCE_ARTIFACTS")

  val ID: ColumnName = ColumnName("ID")
  val checksum: ColumnName = ColumnName("checksum")
  val file_uri: ColumnName = ColumnName("file_uri")
  val filename: ColumnName = ColumnName("filename")
}

class ConditionalOnChecksumUpdater(val delegate: TypeSpecificUpdater) extends TypeSpecificUpdater {
  var hasChecksum = false
  override def updateProperty(propertyName: String, value: Any): Boolean = {
    hasChecksum = hasChecksum || (propertyName == checksum.name && value != null)
    delegate.updateProperty(propertyName, value)
  }
  override def finish()(implicit ciDataProvider: CiDataProvider): Unit = if (hasChecksum) {
    delegate.finish()
  }
  override def batchFinish()(implicit ciDataProvider: CiDataProvider): List[BatchCommandWithSetter] =
    if (hasChecksum)
      delegate.batchFinish()
    else
      List.empty
}
