package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.SearchParameters
import com.xebialabs.deployit.repository.sql.CiConditions._
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.security.sql.PermissionsSchema
import com.xebialabs.deployit.security.sql.RolesSchema.{RoleRoles, Roles}
import org.joda.time.DateTime

import java.util
import scala.jdk.CollectionConverters._

abstract class AbstractSearchParametersSelectBuilder(val selectBuilder: SelectBuilder,
                                                     private val criteria: SearchParameters)
                                                    (implicit schemaInfo: SchemaInfo) {

  withType(selectBuilder, criteria.getType)
  withParent(selectBuilder, criteria.getParent)
  withAncestor(selectBuilder, criteria.getAncestor)
  withName(selectBuilder, criteria.getName, criteria.isExactNameSearch)
  withId(selectBuilder, criteria.getId, criteria.isExactIdSearch)
  withBefore(selectBuilder, criteria.getBefore)
  withAfter(selectBuilder, criteria.getAfter)
  withProperties(selectBuilder, criteria.getProperties)
  withInternalId(selectBuilder, criteria.getInternalId)
  forPage(selectBuilder, criteria.getPage.toInt + 1, criteria.getResultsPerPage.toInt)

  private[this] def withInternalId(selectBuilder: SelectBuilder, internalId: Integer) = {
    if (internalId != null) {
      selectBuilder.where(internalIdMatch(internalId))
    }
  }

  private[this] def withType(selectBuilder: SelectBuilder, ciType: Type) = {
    if (ciType != null) {
      selectBuilder.where(ofType(ciType))
    }
  }

  private[this] def withParent(selectBuilder: SelectBuilder, parent: String) = {
    if (parent != null) {
      selectBuilder.where(childOf(parent))
    }
  }

  private[this] def withAncestor(selectBuilder: SelectBuilder, ancestor: String) = {
    if (ancestor != null) {
      selectBuilder.where(descendantOf(ancestor))
    }
  }

  private[this] def withName(selectBuilder: SelectBuilder, value: String, exact: Boolean) = {
    if (value != null) {
      selectBuilder.where(nameMatch(value, exact))
    }
  }

  private[this] def withId(selectBuilder: SelectBuilder, value: String, exact: Boolean) = {
    if (value != null) {
      selectBuilder.where(idMatch(value, exact))
    }
  }

  private[this] def withBefore(selectBuilder: SelectBuilder, cal: util.Calendar) = {
    if (cal != null) {
      selectBuilder.where(modifiedBefore(new DateTime(cal)))
    }
  }

  private[this] def withAfter(selectBuilder: SelectBuilder, cal: util.Calendar) = {
    if (cal != null) {
      selectBuilder.where(modifiedAfter(new DateTime(cal)))
    }
  }

  private[this] def withProperties(builder: SelectBuilder, properties: util.Map[String, String])(implicit schemaInfo: SchemaInfo) = {
    if (!properties.isEmpty) {
      builder.where(hasProperties(properties))
    }
  }

  private[this] def forPage(selectBuilder: SelectBuilder, page: Int, resultsPerPage: Int)(implicit schemaInfo: SchemaInfo): Unit = {
    selectBuilder.showPage(page, resultsPerPage)
  }
}

class SearchParametersSelectBuilder(val criteria: SearchParameters)(implicit schemaInfo: SchemaInfo)
  extends AbstractSearchParametersSelectBuilder(
    new SelectBuilder(CIS.tableName).select(CIS.path).select(CIS.ci_type).orderBy(OrderBy.asc(CIS.path)), criteria
  )

class SearchParametersExtendedSelectBuilder(val criteria: SearchParameters)(implicit schemaInfo: SchemaInfo)
  extends AbstractSearchParametersSelectBuilder(
    new SelectBuilder(CIS.tableName).select(CIS.path).select(CIS.name).select(CIS.ID).select(CIS.ci_type).orderBy(OrderBy.asc(CIS.path)), criteria
  )

class SearchParametersFullSelectBuilder(val criteria: SearchParameters)(implicit schemaInfo: SchemaInfo)
  extends AbstractSearchParametersSelectBuilder(
    new SelectBuilder(CIS.tableName)
      .select(CIS.ID).select(CIS.ci_type).select(CIS.name).select(CIS.parent_id).select(CIS.secured_ci).select(CIS.token)
      .select(CIS.created_at).select(CIS.created_by).select(CIS.modified_at).select(CIS.modified_by).select(CIS.scm_traceability_data_id)
      .select(CIS.path).orderBy(OrderBy.asc(CIS.path)), criteria
  )

class SearchParametersCountBuilder(val criteria: SearchParameters)(implicit val schemaInfo: SchemaInfo)
  extends AbstractSearchParametersSelectBuilder(
    new SelectBuilder(CIS.tableName).select(SqlFunction.countOne), criteria: SearchParameters
  )

class WithPermissionsBuilder(origSelectBuilder: SelectBuilder, roles: util.List[String], permissions: util.List[String])
                            (implicit val schemaInfo: SchemaInfo) {
  private val aliasCIs = "c"
  private val aliasPermissions = "p"
  private val aliasRoles = "r"
  val selectBuilder: JoinBuilder = new JoinBuilder(origSelectBuilder
    .distinct()
    .as(aliasCIs))
    .join(selectPermissionsWithRoles, joinPermissionsOnSecuredCiAndPermissions)
    .join(new SelectBuilder(Roles.tableName).as(aliasRoles), joinRolesOnSecuredCiAndPermissionRoleId, JoinType.Left)

  private def joinRolesOnSecuredCiAndPermissionRoleId: cond = {
    cond.and(Seq(
      cond.equals(CIS.secured_ci.tableAlias(aliasCIs), Roles.CI_ID.tableAlias(aliasRoles)),
      cond.equals(PermissionsSchema.ROLE_ID.tableAlias(aliasPermissions), Roles.ID.tableAlias(aliasRoles)))
    )
  }

  private def joinPermissionsOnSecuredCiAndPermissions: cond = {
    cond.and(Seq(
      cond.equals(CIS.secured_ci.tableAlias(aliasCIs), PermissionsSchema.CI_ID.tableAlias(aliasPermissions)),
      cond.in(PermissionsSchema.PERMISSION_NAME.tableAlias(aliasPermissions), permissions.asScala)))
  }

  private def selectPermissionsWithRoles: SelectBuilder = {
    new SelectBuilder(PermissionsSchema.tableName).as(aliasPermissions).where(
      cond.or(Seq(
        cond.in(PermissionsSchema.ROLE_ID.tableAlias(aliasPermissions), roles.asScala),
        cond.in(Roles.ID.tableAlias(aliasRoles), roles.asScala),
        cond.subselect(Roles.ID.tableAlias(aliasRoles), new SelectBuilder(RoleRoles.tableName).select(RoleRoles.ROLE_ID).where(cond.in(RoleRoles.MEMBER_ROLE_ID, roles.asScala)))
      ))
    )
  }
}
