package com.xebialabs.deployit.repository.sql.base

import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs}
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util.queryWithInClause
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo, SelectBuilder, SqlCondition => cond}
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.sql.base.schema.CIS
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

trait DirectoryRef extends DirectoryRefQueries {
  val jdbcTemplate: JdbcTemplate

  private def extractResultDirectoryRef(pk: CiPKType, resultSetExtractor: ResultSet => String): Option[String] =
    jdbcTemplate.query(SELECT_DIRECTORY_REF, new RowMapper[String] {
      override def mapRow(rs: ResultSet, rowNum: Int): String = {
        resultSetExtractor(rs)
      }
    }, pk).asScala.find(_ != null)

  def getDirectoryUuid(pk: CiPKType): Option[String] =
    extractResultDirectoryRef(pk, rs => rs.getString(1))

  def getDirectoryReference(pk: CiPKType): Option[String] =
    extractResultDirectoryRef(pk, rs => rs.getString(2))

  def resolveDirectoryRef(pk: CiPKType): String =
    extractResultDirectoryRef(pk,
      rs => extractDirectoryRef(rs.getString(1),
        rs.getString(2)).getOrElse(throw MissingDirectoryReferenceException(pk)))
      .getOrElse(throw MissingDirectoryReferenceException(pk))

  class CisSearchDirRefQueryBuilder(private val pks: Iterable[CiPKType])
                                  (implicit schemaInfo: SchemaInfo) {

    val selectBuilder: SelectBuilder = new SelectBuilder(CIS.tableName)
      .select(CIS.ID)
      .select(CIS.directory_uuid)
      .select(CIS.directory_ref)
      .where(cond.in(CIS.ID, pks))
  }

  private def extractDirectoryRef[T](directoryId: T, directoryRef: T): Option[T] = {
    Option(directoryId).orElse(Option(directoryRef)
      .orElse(None))
  }

  def resolveDirectoryRefs(pks: Iterable[CiPKType]): List[(CiPKType, Option[String])] =
    queryWithInClause(pks.toSet.toSeq) { pkGroup =>
      val selectBuilder = new CisSearchDirRefQueryBuilder(pkGroup).selectBuilder
      jdbcTemplate
        .query(selectBuilder.query, Setter(selectBuilder.parameters), MapRowMapper)
        .asScala
        .map { m =>
          val directoryRef =
            extractDirectoryRef(m.get(CIS.directory_uuid.name), m.get(CIS.directory_ref.name)) match {
              case Some(s: String) => Some(s)
              case _ => None
            }
          (asCiPKType(m.get(CIS.ID.name)), directoryRef)
        }
    }

  def setDirectoryRef(pk: CiPKType, directoryReference: String): Unit = jdbcTemplate.update(UPDATE_DIRECTORY_REF, directoryReference, pk)

  def batchSetDirectoryRef(pk: CiPKType, directoryReference: String): BatchCommandWithArgs = BatchCommand(UPDATE_DIRECTORY_REF, directoryReference, pk)

}

trait DirectoryRefQueries extends Queries {

  lazy val SELECT_DIRECTORY_REF: String = {
    import CIS._
    sqlb"select $directory_uuid, $directory_ref from $tableName where $ID = ?"
  }

  lazy val UPDATE_DIRECTORY_REF: String = {
    import CIS._
    sqlb"update $tableName set $directory_ref = ? where $ID = ?"
  }

}

case class MissingDirectoryReferenceException(primaryId: CiPKType) extends DeployitException(s"Missing directory reference for CI ID: $primaryId")
