package com.xebialabs.deployit.repository

import java.util
import java.util.{Set => JSet}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.CiRepository
import com.xebialabs.deployit.util.Tuple
import com.xebialabs.license.service.LicenseService
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
class SqlRepositoryService(@Autowired val ciRepository: CiRepository, @Autowired val licenseService: LicenseService) extends RepositoryService {
  override def createCis[T <: ConfigurationItem](entities: util.List[T], scmTraceabilityData: SCMTraceabilityData): Unit =
    execute(changeSet(scmTraceabilityData, entities.asScala) { case (c, e) => c.create(e) })

  override def create[T <: ConfigurationItem](entity: T*): Unit = createCis(entity.toList.asJava, null)

  override def updateCis[T <: ConfigurationItem](entities: util.List[T], scmTraceabilityData: SCMTraceabilityData): Unit =
    execute(changeSet(scmTraceabilityData, entities.asScala) { (c, e) => c.update(e) })

  override def update[T <: ConfigurationItem](entity: T*): Unit = updateCis(entity.toList.asJava, null)

  override def createOrUpdateCis[T <: ConfigurationItem](entities: util.List[T], scmTraceabilityData: SCMTraceabilityData): Unit =
    execute(changeSet(scmTraceabilityData, entities.asScala) { (c, e) => c.createOrUpdate(e) })

  override def createOrUpdate[T <: ConfigurationItem](entity: T*): Unit = createOrUpdateCis(entity.toList.asJava, null)

  override def delete(id: String*): Unit = execute(changeSet(null, id) { (c, i) => c.deleteById(i) })

  override def copy[T <: ConfigurationItem](id: String, newId: String): Unit = execute(changeSet(_.copy(id, newId)))

  override def move[T <: ConfigurationItem](id: String, newId: String): Unit = execute(changeSet(_.move(id, newId)))

  override def rename(id: String, newName: String): Unit = execute(changeSet(_.rename(id, newName)))

  override def exists(id: String): Boolean = ciRepository.exists(id)

  override def selectExistingPaths(ids: JSet[String]): JSet[String] = ciRepository.selectExistingPaths(ids)

  override def readByReferenceId[T <: ConfigurationItem](referenceId: String, depth: Int, workDir: WorkDir, useCache: Boolean, decryptPasswords: Boolean): T =
    ciRepository.readByReferenceId(referenceId, workDir, depth, useCache, decryptPasswords)

  override def read[T <: ConfigurationItem](id: String, depth: Int, workDir: WorkDir, useCache: Boolean, decryptPasswords: Boolean): T =
    ciRepository.read(id, workDir, depth, useCache, decryptPasswords)

  override def read[T <: ConfigurationItem](ids: util.List[String], depth: Int, useCache: Boolean, decryptPasswords: Boolean, skipNotExistingCis: Boolean): util.List[T] =
    ciRepository.read(ids, depth, useCache, decryptPasswords, skipNotExistingCis)

  override def list(criteria: SearchParameters): util.List[ConfigurationItemData] = ciRepository.list(criteria)

  override def listEntities[T <: ConfigurationItem](criteria: SearchParameters): util.List[T] = listEntities(criteria, Integer.MAX_VALUE, useCache = true)

  override def listEntities[T <: ConfigurationItem](criteria: SearchParameters, depth: Int, useCache: Boolean): util.List[T] =
    ciRepository.listEntities(criteria, depth, useCache)

  override def execute(changeSet: ChangeSet): Unit = execute(changeSet, new NullProgressLogger)

  override def execute(changeSet: ChangeSet, progressLogger: ProgressLogger): Unit = {
    val licenseTransaction = licenseService.startTransaction()
    try {
      ciRepository.execute(changeSet, progressLogger, licenseService.getCounter, licenseTransaction)
    } catch {
      case e: Exception =>
        licenseService.rollbackTransaction(licenseTransaction)
        throw e
    }
  }

  def findReferencesTo(id: String, searchParameters: SearchParameters): util.List[ConfigurationItemData] =
    ciRepository.findReferencesTo(id, searchParameters)

  private def changeSet[T](scmTraceabilityData: SCMTraceabilityData, entity: Iterable[T])(op: (ChangeSet, T) => Unit) = {
    val initial = new ChangeSet
    initial.setSCMTraceabilityData(scmTraceabilityData)
    entity.foldLeft(initial)((acc, entity) => {
      op(acc, entity)
      acc
    })
  }

  private def changeSet(block: ChangeSet => Unit) = {
    val cs = new ChangeSet
    block(cs)
    cs
  }

  override def checkReferentialIntegrity(changeSet: ChangeSet): Unit = {
    val licenseTransaction = licenseService.startTransaction()
    try {
      ciRepository.validate(changeSet, licenseService.getCounter, licenseTransaction)
    } finally {
      licenseService.rollbackTransaction(licenseTransaction)
    }
  }

  override def count(criteria: SearchParameters): Long = ciRepository.count(criteria).toLong
}
