package com.xebialabs.deployit.repository.sql.properties

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util.queryWithInClause
import com.xebialabs.deployit.core.sql.{JoinBuilder, OrderBy, Queries, QueryBuilder, SelectBuilder, SqlLiteral, SqlCondition => cond}
import com.xebialabs.deployit.repository.sql.base.CiPKType
import com.xebialabs.deployit.repository.sql.specific.configurable.PropertyTableSelectDescriptor
import com.xebialabs.deployit.sql.base.schema.CIS
import org.springframework.jdbc.core.JdbcTemplate

import java.util.{Map => JMap}
import scala.jdk.CollectionConverters._

trait PropertyTableRepository extends PropertyTableRepositoryQueries {
  protected def jdbcTemplate: JdbcTemplate

  def readPropertyTableForCis(pks: Seq[CiPKType],
                              selectDescriptor: PropertyTableSelectDescriptor): List[JMap[String, Object]] =
    queryWithInClause(pks) { group =>
      val select = buildPropertyTableValuesQuery(group, selectDescriptor)
      jdbcTemplate.query(select.query, Setter(select.parameters), MapRowMapper).asScala
    }
}

trait PropertyTableRepositoryQueries extends Queries {
  protected def buildPropertyTableValuesQuery(pks: Seq[CiPKType], selectDescriptor: PropertyTableSelectDescriptor): QueryBuilder = {
    val propsAlias = "p"
    val cisAlias = "cis"

    val initialBuilder = new SelectBuilder(selectDescriptor.tableName)
      .select(selectDescriptor.ciIdColumn, s"${propsAlias}_${selectDescriptor.ciIdColumn.name}")
      .as(propsAlias)

    val select = selectDescriptor
      .selectColumns
      .foldLeft(initialBuilder) { case (builder, column) => builder.select(column) }
      .where(cond.in(selectDescriptor.ciIdColumn, pks))

    val orderedSelect = (OrderBy.asc(selectDescriptor.ciIdColumn) :: selectDescriptor.orderColumns)
      .foldLeft(select) { case (builder, order) => builder.orderBy(order) }

    selectDescriptor.ciRefColumn.map { refColumn =>
      new JoinBuilder(orderedSelect.select(new SqlLiteral(s"$cisAlias.*"))).join(
        table = new SelectBuilder(CIS.tableName).as(cisAlias),
        cond = cond.equals(CIS.ID.tableAlias(cisAlias), refColumn.tableAlias(propsAlias))
      )
    }.getOrElse(orderedSelect)
  }
}
