package com.xebialabs.deployit.repository.sql.properties

import com.xebialabs.deployit.core.sql.SqlFunction.?
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs}
import com.xebialabs.deployit.core.sql.{ColumnName, JoinBuilder, JoinType, Queries, SelectBuilder, SqlLiteral, SqlCondition => cond}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyKind, Type}
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.repository.sql.specific.TypeSpecificInserter
import org.springframework.jdbc.core.JdbcTemplate

import java.util
import scala.jdk.CollectionConverters._

trait CiPropertiesRepository extends CiPropertiesQueries {
  def insertProperty(pk: CiPKType, propName: String, kind: PropertyKind, value: Any, handler: PropertyErrorHandler = defaultErrorHandler): Unit = handler(propName) {
    jdbcTemplate.update(insertProperty(kind), pk, propName, value.asInstanceOf[Object])
  }

  def insertIndexedProperty[T <: Object](pk: CiPKType, propName: String, kind: PropertyKind, values: util.Collection[T], start: Int = 0, handler: PropertyErrorHandler = defaultErrorHandler): Unit = {
    var idx = start
    values.forEach { value =>
      handler(propName) {
        jdbcTemplate.update(insertProperty(kind), pk, propName, idx.asInstanceOf[Integer], value)
        idx = idx + 1
      }
    }
  }

  def insertIndexedCiPropertyLastPosition(pk: CiPKType, propName: String, value: CiPKType): Unit = {
    val i: Number = jdbcTemplate.queryForObject(FIND_NEXT_IDX_LIST_PROPERTY, classOf[Number], pk, propName)
    jdbcTemplate.update(insertProperty(PropertyKind.LIST_OF_CI), pk, propName, i, value)
  }

  def insertKeyedProperty(pk: CiPKType, propName: String, kind: PropertyKind, values: util.Map[String, String], handler: PropertyErrorHandler = defaultErrorHandler): Unit = {
    values.entrySet().forEach { entry =>
      handler(propName) {
        jdbcTemplate.update(insertProperty(kind), pk, propName, entry.getKey, entry.getValue)
      }
    }
  }

  def batchInsertProperty(pk: CiPKType, propName: String, kind: PropertyKind, value: Any): BatchCommandWithArgs =
    BatchCommand(insertProperty(kind), pk, propName, value.asInstanceOf[Object])

  def batchInsertIndexedProperty[T <: Object](pk: CiPKType, propName: String, kind: PropertyKind, values: util.Collection[T], start: Int = 0): Iterable[BatchCommandWithArgs] = {
    var idx = start
    values.asScala.map { value =>
      val c = BatchCommand(insertProperty(kind), pk, propName, idx.asInstanceOf[Integer], value)
      idx = idx + 1
      c
    }
  }

  def batchInsertIndexedCiPropertyLastPosition(pk: CiPKType, propName: String, value: CiPKType): BatchCommandWithArgs = {
    val i: Number = jdbcTemplate.queryForObject(FIND_NEXT_IDX_LIST_PROPERTY, classOf[Number], pk, propName)
    BatchCommand(insertProperty(PropertyKind.LIST_OF_CI), pk, propName, i, value)
  }

  def batchInsertKeyedProperty(pk: CiPKType, propName: String, kind: PropertyKind, values: util.Map[String, String]): Iterable[BatchCommandWithArgs] =
    values.entrySet().asScala.map { entry =>
      BatchCommand(insertProperty(kind), pk, propName, entry.getKey, entry.getValue)
    }

  protected def jdbcTemplate: JdbcTemplate

}

trait InsertCiPropertiesRepository extends CiPropertiesRepository {
  def createTypeSpecificInserters(t: Type, pk: CiPKType): List[TypeSpecificInserter]
}

trait UpdateCiPropertiesRepository extends CiPropertiesRepository {
  def updateProperty(propPk: CiPKType, kind: PropertyKind, value: Any): Unit =
    jdbcTemplate.update(updateProperty(kind), value.asInstanceOf[Object], propPk)

  def batchUpdateProperty(propPk: CiPKType, kind: PropertyKind, value: Any): BatchCommandWithArgs =
    BatchCommand(updateProperty(kind), value.asInstanceOf[Object], propPk)
}

trait DeleteRepositoryCiProperties extends DeleteCiPropertiesQueries {
  def deleteProperty(propPk: CiPKType): Unit = jdbcTemplate.update(DELETE_PROPERTY, propPk)

  def batchDeleteProperty(propPk: CiPKType): BatchCommandWithArgs = BatchCommand(DELETE_PROPERTY, propPk)

  protected def jdbcTemplate: JdbcTemplate
}

trait DeleteCiPropertiesQueries extends Queries {

  lazy val DELETE_PROPERTY: String = {
    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES._
    sqlb"delete from $tableName where $ID = ?"
  }

}

trait CiPropertiesQueries extends Queries {

  import PropertyKind._

  lazy val FIND_NEXT_IDX_LIST_PROPERTY: String = {
    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES._
    sqlb"select COALESCE(MAX($idx) + 1, 0) from $tableName where $ci_id = ? and $name = ?"
  }

  lazy val SELECT_PROPERTIES: String = {
    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES._
    sqlb"select * from $tableName where $ci_id = ?"
  }

  private def buildPropertiesWithRefsQuery(condition: cond): String = {
    val cisAlias = "ci"
    val propertiesAlias = "props"

    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES._
    new JoinBuilder(
      new SelectBuilder(tableName)
        .as(propertiesAlias)
        .select(new SqlLiteral(s"$propertiesAlias.*"))
        .where(condition)
    ).join(
      table = CIS
        .allFields
        .foldLeft(new SelectBuilder(CIS.tableName).as(cisAlias)) { case (acc, column) =>
          acc.select(column, s"a_${column.name}")
        },
      cond = cond.equals(CI_PROPERTIES.ci_ref_value.tableAlias(propertiesAlias), CIS.ID.tableAlias(cisAlias)),
      joinType = JoinType.Left
    )
  }.query

  def buildSelectPropertiesWithRefsByCiIds(ciIds: Iterable[CiPKType]): String =
    buildPropertiesWithRefsQuery(cond.in(CI_PROPERTIES.ci_id, ciIds))

  lazy val SELECT_PROPERTIES_WITH_REFS: String = buildPropertiesWithRefsQuery(cond.equals(CI_PROPERTIES.ci_id, ?))

  def insertProperty(kind: PropertyKind): String = {
    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES._
    val valueColumn = ColumnName(propertyType(kind) + "_value")
    kind match {
      case LIST_OF_CI | LIST_OF_STRING | SET_OF_CI | SET_OF_STRING =>
        sqlb"insert into $tableName ($ci_id, $name, $idx, $valueColumn) values (?, ?, ?, ?)"
      case MAP_STRING_STRING =>
        sqlb"insert into $tableName ($ci_id, $name, $key, $valueColumn) values (?, ?, ?, ?)"
      case _ =>
        sqlb"insert into $tableName ($ci_id, $name, $valueColumn) values (?, ?, ?)"
    }
  }

  private def propertyType(kind: PropertyKind) = kind match {
    case BOOLEAN => "boolean"
    case INTEGER => "integer"
    case ENUM | STRING | LIST_OF_STRING | MAP_STRING_STRING | SET_OF_STRING => "string"
    case DATE => "date"
    case CI | LIST_OF_CI | SET_OF_CI => "ci_ref"
  }

  def updateProperty(kind: PropertyKind): String = {
    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES._
    val valueColumn = ColumnName(propertyType(kind) + "_value")
    sqlb"update $tableName set $valueColumn = ? where $ID = ?"
  }
}
