package com.xebialabs.deployit.repository.sql.base

import java.sql.ResultSet
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.util.queryWithInClause
import com.xebialabs.deployit.core.sql.{OrderBy, Queries, SelectBuilder, SqlCondition => cond}
import com.xebialabs.deployit.plugin.api.reflect.Type
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import com.xebialabs.deployit.sql.base.schema.CIS

import scala.jdk.CollectionConverters._

trait CiExists extends CiExistsQueries {
  val jdbcTemplate: JdbcTemplate

  def exists(id: String): Boolean = jdbcTemplate.queryForObject(COUNT_CIS_BY_PATH, classOf[Integer], idToPath(id)) == Integer.valueOf(1)

  def doWithFirstThatExists(ids: Seq[String])(callback: String => Unit): Unit = queryWithInClause(ids.map(idToPath)) { group =>
    jdbcTemplate.query(buildSelectFirstThatExistsQuery(group), Setter(group), new RowMapper[Unit] {
      override def mapRow(rs: ResultSet, rowNum: Int): Unit = callback(rs.getString(1))
    })
    Nil
  }

  def selectExistingPaths(paths: Seq[String]): Set[String] = queryWithInClause(paths) { group =>
    jdbcTemplate.query(buildSelectPathsThatExistQuery(group), Setter(group), new RowMapper[String] {
      override def mapRow(rs: ResultSet, rowNum: Int): String = rs.getString(1)
    }).asScala
  }.toSet

  def selectExistingCiTypes(paths: Seq[String]): Map[String, Type] = queryWithInClause(paths) { group =>
    jdbcTemplate.query(buildSelectCiTypesThatExistQuery(group), Setter(group), new RowMapper[(String, Type)] {
      override def mapRow(rs: ResultSet, rowNum: Int): (String, Type) =
        rs.getString(1) -> Type.valueOf(rs.getString(2))
    }).asScala
  }.toMap
}

trait CiExistsQueries extends Queries {
  lazy val COUNT_CIS_BY_PATH: String = {
    import com.xebialabs.deployit.sql.base.schema.CIS._
    sqlb"select count(*) from $tableName where $path = ?"
  }

  private def createSelectPathsByPathsBuilder(paths: Iterable[String]): SelectBuilder =
    new SelectBuilder(CIS.tableName)
      .select(CIS.path)
      .where(cond.in(CIS.path, paths))

  def buildSelectFirstThatExistsQuery(paths: Iterable[String]): String =
    createSelectPathsByPathsBuilder(paths).orderBy(OrderBy.asc(CIS.path)).limit(1).query

  def buildSelectPathsThatExistQuery(paths: Iterable[String]): String =
    createSelectPathsByPathsBuilder(paths).query

  def buildSelectCiTypesThatExistQuery(paths: Iterable[String]): String =
    createSelectPathsByPathsBuilder(paths).select(CIS.ci_type).query
}
