package com.xebialabs.deployit.repository.sql.cache

import ai.digital.deploy.cache.config.CacheMode
import com.xebialabs.deployit.repository.sql.base.CiPKType
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component
import com.xebialabs.deployit.repository.sql.cache.queue.CacheEvictionProducer
import org.springframework.beans.factory.annotation.Autowired

import scala.jdk.CollectionConverters._

@Component
class CisCacheDataProcessor(@Autowired(required = false) cacheEvictionProducer: CacheEvictionProducer) extends Logging {

  def onUpdate(info: ConsolidatedCiUpdateInfo): Unit = {
    try {
      if (CiCacheDataServicesHolder.getCachesProperties.enabled) {
        val cis = info.updatedCisInfos
        logger.trace(s"Clearing Ci data for updated $cis.")
        val cisMap = cis.map(ci => ci.pk -> ci.id).toMap
        CiCacheDataServicesHolder
          .getCiCacheDataServiceFacade
          .removeAll(cisMap)
        cacheEvictionSendInfo(cisMap)
      }
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred onUpdate cache data process", exception)
    }
  }

  def onDelete(info: ConsolidatedCiDeletedInfo): Unit = {
    try {
      if (CiCacheDataServicesHolder.getCachesProperties.enabled) {
        val cis = info.getCis.asScala
        val cisMap = cis.map(ci => ci.get$internalId() -> ci.getId).toMap
        logger.trace(s"Clearing Ci data for deleted ($cisMap).")

        CiCacheDataServicesHolder
          .getCiCacheDataServiceFacade
          .removeAll(cisMap)
        cacheEvictionSendInfo(cisMap)
      }
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred onDelete cache data process", exception)
    }
  }

  def onRename(info: ConsolidatedCiRenamedInfo): Unit = {
    try {
      if (CiCacheDataServicesHolder.getCachesProperties.enabled) {
        val cis = info.pathModifiedCisInfo
        logger.trace(s"Clearing Ci data for renamed $cis.")
        val cisMap = cis.map(ci => ci.pk -> ci.oldId).toMap
        CiCacheDataServicesHolder
          .getCiCacheDataServiceFacade
          .removeAll(cisMap)
        cacheEvictionSendInfo(cisMap)
      }
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred onRename cache data process", exception)
    }
  }

  def onMove(info: ConsolidatedCiMovedInfo): Unit = {
    try {
      if (CiCacheDataServicesHolder.getCachesProperties.enabled) {
        val cis = info.pathModifiedCisInfo
        logger.trace(s"Clearing Ci data for moved $cis.")
        val cisMap = cis.map(ci => ci.pk -> ci.oldId).toMap
        CiCacheDataServicesHolder
          .getCiCacheDataServiceFacade
          .removeAll(cisMap)
        cacheEvictionSendInfo(cisMap)
      }
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred onMove cache data process", exception)
    }
  }

  def cacheEvictionSendInfo(info: Any): Unit = {
    if(!CacheMode.embeddedCacheMode) {
      try {
        cacheEvictionProducer.sendEvictionInfo(info)
      } catch {
        case exception: Exception =>
          logger.error("Exception occurred while sending info", exception)
      }
    }
  }

  def removeCaches(cis: Map[CiPKType, String]): Unit = {
    try {
      logger.trace(s"Clearing Ci data $cis.")
      CiCacheDataServicesHolder
        .getCiCacheDataServiceFacade
        .removeAll(cis)
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred while removing caches", exception)
    }
  }

  def removePropertiesCaches(cis: Set[CiPKType]): Unit = {
    try {
      logger.trace(s"Clearing Ci data $cis.")
      CiCacheDataServicesHolder
        .getCiCacheDataServiceFacade
        .removeAllProperties(cis)
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred while removing caches", exception)
    }
  }

  def onPropertiesUpdate(info: ConsolidatedCiPropertiesUpdatedInfo): Unit = {
    try {
      if (CiCacheDataServicesHolder.getCachesProperties.enabled) {
        val cis = info.updatedPropertiesInfo
        val cisSet = cis.map(ci => ci.pk).toSet
        logger.trace(s"Clearing Ci data for updated $cis.")
        CiCacheDataServicesHolder
          .getCiCacheDataServiceFacade
          .removeAllProperties(cisSet)
        cacheEvictionSendInfo(cisSet)
      }
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred onUpdate cache data process", exception)
    }
  }

  def onPropertiesDelete(info: ConsolidatedCiPropertiesDeletedInfo): Unit = {
    try {
      if (CiCacheDataServicesHolder.getCachesProperties.enabled) {
        val cis = info.deletedPropertiesCiInfo
        val cisSet = cis.map(ci => ci.pk).toSet
        logger.trace(s"Clearing Ci data for deleted (${cis.map(ci => ci.pk)}).")
        CiCacheDataServicesHolder
          .getCiCacheDataServiceFacade
          .removeAllProperties(cisSet)
        cacheEvictionSendInfo(cisSet)
      }
    } catch {
      case exception: Exception =>
        logger.error("Exception occurred onDelete cache data process", exception)
    }
  }
}
