package com.xebialabs.deployit.repository.sql.artifacts

import com.xebialabs.deployit.repository.sql.base.CiPKType

import java.io.InputStream
import java.util
import scala.jdk.CollectionConverters._
import scala.util.Try

class PrimaryWithFallbacksDataArtifactRepository(private val primary: ArtifactDataRepository,
                                                 private val others: List[ArtifactDataRepository])
  extends ArtifactDataRepository {

  private val fallbacks = others.filter(_.countArtifacts > 0)

  def hasFallbacks: Boolean = fallbacks.nonEmpty

  def targets: List[(ArtifactDataRepository, ArtifactDataRepository)] =
    fallbacks.map((_, primary))

  override def refFromChecksum(checksum: String): String = primary.refFromChecksum(checksum)

  override def store(pk: CiPKType, inputStream: InputStream): Unit = primary.store(pk, inputStream)

  override def insert(pk: CiPKType, reference: String): Unit = primary.insert(pk, reference)

  override def copy(fromId: String, pk: CiPKType): Unit = primary.copy(fromId, pk)

  override def retrieve(id: String): InputStream = primaryOrFallbacks( _.retrieve(id) )

  private def primaryOrFallbacks[T]( block: ArtifactDataRepository => T): T =
    fallbacks.foldLeft(Try(block.apply(primary))) { (t: Try[T], next: ArtifactDataRepository) =>
      if (t.isSuccess)
        t
      else
        Try(block.apply(next))
    }.getOrElse(throw new NoSuchElementException("No fallback artifacts"))

  override def delete(pk: CiPKType): Unit = (primary :: fallbacks).foreach(_.delete(pk))

  override def countArtifacts: Int = (primary :: fallbacks).foldLeft(0) { (i: Int, r: ArtifactDataRepository) => i + r.countArtifacts }
}

object ArtifactDataRepositoryFactory {
  def createArtifactDataRepository(configuredRepositoryName: String,
                                   allRepositories: util.Map[String, ArtifactDataRepository],
                                   allowFallbacks: Boolean): ArtifactDataRepository = {
    val configuredRepo = allRepositories.remove(configuredRepositoryName)
    val primaryWithFallback = new PrimaryWithFallbacksDataArtifactRepository(configuredRepo, allRepositories.values().asScala.toList)
    if (primaryWithFallback.hasFallbacks)
      if (allowFallbacks)
        primaryWithFallback
      else
        throw new IllegalStateException("Artifacts found in other storage. If the artifacts should be moved, please set " +
          "configuration option \"xl.repository.artifacts.allow-move\" to true. " +
          "Please refer help doc titled 'Move artifacts from the file system to a database'")
    else
      configuredRepo
  }

}
