package com.xebialabs.deployit.repository.sql.commands

import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter, toMap}
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.engine.spi.event.{CiMovedEvent, CiRenamedEvent}
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.ItemAlreadyExistsException
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.repository.sql.artifacts.ArtifactDataRepository
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.cache.{CiPathModifiedInfo, CisCacheDataProcessor, ConsolidatedCiMovedInfo, ConsolidatedCiRenamedInfo}
import com.xebialabs.deployit.repository.validation.CommandValidator
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.task.archive.TaskArchiveStore
import com.xebialabs.deployit.util.Tuple
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import java.util.{Map => JavaMap}
import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

class MoveCommand(override val jdbcTemplate: JdbcTemplate,
                  artifactDataRepository: ArtifactDataRepository,
                  commandValidator: CommandValidator,
                  taskArchiveStore: TaskArchiveStore,
                  val moveCis: Iterable[Tuple[String, String]],
                  cisCacheDataProcessor: CisCacheDataProcessor)(implicit val schemaInfo: SchemaInfo)
  extends UpdatePathCommand(jdbcTemplate, artifactDataRepository) with CiPathValidation with SecuredCi with DirectoryRef {

  override def execute(context: ChangeSetContext): Unit = moveCis.foreach { t => move(t.a, t.b, context) }

  override def validate(context: ChangeSetContext): Unit = moveCis.foreach { t => validateMove(t.a, t.b, context) }

  private def move(oldId: String, newId: String, context: ChangeSetContext): Unit = {
    val (oldPath: String, oldDirRef: String, map: JavaMap[String, Object]) = queryAndPreValidate(oldId, newId, context)
    val newPath = idToPath(newId)
    val (parentPk: CiPKType, modifiedCisInfo: ListBuffer[CiPathModifiedInfo]) = updateNameAndPath(oldPath, newPath)
    val internalId = asCiPKType(map.get(CIS.ID.name))
    val ci = readBaseCiFromMap(map)
    val newSecuredCiPk = updateSecurity(internalId, parentPk, newPath)
    val newSecuredDirRef = updateSecuredDirRefs(internalId, parentPk, newPath)
    val newDirectoryRef = resolveDirectoryRef(parentPk)
    updateDirectoryRef(newPath, oldDirRef, newDirectoryRef)
    modifiedCisInfo.addOne(CiPathModifiedInfo(
      ci.get$internalId(),
      pathToId(oldPath),
      oldPath,
      ci.getName,
      pathToId(newPath),
      newPath,
      newPath.substring(newPath.lastIndexOf('/') + 1)
    ))
    cisCacheDataProcessor.onMove(ConsolidatedCiMovedInfo(ci, newSecuredCiPk, newSecuredDirRef, newDirectoryRef, modifiedCisInfo.toList))
    EventBusHolder.publish(new CiMovedEvent(ci, newId))
  }

  private def queryAndPreValidate(oldId: String, newId: String, context: ChangeSetContext) = {
    Checks.checkArgument(!(oldId == newId), "Cannot move ci [%s] to same location", newId)
    val oldPath = idToPath(oldId)
    val map = jdbcTemplate.queryForObject(SELECT_CI_BY_PATH, MapRowMapper, oldPath)
    val ciType = Type.valueOf(map.get(CIS.ci_type.name).asInstanceOf[String])
    val oldDirRef = map.get(CIS.directory_ref.name).asInstanceOf[String]
    commandValidator.checkMoveAllowed(ciType)
    validateCisStoredInCorrectPath(Seq(newId -> ciType))(context)
    (oldPath, oldDirRef, map)
  }

  private def updateDirectoryRef(newPath: String, oldDirectoryRef: String, directoryRef: String) =
    jdbcTemplate.update(UPDATE_DIRECTORY_REF_BY_PATH, directoryRef, newPath, newPath + "/%", oldDirectoryRef)

  private def updateSecurity(pk: CiPKType, parentPk: CiPKType, newPath: String): CiPKType = {
    val newSecuredCiPk: CiPKType = getSecuredCi(pk) match {
      case Some(`pk`) =>
        // No action: moved CI has security assigned to it => preserve after move
        `pk`
      case Some(securedPk) =>
        val appType = "udm.Application"
        val envType = "udm.Environment"
        val cisTypeInCondition = cond.in(CIS.ci_type, Seq(appType, envType))
        val appEnvsSecureCiMap =
          getTypeSecuredCiMapForSecuredCiAndPath(newPath, securedPk, cisTypeInCondition)
            .groupBy { case (ciType, _) => ciType}
            .view
            .mapValues {_.map { case (_, id) => id }.toList}
            .toMap

        val otherSecureCiMap =
          getTypeSecuredCiMapForSecuredCiAndPath(newPath, securedPk, cond.not(cisTypeInCondition))
            .map { case (_, id) => id}
            .toList

        getSecuredCi(parentPk) match {
          case Some(newSecuredPk) =>
            jdbcTemplate.update(UPDATE_SECURED_ID, newSecuredPk, newPath, newPath + "/%", securedPk)
            taskArchiveStore.updateSecureCi(
              newSecuredPk, appEnvsSecureCiMap.get(Type.valueOf(envType)), appEnvsSecureCiMap.get(Type.valueOf(appType)), Option(otherSecureCiMap))
            newSecuredPk
          case None =>
            jdbcTemplate.update(UPDATE_SECURED_ID, null, newPath, newPath + "/%", securedPk)
            taskArchiveStore.updateSecureCi(
              null, appEnvsSecureCiMap.get(Type.valueOf(envType)), appEnvsSecureCiMap.get(Type.valueOf(appType)), Option(otherSecureCiMap))
            null
        }
      case None =>
        // No action: moved CI has no security.
        null
    }
    newSecuredCiPk
  }

  private def updateSecuredDirRefs(pk: CiPKType, parentPk: CiPKType, newPath: String): String = {
    val dirUuid = getDirectoryUuid(pk).orNull
    val newSecuredDirRef: String = getSecuredDirectoryReference(pk) match {
      case Some(securedDirRef) if dirUuid == securedDirRef =>
        // No action: moved CI has security assigned to it => preserve after move
        securedDirRef
      case Some(securedDir) =>
        val appType = "udm.Application"
        val envType = "udm.Environment"
        val cisTypeInCondition = cond.in(CIS.ci_type, Seq(appType, envType))
        val appEnvsCiMap =
          getTypeCiIdMapForSecuredDirectorRefAndPath(newPath, securedDir, cisTypeInCondition)
            .groupBy { case (ciType, _) => ciType}
            .view
            .mapValues {_.map { case (_, id) => id }.toList}
            .toMap

        val otherCiMap =
          getTypeCiIdMapForSecuredDirectorRefAndPath(newPath, securedDir, cond.not(cisTypeInCondition))
            .map { case (_, id) => id}
            .toList

        getSecuredDirectoryReference(parentPk) match {
          case Some(newSecuredDir) =>
            jdbcTemplate.update(
              UPDATE_SECURED_DIRECTORY_REF_BY_PATH,
              newSecuredDir,
              newPath,
              s"$newPath/%",
              securedDir
            )
            taskArchiveStore.updateSecuredDirectoryReference(
              newSecuredDir, appEnvsCiMap.get(Type.valueOf(envType)), appEnvsCiMap.get(Type.valueOf(appType)), Option(otherCiMap))
            newSecuredDir
          case None =>
            throw MissingSecuredDirectoryReferenceException(parentPk)
        }
      case None => null
    }
    newSecuredDirRef
  }

  private def validateMove(oldId: String, newId: String, context: ChangeSetContext): Unit = {
    queryAndPreValidate(oldId, newId, context)
    if (exists(newId))
      throw new ItemAlreadyExistsException("The destination id [%s] exists.", newId)
  }

  private def getTypeSecuredCiMapForSecuredCiAndPath(pathString: String, securedCi: Number, condition: cond): mutable.Buffer[(Type, Integer)] = {
    val builder = new SelectBuilder(CIS.tableName).select(CIS.ID).select(CIS.ci_type)
      .where(condition)
      .where(cond.or(Seq(cond.equals(CIS.path, pathString), cond.like(CIS.path, s"$pathString/%"))))
      .where(cond.equals(CIS.secured_ci, securedCi))
    jdbcTemplate.query(builder.query, Setter(builder.parameters), ciTypeIdMapper).asScala
  }

  private def getTypeCiIdMapForSecuredDirectorRefAndPath(pathString: String,
                                                         securedDirectoryRef: String,
                                                         condition: cond): mutable.Buffer[(Type, Integer)] = {
    val builder = new SelectBuilder(CIS.tableName).select(CIS.ID).select(CIS.ci_type)
      .where(condition)
      .where(cond.or(Seq(cond.equals(CIS.path, pathString), cond.like(CIS.path, s"$pathString/%"))))
      .where(cond.equals(CIS.secured_directory_ref, securedDirectoryRef))
    jdbcTemplate.query(builder.query, Setter(builder.parameters), ciTypeIdMapper).asScala
  }

  private val ciTypeIdMapper: RowMapper[(Type, Integer)] = (rs: ResultSet, _: Int) => {
    val map = toMap(rs)
    val pk = asCiPKType(map.get(CIS.ID.name))
    val ciType = Type.valueOf(map.get(CIS.ci_type.name).toString)
    (ciType, pk)
  }
}

class RenameCommand(override val jdbcTemplate: JdbcTemplate,
                    artifactRepository: ArtifactDataRepository,
                    val renameCis: Iterable[Tuple[String, String]],
                    cisCacheDataProcessor: CisCacheDataProcessor)
                   (implicit val schemaInfo: SchemaInfo)
  extends UpdatePathCommand(jdbcTemplate, artifactRepository) {

  override def execute(context: ChangeSetContext): Unit = renameCis.foreach(t => rename(t.a, t.b))

  override def validate(context: ChangeSetContext): Unit = renameCis.foreach(t => validateRename(t.a, t.b))

  private def rename(oldId: String, newName: String): Unit = {
    val (oldPath: String, map: JavaMap[String, Object]) = queryAndPreValidate(oldId, newName)
    val ci = readBaseCiFromMap(map)
    val newPath = parentPath(oldId).getOrElse(throw new NoSuchElementException(s"No parent path for ID $oldId")) + "/" + newName
    val modifiedCisInfo = updateNameAndPath(oldPath, newPath)._2
    modifiedCisInfo.addOne(CiPathModifiedInfo(
      ci.get$internalId(),
      pathToId(oldPath),
      oldPath,
      ci.getName,
      pathToId(newPath),
      newPath,
      newName
    ))
    cisCacheDataProcessor.onRename(ConsolidatedCiRenamedInfo(ci, modifiedCisInfo.toList))
    EventBusHolder.publish(new CiRenamedEvent(ci, newName))
  }

  private def validateRename(oldId: String, newName: String): Unit = {
    queryAndPreValidate(oldId, newName)
    val newPath = parentPath(oldId).getOrElse(throw new NoSuchElementException(s"No parent path for ID $oldId")) + "/" + newName
    if (exists(newPath))
      throw new ItemAlreadyExistsException("The destination id [%s] exists.", pathToId(newPath))
  }

  private def queryAndPreValidate(oldId: String, newName: String) = {
    Checks.checkArgument(!newName.contains('/'), "New name [%s] should not contain a /", newName)
    val oldPath = idToPath(oldId)
    val map = jdbcTemplate.queryForObject(SELECT_CI_BY_PATH, MapRowMapper, oldPath)
    val ciType = Type.valueOf(map.get(CIS.ci_type.name).asInstanceOf[String])
    if (ciType == Type.valueOf(classOf[Root]))
      throw new DeployitException("Cannot rename a core.Root configuration item")
    (oldPath, map)
  }
}

abstract class UpdatePathCommand(jdbcTemplate: JdbcTemplate,
                                 artifactDataRepository: ArtifactDataRepository)
  extends AbstractChangeSetCommand(jdbcTemplate)
    with UpdatePathCommandQueries with CiExists with CiLengthValidation with SCMTraceabilityDataQueries {

  protected def updateNameAndPath(oldPath: String, newPath: String): (CiPKType, mutable.ListBuffer[CiPathModifiedInfo]) = {
    val renamedCisInfo = mutable.ListBuffer[CiPathModifiedInfo]()
    val oldPathLike = s"$oldPath/%"
    validateCiNameAndIdLengths(pathToId(newPath))

    jdbcTemplate.query(SELECT_CI_CHILDREN_BY_PATH, MapRowMapper, oldPathLike).forEach {
      map => {
        val pk = asCiPKType(map.get(CIS.ID.name))
        val childrenOldPath = map.get(CIS.path.name).asInstanceOf[String]
        val childrenNewPath = newPath + "/" + childrenOldPath.substring(oldPath.length + 1)
        validateCiNameAndIdLengths(pathToId(childrenNewPath))
        renamedCisInfo.addOne(CiPathModifiedInfo(pk,
          pathToId(childrenOldPath),
          childrenOldPath,
          childrenOldPath.substring(childrenOldPath.lastIndexOf('/') + 1),
          pathToId(childrenNewPath),
          childrenNewPath,
          childrenNewPath.substring(childrenNewPath.lastIndexOf('/') + 1)
        ))
      }
    }
    if (exists(newPath))
      throw new ItemAlreadyExistsException("The destination id [%s] exists.", pathToId(newPath))
    validateCiNameAndIdLengths(pathToId(newPath))

    val parentPk = jdbcTemplate.queryForObject(SELECT_ID_BY_PATH, classOf[CiPKType], parentPath(newPath)
      .getOrElse(throw new NoSuchElementException(s"No parent path for ID $newPath")))
    jdbcTemplate.update(SCM_TRACEABILITY_DELETE_WHERE_CI_PATH_LIKE, oldPath, oldPathLike)

    jdbcTemplate.update(UPDATE_NAME, newPath.substring(newPath.lastIndexOf('/') + 1), newPath, parentPk, oldPath)
    jdbcTemplate.update(UPDATE_CHILDREN_PATH, newPath + "/", (oldPath.length + 2).asInstanceOf[Number], oldPathLike)
    (parentPk, renamedCisInfo)
  }
}

trait UpdatePathCommandQueries extends Queries {

  lazy val UPDATE_NAME: String = {
    import CIS._
    sqlb"update $tableName set $name = ?, $path = ?, $parent_id = ?, $scm_traceability_data_id = null where $path = ?"
  }

  lazy val UPDATE_CHILDREN_PATH: String = {
    import CIS._
    sqlb"update $tableName set $path = ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(path))}, $scm_traceability_data_id = null where $path like ?"
  }

  lazy val UPDATE_SECURED_ID: String = {
    import CIS._
    sqlb"update $tableName set $secured_ci = ? where ($path = ? or $path like ?) and $secured_ci = ?"
  }

  lazy val UPDATE_DIRECTORY_REF_BY_PATH: String = {
    import CIS._
    sqlb"update $tableName set $directory_ref = ? where ($path = ? or $path like ?) and $directory_ref = ?"
  }

  lazy val UPDATE_SECURED_DIRECTORY_REF_BY_PATH: String = {
    import CIS._
    sqlb"update $tableName set $secured_directory_ref = ? where ($path = ? or $path like ?) and $secured_directory_ref = ?"
  }

  lazy val SELECT_CIS_BY_PATH_AND_SECURED_ID: String = {
    import CIS._
    sqlb"select $path from $tableName where ($path = ? or $path like ?) and $secured_ci = ?"
  }

  lazy val SELECT_CI_CHILDREN_BY_PATH: String = {
    import CIS._
    sqlb"select $ID, $path from $tableName where $path like ? order by $path"
  }
}
