package com.xebialabs.deployit.repository.sql.commands

import java.sql.{PreparedStatement, Types}
import java.util.{ArrayList => JavaArrayList, HashSet => JavaHashSet, List => JavaList, Map => JavaMap, Set => JavaSet}
import com.xebialabs.deployit.artifact.resolution.InternalArtifactResolver
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs, BatchCommandWithSetter, BatchExecutorRepository}
import com.xebialabs.deployit.core.sql.spring.MapRowMapper
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo, setTimestamp}
import com.xebialabs.deployit.engine.spi.event.{CiCopiedEvent, CisCreatedEvent}
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.io.ArtifactFileUtils.hasRealOrResolvedFile
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, ExternalProperty}
import com.xebialabs.deployit.repository.ItemAlreadyExistsException
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.repository.sql.artifacts.{ArtifactDataRepository, ArtifactRepository}
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.repository.sql.properties.{InsertCiPropertiesRepository, PropertyInserter, PropertyManipulator}
import com.xebialabs.deployit.repository.sql.reader.CiReader
import com.xebialabs.deployit.repository.sql.specific.TypeSpecificInserter
import com.xebialabs.deployit.repository.sql.{CiHistoryRepository, CiRepository}
import com.xebialabs.deployit.repository.validation.CommandValidator
import com.xebialabs.deployit.task.archive.{BatchMessage, TaskArchiveStore}
import com.xebialabs.deployit.util.{PasswordEncrypter, Tuple}
import com.xebialabs.license.LicenseCiCounter
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry
import com.xebialabs.xlplatform.coc.service.{PersistenceParams, SCMTraceabilityService}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate}
import com.xebialabs.deployit.core.sql.spring.Setter.setString
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.sql.cache.{CiPropertiesUpdateInfo, CisCacheDataProcessor, ConsolidatedCiPropertiesUpdatedInfo, UpdatedPropertyInfo}

import scala.jdk.CollectionConverters._
import scala.collection.mutable

class CreateCommand(override val jdbcTemplate: JdbcTemplate,
                    artifactRepository: ArtifactRepository,
                    artifactDataRepository: ArtifactDataRepository,
                    override val ciRepository: CiRepository,
                    ciHistoryRepository: CiHistoryRepository,
                    batchExecutorRepository: BatchExecutorRepository,
                    passwordEncrypter: PasswordEncrypter, licenseCiCounter: LicenseCiCounter,
                    createTypeSpecificInserters: (Type, CiPKType) => List[TypeSpecificInserter],
                    scmTraceabilityService: SCMTraceabilityService,
                    val cis: Iterable[ConfigurationItem],
                    checksumAlgorithmProvider: ChecksumAlgorithmProvider,
                    taskArchiveStore: TaskArchiveStore,
                    cisCacheDataProcessor: CisCacheDataProcessor)
                   (implicit val schemaInfo: SchemaInfo)
  extends InsertCommand(jdbcTemplate, artifactRepository, artifactDataRepository, ciRepository, ciHistoryRepository, batchExecutorRepository,
    passwordEncrypter, licenseCiCounter, createTypeSpecificInserters, scmTraceabilityService, checksumAlgorithmProvider,
    taskArchiveStore) {

  override def preloadCache(context: ChangeSetContext): Unit =
    if (cis.nonEmpty) {
      preloadCacheWithParents(cis, context)
    }

  override def execute(context: ChangeSetContext): Unit =
    if (cis.nonEmpty) {
      insertCis(cis, context, cisCacheDataProcessor)
      EventBusHolder.publish(new CisCreatedEvent(context.scmTraceabilityData.orNull, cis.toList.asJava))
    }

  override def validate(context: ChangeSetContext): Unit = insertCisValidation(cis, context)
}

class CopyCommand(override val jdbcTemplate: JdbcTemplate,
                  artifactRepository: ArtifactRepository,
                  artifactDataRepository: ArtifactDataRepository,
                  override val ciRepository: CiRepository,
                  ciHistoryRepository: CiHistoryRepository,
                  batchExecutorRepository: BatchExecutorRepository,
                  passwordEncrypter: PasswordEncrypter,
                  licenseCiCounter: LicenseCiCounter,
                  ciReader: CiReader,
                  createTypeSpecificInserters: (Type, CiPKType) => List[TypeSpecificInserter],
                  commandValidator: CommandValidator,
                  scmTraceabilityService: SCMTraceabilityService,
                  val copyCis: Iterable[Tuple[String, String]],
                  checksumAlgorithmProvider: ChecksumAlgorithmProvider,
                  taskArchiveStore: TaskArchiveStore,
                  cisCacheDataProcessor: CisCacheDataProcessor)
                 (implicit val schemaInfo: SchemaInfo)
  extends InsertCommand(jdbcTemplate, artifactRepository, artifactDataRepository, ciRepository, ciHistoryRepository, batchExecutorRepository,
    passwordEncrypter, licenseCiCounter, createTypeSpecificInserters, scmTraceabilityService, checksumAlgorithmProvider,
    taskArchiveStore) {

  override def execute(context: ChangeSetContext): Unit = copyCis.foreach(t => copy(t.a, t.b, context))

  override def validate(context: ChangeSetContext): Unit = copyCis.foreach(t => validateCopy(t.a, t.b, context))

  private def copy(fromId: String, toId: String, context: ChangeSetContext): Unit = {
    Checks.checkArgument(!(fromId == toId), "Cannot copy ci [%s] to same location", toId)
    val path = idToPath(fromId)
    val oldCi = ciReader.readByPath(path, 0, useCache = false)
    validateCi(fromId, toId, oldCi)
    val tuples = jdbcTemplate.query(SELECT_CI_TREE_BY_PATH, MapRowMapper, path, s"$path/%").asScala.map { map =>
      val ci = ciReader.readUsingCache(asCiPKType(map.get(CIS.ID.name)), 1)(map)
      val copiedId = ci.getId
      renameCi(ci, fromId, toId)
      cleanUp(ci)
      copiedId -> ci
    }
    replaceReferences(tuples.toMap)
    insertCis(tuples.map { case (_, currentCi) => currentCi }, context, cisCacheDataProcessor, { (ciPk: CiPKType, copiedCi: ConfigurationItem) =>
      copiedCi match {
        case sa: SourceArtifact =>
          val fromCiId = fromId + sa.getId.substring(toId.length)
          artifactDataRepository.copy(fromCiId, ciPk)
          artifactRepository.updateFilename(ciPk, artifactRepository.getFilename(fromCiId))
        case _ =>
      }
    })
    EventBusHolder.publish(new CiCopiedEvent(oldCi, toId))
  }

  private def replaceReferences(references: Map[String, ConfigurationItem]): Unit = {
    references.values.foreach { newCi: ConfigurationItem =>
      newCi.getType.getDescriptor.getPropertyDescriptors.asScala.foreach { pd =>
        def replaceCiReference(): Unit = {
          pd.get(newCi) match {
            case ci: ConfigurationItem =>
              pd.set(newCi, references.getOrElse(ci.getId, ci))
            case _ => // nothing
          }
        }

        def replaceListReferences(): Unit = {
          val itr = pd.get(newCi).asInstanceOf[JavaList[ConfigurationItem@unchecked]].listIterator()
          while (itr.hasNext) {
            val oldId = itr.next().getId
            references.get(oldId).foreach(itr.set)
          }
        }

        def replaceSetReferences(): Unit = {
          val toAdd = new JavaHashSet[ConfigurationItem]()
          val set = pd.get(newCi).asInstanceOf[JavaSet[ConfigurationItem@unchecked]]
          val itr = set.iterator()
          while (itr.hasNext) {
            val oldId = itr.next().getId
            references.get(oldId).foreach(ci => {
              toAdd.add(ci)
              itr.remove()
            })
          }
          set.addAll(toAdd)
        }

        pd.getKind match {
          case PropertyKind.CI => replaceCiReference()
          case PropertyKind.LIST_OF_CI => replaceListReferences()
          case PropertyKind.SET_OF_CI => replaceSetReferences()
          case _ => // nothing
        }
      }
    }
  }

  private def validateCopy(fromId: String, toId: String, context: ChangeSetContext): Unit = {
    Checks.checkArgument(!(fromId == toId), "Cannot copy ci [%s] to same location", toId)
    val path = idToPath(fromId)
    val ci = ciReader.readByPath(path, 0)
    validateCi(fromId, toId, ci)
    insertCisValidation(jdbcTemplate.query(SELECT_CI_TREE_BY_PATH, MapRowMapper, path, s"$path/%").asScala.map { map =>
      val ci = ciReader.readUsingCache(asCiPKType(map.get(CIS.ID.name)), 1)(map)
      renameCi(ci, fromId, toId)
      ci
    }, context)
  }

  private def renameCi(ci: ConfigurationItem, fromId: String, toId: String): Unit = {
    val newId = toId + ci.getId.substring(fromId.length)
    ci.setId(newId)
  }

  private def cleanUp(ci: ConfigurationItem): Unit = {
    asBaseConfigurationItem(ci) { baseCi =>
      baseCi.get$ciAttributes().setScmTraceabilityDataId(null)
    }
  }

  private def validateCi(fromId: String, toId: String, ci: ConfigurationItem): Unit = {
    Checks.checkArgument(!(ci.getType == Type.valueOf(classOf[Root])), "Cannot copy root node [%s].", toId)
    commandValidator.checkCopyAllowed(
      ciReader.readByPath(parentPath(fromId).getOrElse(throw new NoSuchElementException(s"Cannot find CI with ID $fromId")), 0).getType,
      ciReader.readByPath(parentPath(toId).getOrElse(throw new NoSuchElementException(s"Cannot find CI with ID $toId")), 0).getType
    )
    if (exists(toId))
      throw new ItemAlreadyExistsException("The destination id [%s] exists.", toId)
  }
}

abstract class InsertCommand(override val jdbcTemplate: JdbcTemplate,
                             artifactRepository: ArtifactRepository,
                             artifactDataRepository: ArtifactDataRepository,
                             override val ciRepository: CiRepository,
                             ciHistoryRepository: CiHistoryRepository,
                             batchExecutorRepository: BatchExecutorRepository,
                             passwordEncrypter: PasswordEncrypter,
                             licenseCiCounter: LicenseCiCounter,
                             createTypeSpecificInserters: (Type, CiPKType) => List[TypeSpecificInserter],
                             scmTraceabilityService: SCMTraceabilityService,
                             checksumAlgorithmProvider: ChecksumAlgorithmProvider,
                             securityArchive: TaskArchiveStore)
  extends AbstractInsertOrUpdateCommand(jdbcTemplate, artifactRepository, artifactDataRepository, checksumAlgorithmProvider)
    with CiPathValidation
    with CiLengthValidation
    with InsertCiPropertiesRepository
    with InsertCommandQueries
    with SecuredCi
    with DirectoryRef
    with LookupValuesQueries
    with CiPreloadCache {

  def createTypeSpecificInserters(t: Type, pk: CiPKType): List[TypeSpecificInserter] = createTypeSpecificInserters.apply(t, pk)

  val createdCis = new JavaArrayList[ConfigurationItem]()

  private def noAction(pk: CiPKType, ci: ConfigurationItem): Unit = {}

  private def saveTraceabilityData(ci: ConfigurationItem, context: ChangeSetContext): Unit = {
    asBaseConfigurationItem(ci) { baseCi =>
      val id = scmTraceabilityService.persistOrDelete(PersistenceParams(
        Option(baseCi.get$ciAttributes().getScmTraceabilityDataId),
        context.scmTraceabilityData
      ))
      baseCi.get$ciAttributes().setScmTraceabilityDataId(id.orNull)
    }
  }

  def insertCis(cis: Iterable[ConfigurationItem], context: ChangeSetContext,
                cisCacheDataProcessor: CisCacheDataProcessor,
                afterCreate: (CiPKType, ConfigurationItem) => Unit = noAction): Unit = {

    verifyCis(cis, context)
    cis.foreach { ci =>
      saveTraceabilityData(ci, context)
    }

    createCis(cis, context)

    val cisExternalProps = createCiExternalProperty(cis, context)
    batchExecutorRepository.execute(cisExternalProps)

    val ciHistories = createCiHistory(cis, context)
    batchExecutorRepository.execute(ciHistories)

    cis.foreach { ci =>
      val pk = context.pkCache(ci)
      insertSourceArtifactData(pk, ci)
    }

    val (
      updateParents,
      setSecuredCis,
      updateSecureCis,
      setDirectoryRef,
      setSecuredDirectoryRefs) = updateParentReferences(cis, context)
    batchExecutorRepository.execute(updateParents)
    batchExecutorRepository.execute(setSecuredCis)
    batchExecutorRepository.execute(setDirectoryRef)
    batchExecutorRepository.execute(setSecuredDirectoryRefs)
    securityArchive.batchUpdateSecureCi(updateSecureCis)

    val (genericProperties, typeSpecificProperties) = insertCiProperties(cis, context)
    batchExecutorRepository.execute(genericProperties)
    batchExecutorRepository.executeWithSetter(typeSpecificProperties)

    cis.foreach { ci =>
      val pk = context.pkCache(ci)
      updateSourceArtifactFilename(pk, ci)
      afterCreate(pk, ci)
    }
    val ciPropertiesUpdateInfo = mutable.ListBuffer[CiPropertiesUpdateInfo]()
    createdCis.forEach(updateParentListProperties(_, context, ciPropertiesUpdateInfo))

    if(ciPropertiesUpdateInfo.nonEmpty) {
      cisCacheDataProcessor.onPropertiesUpdate(ConsolidatedCiPropertiesUpdatedInfo(ciPropertiesUpdateInfo))
    }
  }

  def insertCisValidation(cis: Iterable[ConfigurationItem], context: ChangeSetContext): Unit = {
    verifyCis(cis, context)
    createCisValidation(cis, context)
  }

  private def verifyCis(cis: Iterable[ConfigurationItem], context: ChangeSetContext): Unit = {
    verifyCisMeetCreationPreconditions(cis)
    licenseCiCounter.registerCisCreation(cis.toSeq, context.licenseTransaction)
  }

  private[commands] def updateParentListProperties(ci: ConfigurationItem, context: ChangeSetContext,
                                                   ciPropertiesUpdateInfo: mutable.ListBuffer[CiPropertiesUpdateInfo]): Unit = {
    parentPath(ci.getId).foreach { parentPath =>

      val ciPk = context.pkCache(ci)
      val parent = findCiForPath(parentPath, context)
      val updatedPropertiesInfo = mutable.ListBuffer[UpdatedPropertyInfo]()
      findContainmentListProperty(parent.ciType, ci.getType)
        .foreach { pd =>
          if (jdbcTemplate.queryForList(FIND_AS_CONTAINMENT_LIST_PROPERTY, parent.pk, pd.getName, ciPk).isEmpty) {
            updatedPropertiesInfo.addOne(UpdatedPropertyInfo(PropertyKind.LIST_OF_CI, ciPk, null))
            insertIndexedCiPropertyLastPosition(parent.pk, pd.getName, ciPk)
          }
        }
      if(updatedPropertiesInfo.nonEmpty) {
        ciPropertiesUpdateInfo.addOne(CiPropertiesUpdateInfo(parent.pk, updatedPropertiesInfo))
      }
    }
  }

  private def findContainmentListProperty(typeToScan: Type, typeOfList: Type): Option[PropertyDescriptor] =
    typeToScan.getDescriptor.getPropertyDescriptors.asScala.find { pd =>
      pd.getKind == PropertyKind.LIST_OF_CI && pd.isAsContainment && typeOfList.instanceOf(pd.getReferencedType)
    }

  private def verifyCisMeetCreationPreconditions(cis: Iterable[ConfigurationItem]): Unit = {
    cis.zipWithIndex.foreach { case (ci, idx) =>
      Checks.checkNotNull(ci, s"ci at index $idx is null.")
      Checks.checkNotNull(ci.getId, s"ci at index $idx has null id.")
      validateCiNameAndIdLengths(ci.getId)
      ci match {
        case sourceArtifact: SourceArtifact =>
          val hasCorrectFileUri = sourceArtifact.getFileUri != null && ArtifactResolverRegistry.validate(sourceArtifact)
          Checks.checkArgument(hasRealOrResolvedFile(sourceArtifact) || hasCorrectFileUri, "Artifact %s should have either file or correct fileUri set", ci.getId)
        case _ =>
      }
    }
  }

  private[commands] def createCis(cis: Iterable[ConfigurationItem], context: ChangeSetContext) = {
    createCisValidation(cis, context, checkExistence = false)
    cis.foreach { ci =>
      createdCis.add(ci)
      context.log("Creating %s of type %s", ci.getId, ci.getType)
    }

    try {
      jdbcTemplate.batchUpdate(INSERT, new BatchPreparedStatementSetter() {
        override def setValues(preparedStatement: PreparedStatement, i: Int): Unit = {
          val ci = createdCis.get(i)
          val token = generateToken

          val directoryId =
            if (ci.getType == Type.valueOf(classOf[Root]) || ci.getType == Type.valueOf(classOf[Directory]))
              Some(generateDirectoryId)
            else
              None

          val path = idToPath(ci.getId)

          preparedStatement.setString(1, ci.getType.toString)
          setString(preparedStatement, 2, ci.getName)
          preparedStatement.setString(3, token)
          directoryId match {
            case Some(id) => preparedStatement.setString(4, id)
            case None => preparedStatement.setNull(4, Types.VARCHAR)
          }
          setTimestamp(preparedStatement, 5, context.now)
          setString(preparedStatement, 6, context.userName)
          setTimestamp(preparedStatement, 7, context.now)
          setString(preparedStatement, 8, context.userName)
          setString(preparedStatement, 9, path)
          preparedStatement.setNull(10, Types.INTEGER)
          preparedStatement.setString(11, generateReferenceId)
          asBaseConfigurationItem(ci) { bci =>
            val dataId = bci.get$ciAttributes().getScmTraceabilityDataId
            if (dataId != null) {
              preparedStatement.setInt(10, bci.get$ciAttributes().getScmTraceabilityDataId)
            }

            bci.set$token(token)
          }
        }

        override def getBatchSize: Int = createdCis.size()
      })
    } catch {
      case _: DuplicateKeyException =>
        if (cis.size == 1)
          cis.foreach(ci => throw new ItemAlreadyExistsException("A Configuration Item with ID [%s] already exists.", ci.getId))
        else
          throw new ItemAlreadyExistsException("A Configuration Item with ID already exists.")
    }
  }

  private[commands] def createCiExternalProperty(cis: Iterable[ConfigurationItem], context: ChangeSetContext): Iterable[BatchCommandWithArgs] = {
    preloadCacheForCis(cis, context)

    cis.flatMap { ci =>

      val ciIdAndType = context.pathCache(idToPath(ci.getId))

      context.pkCache.put(ci, ciIdAndType.pk)

      val externalPropsInsert: Iterable[BatchCommandWithArgs] = asBaseConfigurationItem(ci) { bci =>
        bci.set$internalId(ciIdAndType.pk)
        insertExternalProperties(ciIdAndType.pk.intValue(), bci.get$externalProperties(), context)
      }
      Option(externalPropsInsert).getOrElse(Iterable.empty)
    }
  }

  private[commands] def createCiHistory(cis: Iterable[ConfigurationItem], context: ChangeSetContext): Iterable[BatchCommandWithArgs] = {
    cis.flatMap { ci =>
      val ciPk = context.pkCache(ci)
      ciHistoryRepository.batchInsert(ciPk, ci, context.now, context.userName)
    }
  }

  private def createCisValidation(cis: Iterable[ConfigurationItem], context: ChangeSetContext, checkExistence: Boolean = true): Unit = {
    if (checkExistence) {
      doWithFirstThatExists(cis.map(_.getId).toSeq) { path =>
        throw new ItemAlreadyExistsException("A Configuration Item with ID [%s] already exists.", pathToId(path))
      }
    }
    cis.foreach(ci => context.ciCache.getOrElseUpdate(ci.getId, ci))
    validateCisStoredInCorrectPath(cis.map(ci => ci.getId -> ci.getType))(context)
  }

  private[commands] def insertSourceArtifactData(pk: CiPKType, item: ConfigurationItem): Unit = {
    item match {
      case sa: SourceArtifact if hasRealOrResolvedFile(sa) && (sa.getFileUri == null || InternalArtifactResolver.canHandle(sa.getFileUri)) =>
        // Internal file: store file data.
        handleSourceArtifactFileData(pk, sa)
      case _ =>
      // do nothing
    }
  }

  private[commands] def updateSourceArtifactFilename(pk: CiPKType, item: ConfigurationItem): Unit = {
    item match {
      case sa: SourceArtifact if hasRealOrResolvedFile(sa) =>
        // Store file name
        handleSourceArtifactFilename(pk, sa)
        if (sa.getFileUri != null && !InternalArtifactResolver.canHandle(sa.getFileUri))
        // External file: set file to null
          sa.setFile(null)
      case _ =>
      // do nothing
    }
  }

  private def insertCiProperties(cis: Iterable[ConfigurationItem], context: ChangeSetContext):
  (Vector[BatchCommandWithArgs], Vector[BatchCommandWithSetter]) = {
    cis.foldLeft((Vector.empty[BatchCommandWithArgs], Vector.empty[BatchCommandWithSetter])) { (oldProps, ci) =>
      val pk = context.pkCache(ci)
      val props = new DefaultPropertyInserter(ci, context, this, passwordEncrypter)
        .insertProperties(ci.getType, pk)
      (oldProps._1 ++ props._1, oldProps._2 ++ props._2)
    }
  }

  private[commands] def updateParentReferences(cis: Iterable[ConfigurationItem], context: ChangeSetContext):
  (Iterable[BatchCommandWithArgs], Iterable[BatchCommandWithArgs], Iterable[BatchMessage], Iterable[BatchCommandWithArgs],
    Iterable[BatchCommandWithArgs]) = {
    val parentsWithCiPk = cis
      .flatMap { ci =>
        val ciParentPath = parentPath(ci.getId)
        ciParentPath.map(path => (path, context.pkCache(ci)))
      }
      .map(parentPathWithCiPk => (findCiForPath(parentPathWithCiPk._1, context), parentPathWithCiPk._2))

    val updateParents = parentsWithCiPk.map(parentWithPk => BatchCommand(UPDATE_PARENT, parentWithPk._1.pk, parentWithPk._2))

    val securedCiMap = mutable.Map(getSecuredCis(parentsWithCiPk.map(_._1.pk)): _*)


    val setSecuredCis = parentsWithCiPk.flatMap { parentsWithCi =>
      securedCiMap.get(parentsWithCi._1.pk)
        .map { secureCi =>
          securedCiMap.update(parentsWithCi._2, secureCi)
          batchSetSecuredCi(parentsWithCi._2, secureCi)
        }
    }

    val updateSecureCis: Iterable[BatchMessage] = parentsWithCiPk.flatMap { parentsWithCi =>
      val secureCi = securedCiMap.get(parentsWithCi._1.pk)
      secureCi.map(securityArchive.collectBatchUpdateSecureCi(parentsWithCi._2, _)).getOrElse(List.empty)
    }

    val directoryRefMap = mutable.Map(resolveDirectoryRefs(parentsWithCiPk.map(_._1.pk)): _*)
    val setDirectoryRefs = parentsWithCiPk.flatMap { parentsWithCi =>
      directoryRefMap.get(parentsWithCi._1.pk)
        .map { directoryUUID =>
          if (directoryRefMap.getOrElseUpdate(parentsWithCi._2, directoryUUID).isEmpty) {
            directoryRefMap.update(parentsWithCi._2, directoryUUID)
          }
          batchSetDirectoryRef(parentsWithCi._2, directoryUUID.getOrElse(throw MissingDirectoryReferenceException(parentsWithCi._2)))
        }
    }

    val securedDirectoryRefMap = mutable.Map(resolveSecuredDirectoryRef(parentsWithCiPk.map(_._1.pk)): _*)
    val setSecuredDictionaryRefs = parentsWithCiPk.flatMap { parentsWithCi =>
      securedDirectoryRefMap.get(parentsWithCi._1.pk).map {
        securedDirRef =>
          securedDirectoryRefMap.update(parentsWithCi._2, securedDirRef)
          batchSetSecuredDirectoryRef(parentsWithCi._2, securedDirRef.getOrElse(throw MissingSecuredDirectoryReferenceException(parentsWithCi._2)))
      }
    }
    (updateParents, setSecuredCis, updateSecureCis, setDirectoryRefs, setSecuredDictionaryRefs)
  }

  class DefaultPropertyInserter(val item: ConfigurationItem,
                                context: ChangeSetContext,
                                override val ciPropertiesRepository: InsertCiPropertiesRepository,
                                override val passwordEncrypter: PasswordEncrypter)
    extends PropertyInserter with PropertyManipulator {

    override protected val propertyFilter: PropertyDescriptor => Boolean = (pd: PropertyDescriptor) => !isTransientOrExternal(pd, item)

    override protected def getValue(pd: PropertyDescriptor): Option[Any] = getValue(item, pd, findPkfor(_, context))
  }

  private def insertExternalProperties(id: CiPKType,
                                       externalProperties: JavaMap[String, ExternalProperty],
                                       context: ChangeSetContext): Iterable[BatchCommandWithArgs] =
    externalProperties.asScala.map {
      case (name, lookup: LookupValueKey) =>
        BatchCommand(INSERT_LOOKUP_VALUES, id, name, lookup.getKey, findCiForId(lookup.getProviderId, context).pk)
      case (_, externalProperty: ExternalProperty) =>
        throw new IllegalArgumentException(s"Cannot handle unknown external property type: $externalProperty.")
    }
}

trait InsertCommandQueries extends Queries {

  lazy val FIND_AS_CONTAINMENT_LIST_PROPERTY: String = {
    import CI_PROPERTIES._
    sqlb"select * from $tableName where $ci_id = ? and $name = ? and $ci_ref_value = ?"
  }

  lazy val SELECT_CI_TREE_BY_PATH: String = {
    import CIS._
    sqlb"select * from $tableName where $path = ? or $path like ? order by $path"
  }

}
