package com.xebialabs.deployit.security.archive.sql

import java.sql.{PreparedStatement, ResultSet}

import com.xebialabs.deployit.core.sql.spring.{DeployJdbcTemplate, Setter}
import com.xebialabs.deployit.core.sql.spring.Setter.setString
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.security.Role
import com.xebialabs.deployit.security.archive.{ArchiveSecurity, NoSuchElementInSecurityArchiveException, PermissionDto}
import javax.sql.DataSource
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, RowMapper, SingleColumnRowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
@Transactional("reportingTransactionManager")
class SqlSecurityArchive(@Autowired @Qualifier("reportingDataSource") val dataSource: DataSource)
                        (@Autowired @Qualifier("reportingSchema") implicit val schemaInfo: SchemaInfo)
  extends ArchiveSecurity with ArchiveRoleQueries with ArchivePermissionsQueries {
  private val jdbcTemplate = new DeployJdbcTemplate(dataSource, false)

  override def getRole(id: String): Role = {
    try {
      val role: Role = jdbcTemplate.queryForObject(SELECT_ROLE_BY_ID, (rs: ResultSet, _: Int) => {
        new Role(rs.getString(1), rs.getString(2))
      }, id)
      val principals = jdbcTemplate.query(SELECT_ROLE_PRINCIPALS_BY_ROLE_ID, Setter(Seq(id), Setter.setStringForReporting), new SingleColumnRowMapper[String]())
      role.setPrincipals(principals)
      val childRoles = jdbcTemplate.query(SELECT_CHILD_ROLES_BY_ROLE_ID, Setter(Seq(id), Setter.setStringForReporting), new SingleColumnRowMapper[String]())
      role.setRoles(childRoles)
      role
    }
    catch {
      case _: EmptyResultDataAccessException => throw NoSuchElementInSecurityArchiveException()
    }
  }

  override def getPermissions(roleId: String): PermissionDto = {
    val tuples = jdbcTemplate.query(SELECT_ROLE_PERMISSIONS_BY_ROLE_ID, Setter(Seq(roleId), Setter.setStringForReporting), new RowMapper[(Int, String)] {

      override def mapRow(rs: ResultSet, rowNum: Int): (Int, String) = {
        (rs.getInt(2), rs.getString(1))
      }
    }).asScala.groupBy(_._1).map { case (k, v) => (k, v.map(_._2).toList) }

    if (tuples.nonEmpty) {
      val headKeySet = tuples.keySet.headOption.getOrElse(throw new IllegalStateException("Cannot get head from tuple keySet"))
      PermissionDto(roleId, headKeySet, tuples(headKeySet))
    } else {
      throw NoSuchElementInSecurityArchiveException()
    }

  }

  override def doCreate(id: String, name: String, childRoleIds: Seq[String], principals: Seq[String], onConfigurationItem: Number): Unit = {
    jdbcTemplate.update(INSERT_ROLE, id, name, onConfigurationItem)
    batchInsert(INSERT_CHILD_ROLE, id, childRoleIds)
    batchInsert(INSERT_ROLE_PRINCIPAL, id, principals)
  }

  override def doUpdate(original: Role, updated: Role, originalPrincipals: Set[String], updatedPrincipals: Set[String],
                        originalChildRoleIds: Set[String], updateChildRoleIds: Set[String]): Unit = {
    if (original.getName != updated.getName) {
      jdbcTemplate.update(UPDATE_ROLE, updated.getName, updated.getId)
    }
    updateChildEntities(INSERT_ROLE_PRINCIPAL, DELETE_ROLE_PRINCIPAL_BY_PRINCIPAL, original.getId, originalPrincipals, updatedPrincipals)
    updateChildEntities(INSERT_CHILD_ROLE, DELETE_CHILD_ROLE_BY_CHILD, original.getId, originalChildRoleIds, updateChildRoleIds)
  }

  override def doDelete(id: String): Unit = {
    jdbcTemplate.update(DELETE_ROLE_PERMISSIONS, id)
    jdbcTemplate.update(DELETE_CHILD_ROLE, id)
    jdbcTemplate.update(DELETE_ROLE_PRINCIPAL, id)
    jdbcTemplate.update(DELETE_ROLE, id)
  }

  override def doListDelete(ids: List[String]): Unit = {
    val setter = new BatchPreparedStatementSetter {
      override def getBatchSize: Int = ids.size

      override def setValues(ps: PreparedStatement, i: Int): Unit = ps.setString(1, ids(i))
    }
    jdbcTemplate.batchUpdate(DELETE_ROLE_PERMISSIONS, setter)
    jdbcTemplate.batchUpdate(DELETE_CHILD_ROLE, setter)
    jdbcTemplate.batchUpdate(DELETE_ROLE_PRINCIPAL, setter)
    jdbcTemplate.batchUpdate(DELETE_ROLE, setter)
  }

  override def removeCiPermissions(CiPk: Number): Unit = jdbcTemplate.update(DELETE_PERMISSIONS_FOR_CI, CiPk)

  override def updatePermissions(pk: Number,
                                 createdPermissions: mutable.Map[String, List[String]],
                                 deletedPermissions: mutable.Map[String, List[String]]): Unit = {
    createdPermissions.foreach(i => jdbcTemplate.batchUpdate(INSERT_PERMISSION,
      new PermissionStatementSetter(PermissionDto(i._1, pk, i._2))))
    deletedPermissions.foreach(i => jdbcTemplate.batchUpdate(DELETE_PERMISSION,
      new PermissionStatementSetter(PermissionDto(i._1, pk, i._2))))
  }

  private def batchInsert(query: String, roleId: String, children: Seq[String]): Unit = {
    jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {
      override def getBatchSize: Int = children.length

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        ps.setString(1, roleId)
        setString(ps, 2, children(i))
      }
    })
  }

  private def updateChildEntities(insertQuery: String, deleteQuery: String,
                                  roleId: String, originals: Set[String], updates: Set[String]): Unit = {
    val (toCreate, toDelete) = diffSets(originals, updates)
    toCreate.foreach(jdbcTemplate.update(insertQuery, roleId, _))
    toDelete.foreach(jdbcTemplate.update(deleteQuery, roleId, _))
  }

  private def diffSets(original: Set[String], updated: Set[String]) = (
    updated.diff(original),
    original.diff(updated)
  )

}

class PermissionStatementSetter(permission: PermissionDto)
  extends BatchPreparedStatementSetter {
  override def getBatchSize: Int = permission.permissions.size

  override def setValues(ps: PreparedStatement, i: Int): Unit = {
    ps.setString(1, permission.roleId)
    setString(ps, 2, permission.permissions(i))
    ps.setInt(3, permission.onConfigurationItem.intValue())
  }
}

trait ArchiveRoleQueries extends Queries {

  lazy val SELECT_ROLE_BY_ID: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoles._
    sqlb"select $ID, $NAME from $tableName where $ID = ?"
  }

  lazy val SELECT_CHILD_ROLES_BY_ROLE_ID: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoleRoles._
    sqlb"select $MEMBER_ROLE_ID from $tableName where $ROLE_ID = ?"
  }

  lazy val SELECT_ROLE_PRINCIPALS_BY_ROLE_ID: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRolePrincipals._
    sqlb"select $PRINCIPAL_NAME from $tableName where $ROLE_ID = ?"
  }

  lazy val INSERT_ROLE: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoles._
    sqlb"insert into $tableName ($ID, $NAME, $CI_ID) values (?, ?, ?)"
  }

  lazy val UPDATE_ROLE: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoles._
    sqlb"update $tableName set $NAME = ? where $ID = ?"
  }

  lazy val INSERT_CHILD_ROLE: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoleRoles._
    sqlb"insert into $tableName ($ROLE_ID, $MEMBER_ROLE_ID) values (?, ?)"
  }

  lazy val INSERT_ROLE_PRINCIPAL: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRolePrincipals._
    sqlb"insert into $tableName ($ROLE_ID, $PRINCIPAL_NAME) values (?, ?)"
  }

  lazy val DELETE_ROLE: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoles._
    sqlb"delete from $tableName where $ID = ?"
  }

  lazy val DELETE_CHILD_ROLE: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoleRoles._
    sqlb"delete from $tableName where $ROLE_ID = ?"
  }

  lazy val DELETE_CHILD_ROLE_BY_CHILD: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRoleRoles._
    sqlb"delete from $tableName where $ROLE_ID = ? and $MEMBER_ROLE_ID = ?"
  }

  lazy val DELETE_ROLE_PRINCIPAL: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRolePrincipals._
    sqlb"delete from $tableName where $ROLE_ID = ?"
  }

  lazy val DELETE_ROLE_PRINCIPAL_BY_PRINCIPAL: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.ArchiveRolePrincipals._
    sqlb"delete from $tableName where $ROLE_ID = ? and $PRINCIPAL_NAME = ?"
  }
}

trait ArchivePermissionsQueries extends Queries {

  import com.xebialabs.deployit.security.archive.sql.schema.ArchivePermissionsSchema._


  lazy val SELECT_ROLE_PERMISSIONS_BY_ROLE_ID: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchivePermissionsSchema._
    sqlb"select $PERMISSION_NAME, $CI_ID from $tableName where $ROLE_ID = ?"
  }

  lazy val INSERT_PERMISSION: String = {
    sqlb"insert into $tableName ($ROLE_ID, $PERMISSION_NAME, $CI_ID) values (?,?,?)"
  }

  lazy val DELETE_PERMISSION: String = {
    sqlb"delete from $tableName where $ROLE_ID = ? and $PERMISSION_NAME = ? and $CI_ID = ?"
  }

  lazy val DELETE_PERMISSIONS_FOR_CI: String = {
    sqlb"delete from $tableName where $CI_ID = ?"
  }

  lazy val DELETE_ROLE_PERMISSIONS: String = {
    import com.xebialabs.deployit.security.archive.sql.schema.ArchivePermissionsSchema._
    sqlb"delete from $tableName where $ROLE_ID = ?"
  }
}
