package com.xebialabs.deployit.security.sql

import com.xebialabs.deployit.core.sql.{SchemaInfo, SelectBuilder, Selectable, SubselectCondition, SqlCondition => cond}
import com.xebialabs.deployit.security.Permissions.authenticationToPrincipals
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.{Permission, PlatformPermissions}
import org.springframework.security.core.Authentication

import java.util
import java.util.Collections
import scala.jdk.CollectionConverters._

trait SqlPermissionFilter {

  def roleService: RoleService

  def checker: PermissionChecker

  def addReadPermission(builder: SelectBuilder, securedCiId: Selectable)(implicit schemaInfo: SchemaInfo): Unit = {
    val allPrincipals = authenticationToPrincipals(Permissions.getAuthentication)
    val allRoles = getRoles
    if (!isAdmin(allPrincipals, allRoles.asJava, Permissions.getAuthentication)) {
      val roleIds = allRoles.map(_.getId)
      val permissionNames = Permission.getReadPermissions.asScala.map(_.getPermissionName).toSeq
      builder.where(cond.subselect(securedCiId, SqlPermissionConditions.permittedCis(securedCiId, roleIds, permissionNames)))
    }
  }

  private[this] def getRoles: Seq[Role] = {
    val viewAsRoles: Option[util.List[Role]] =
      Option(SecurityServiceLocator.getViewAsData).flatMap { viewAsData =>
        // roles from viewAs
        Option(viewAsData.getRoles).orElse(
          // roles from viewAs.user
          Option(viewAsData.getUser).map(rolesFor)
        )
      }
    val roles: util.List[Role] = viewAsRoles.getOrElse(
      // roles from logged in user.
      rolesFor(Permissions.getAuthentication)
    )
    roles.asScala.toSeq
  }

  private[this] def rolesFor(authentication: Authentication): util.List[Role] = {
    roleService.getRolesFor(authentication, null, null, null)
  }

  private[this] def isAdmin(allPrincipals: util.Collection[String], allRoles: util.List[Role], auth: Authentication) = {
    allPrincipals.contains(PermissionEnforcer.ROLE_ADMIN) || checker.checkPermission(Collections.singletonList(PlatformPermissions.ADMIN), "", allRoles, auth)
  }
}

object SqlPermissionConditions {

  import PermissionsSchema._

  def permittedCis(securedCiId: Selectable, roleIds: Seq[String], permissionsNames: Seq[String])(implicit schemaInfo: SchemaInfo): SelectBuilder = {
    new SelectBuilder(tableName).select(CI_ID)
      .where(cond.in(PERMISSION_NAME, permissionsNames))
      .where(cond.or(Seq(
        cond.in(ROLE_ID, roleIds),
        roleByCIPrincipalAndParent(cond.equals(RolesSchema.Roles.CI_ID, securedCiId), roleIds, Permissions.getAuthentication)
      )))
  }

  def roleByCIPrincipalAndParent(idCondition: cond, roleIds: Seq[String], authentication: Authentication)(implicit schemaInfo: SchemaInfo): SubselectCondition = {
    import RolesSchema._
    def roleByPrincipal = {
      cond.subselect(Roles.ID, {
        import RolePrincipals._
        new SelectBuilder(tableName).select(ROLE_ID).where(cond.in(PRINCIPAL_NAME, Permissions.authenticationToPrincipals(authentication).asScala))
      })
    }

    def roleByParentRole = {
      cond.subselect(Roles.ID, {
        import RoleRoles._
        new SelectBuilder(tableName).select(ROLE_ID).where(cond.in(MEMBER_ROLE_ID, roleIds))
      })
    }

    cond.subselect(ROLE_ID,
      new SelectBuilder(Roles.tableName)
        .select(Roles.ID)
        .where(idCondition)
        .where(cond.or(Seq(
          roleByPrincipal,
          roleByParentRole
        )))
    )
  }

}
