package com.xebialabs.deployit.security.sql

import com.xebialabs.deployit.core.sql.spring.{Setter, XlSingleColumnRowMapper}
import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SchemaInfo, SelectBuilder, TableName, SqlCondition => cond}
import com.xebialabs.deployit.security.repository.XldUserDefaultCredentialsRepository
import com.xebialabs.deployit.security.sql.XldUserDefaultCredentialsSchema._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlXldUserDefaultCredentialsRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                            (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends XldUserDefaultCredentialsRepository with XldUserDefaultCredentialsQueries {
  override def findDefaultCredentialsId(username: String): Option[Long] = {
    val selectBuilder = new SelectBuilder(tableName).select(USER_CREDENTIALS_ID)
      .where(cond.equals(PROFILE_USERNAME, username.toLowerCase()))
    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new XlSingleColumnRowMapper(classOf[Long]))
      .asScala.headOption
  }

  override def updateDefaultCredentials(username: String, credentialsId: Long): Unit =
    jdbcTemplate.update(UPDATE, Setter(Seq(credentialsId, username.toLowerCase())))

  override def saveDefaultCredentials(username: String, credentialsId: Long): Unit =
    jdbcTemplate.update(INSERT, Setter(Seq(username.toLowerCase(), credentialsId)))

  override def deleteDefaultCredentials(username: String): Unit =
    jdbcTemplate.update(DELETE_BY_USERNAME, username.toLowerCase())

  override def deleteDefaultCredentials(credentialsId: Long): Unit =
    jdbcTemplate.update(DELETE_BY_CRED_ID, Setter(Seq(credentialsId)))
}

object XldUserDefaultCredentialsSchema {
  val tableName: TableName = TableName("XLD_USER_DEFAULT_CREDENTIALS")
  val PROFILE_USERNAME: ColumnName = ColumnName("PROFILE_USERNAME")
  val USER_CREDENTIALS_ID: ColumnName = ColumnName("USER_CREDENTIALS_ID")
}

trait XldUserDefaultCredentialsQueries extends Queries {
  val INSERT = sqlb"insert into $tableName ($PROFILE_USERNAME, $USER_CREDENTIALS_ID) values (?,?)"
  val UPDATE = sqlb"update $tableName set $USER_CREDENTIALS_ID = ? where $PROFILE_USERNAME = ?"
  val DELETE_BY_USERNAME = sqlb"delete from $tableName where $PROFILE_USERNAME = ?"
  val DELETE_BY_CRED_ID = sqlb"delete from $tableName where $USER_CREDENTIALS_ID = ?"
}
