package com.xebialabs.deployit.security.sql

import java.util

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.authentication.{AuthenticationFailureException, UserAlreadyExistsException}
import com.xebialabs.deployit.security.repository.XldUserProfileRepository
import com.xebialabs.deployit.security.sql.UserSchema._
import com.xebialabs.deployit.security.{RepoUser, SHA256PasswordEncoder, User, UserService}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.{JdbcTemplate, SingleColumnRowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlUserService(
                      @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                      @Autowired val userProfileRepository: XldUserProfileRepository
                    )(
                      @Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo
                    ) extends UserService
  with UserQueries {
  val passwordEncoder = new SHA256PasswordEncoder()

  private def withUsernameFilter(username: String, builder: SelectBuilder) =
    if (username != null) {
      builder.where(cond.like(SqlFunction.lower(USERNAME), s"%${username.toLowerCase}%"))
    }

  override def countUsers(username: String): Long = {
    val builder = new SelectBuilder(tableName).select(SqlFunction.countAll)
    withUsernameFilter(username, builder)
    jdbcTemplate
      .query(builder.query, Setter(builder.parameters), new SingleColumnRowMapper(classOf[Long]))
      .asScala
      .headOption
      .getOrElse(0L)
  }

  private def checkValidUsername(username: String): Unit = {
    if (username == null || username.isEmpty) throw new IllegalArgumentException("Username can neither be null nor empty.")
    readUser(username).foreach(_ => throw new UserAlreadyExistsException(username))
  }

  private def checkAdminUsername(username: String): Unit =
    if ("admin".equalsIgnoreCase(username))
      throw new IllegalArgumentException("Admin user cannot be deleted.")

  private def readUser(username: String) =
    jdbcTemplate
      .queryForList(SELECT, username.toLowerCase).asScala
      .map { map =>
        InternalUser(
          map.get(USERNAME.name).asInstanceOf[String],
          map.get(PASSWORD.name).asInstanceOf[String]
        )
      }.headOption

  override def read(username: String): User =
    readUser(username)
      .map(user => new RepoUser(user.username, user.username == "admin")).getOrElse(
      throw new NotFoundException(s"No such user: $username")
    )

  override def authenticate(username: String, password: String): Unit = {
    val user = readUser(username).getOrElse(throw new AuthenticationFailureException(s"Cannot authenticate $username"))
    if (!passwordEncoder.matches(password, user.passwordHash))
      throw new AuthenticationFailureException(s"Wrong credentials supplied for user $username")
  }

  override def modifyPassword(username: String, newPassword: String): Unit =
    jdbcTemplate.update(UPDATE, passwordEncoder.encode(newPassword), username)

  override def modifyPassword(username: String, newPassword: String, oldPassword: String): Unit =
    readUser(username).foreach { user =>
      if (!passwordEncoder.matches(oldPassword, user.passwordHash))
        throw new IllegalArgumentException("Failed to change password: Old password does not match.")
      modifyPassword(username, newPassword)
    }

  override def delete(username: String): Unit = {
    checkAdminUsername(username)
    jdbcTemplate.update(DELETE, username)
    userProfileRepository.delete(username)
  }

  override def create(username: String, password: String): Unit = {
    checkValidUsername(username)
    jdbcTemplate.update(INSERT, username, passwordEncoder.encode(password))
    userProfileRepository.createProfile(username)
  }

  override def listUsernames(): util.List[String] =
    jdbcTemplate.queryForList(SELECT_NAMES, classOf[String])

  override def listUsernames(username: String, paging: Paging, order: Ordering): util.List[String] = {
    val builder = new SelectBuilder(tableName).select(USERNAME)

    withUsernameFilter(username, builder)

    if (paging != null) {
      builder.showPage(paging.page, paging.resultsPerPage)
    }

    if (order != null) {
      builder.orderBy(
        if (order.isAscending) OrderBy.asc(SqlFunction.lower(USERNAME)) else OrderBy.desc(SqlFunction.lower(USERNAME))
      )
    }

    val names = jdbcTemplate
      .query(builder.query, Setter(builder.parameters), MapRowMapper).asScala
      .map(_.get(USERNAME.name).toString).toList
    asMutableJavaList(names)
  }
}

case class InternalUser(username: String, passwordHash: String)

object UserSchema {
  val tableName: TableName = TableName("XL_USERS")

  val USERNAME: ColumnName = ColumnName("USERNAME")
  val PASSWORD: ColumnName = ColumnName("PASSWORD")
}

trait UserQueries extends Queries {
  val SELECT = sqlb"select * from $tableName where lower($USERNAME) = ?"
  val SELECT_NAMES = sqlb"select $USERNAME from $tableName"
  val INSERT = sqlb"insert into $tableName ($USERNAME, $PASSWORD) values (?, ?)"
  val UPDATE = sqlb"update $tableName set $PASSWORD = ? where $USERNAME = ?"
  val DELETE = sqlb"delete from $tableName where $USERNAME = ?"
}
