package com.xebialabs.deployit.security.sql

import ai.digital.deploy.permissions.client.PermissionServiceClient
import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter, XlSingleColumnRowMapper, transactional}
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, SqlFunction => func, _}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.core.Securable
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.archive.ArchiveSecurity
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.sql.PermissionsSchema._
import com.xebialabs.deployit.security.sql.RolesSchema.Roles
import com.xebialabs.deployit.sql.base.schema.CIS
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate, ResultSetExtractor}
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional

import java.math.BigDecimal
import java.sql.PreparedStatement
import java.util.{UUID, HashSet => JHashSet}
import java.{lang, util}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait PermissionService extends PermissionEditor with PermissionLister with PermissionChecker {

  def removeCiPermissions(CiPk: Number): Unit
}

@Component("sqlPermissionService")
class SqlPermissionServiceImpl(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                               @Autowired val ciResolver: CiResolver,
                               @Autowired val securityArchive: ArchiveSecurity,
                               @Autowired @Qualifier("sqlRoleRepository")  sqlRoleRepository: RoleService,
                               @Autowired val permissionServiceClient: PermissionServiceClient,
                               @Autowired @Qualifier("mainTransactionManager") val mainTransactionManager: PlatformTransactionManager)
                              (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends PermissionService with PermissionsQueries with Logging {

  private final val SECURABLE = Type.valueOf(classOf[Securable])

  @Transactional("mainTransactionManager")
  override def removeCiPermissions(CiPk: Number): Unit = {
    jdbcTemplate.update(DELETE_CI, CiPk)
    securityArchive.removeCiPermissions(CiPk)

    ciResolver.getDirectoryUuid(CiPk) match {
      case Some(uuid) => wrapPermissionServiceCall(() => permissionServiceClient.removeForReference(UUID.fromString(uuid)))
      case _ =>
    }
  }

  private def withRolePattern(builder: SelectBuilder, rolePattern: Option[String]) = rolePattern match {
    case Some(pattern) if pattern.nonEmpty =>
      builder.where(cond.likeEscaped(func.lower(Roles.NAME), s"%$pattern%".toLowerCase()))
    case _ => builder
  }

  private def withPage(builder: SelectBuilder, paging: Paging): Unit = Option(paging) match {
    case Some(p) if p.resultsPerPage != -1 => builder.showPage(p.page, p.resultsPerPage)
    case _ =>
  }

  private def withOrder(builder: SelectBuilder, order: Ordering) = Option(order) match {
    case Some(ord) =>
      builder.orderBy(
        if (ord.isAscending) OrderBy.asc(func.lower(Roles.NAME)) else OrderBy.desc(func.lower(Roles.NAME))
      )
    case _ =>
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def readPermissions(onConfigurationItem: String, rolePattern: String, paging: Paging, order: Ordering, includeInherited: lang.Boolean): util.Map[Role, util.Set[Permission]] = {
    var rolePermissions: mutable.Buffer[util.Map[String, AnyRef]] = null

    val ciId = if (includeInherited == lang.Boolean.TRUE) this.getSecuredCiId(onConfigurationItem) else this.getCiResolverFromId(onConfigurationItem).pk

    if (rolePattern == null && (paging == null || paging.resultsPerPage == -1) && order == null) {
      rolePermissions = jdbcTemplate.queryForList(SELECT, ciId).asScala
    } else {
      val roleSelection = new SelectBuilder(Roles.tableName).select(Roles.ID)
      withRolePattern(roleSelection, Option(rolePattern))
      withOrder(roleSelection, order)
      withPage(roleSelection, paging)

      val roleIds: util.List[String] = jdbcTemplate.query(roleSelection.query, Setter(roleSelection.parameters), new XlSingleColumnRowMapper(classOf[String]))

      if (roleIds.isEmpty) {
        return new util.HashMap()
      }

      val fullSelection = new SelectFragmentBuilder(SELECT_FRAGMENT)
        .where(cond.in(Roles.ID.tableAlias("role"), roleIds.asScala))
        .where(cond.equals(PermissionsSchema.CI_ID.tableAlias("permission"), ciId))
      rolePermissions = jdbcTemplate.query(fullSelection.query, Setter(fullSelection.parameters), MapRowMapper).asScala
    }

    asMutableJavaMap(
      rolePermissions
        .groupBy(_.get("roleId").asInstanceOf[String])
        .map {
          case (roleId: String, rows: mutable.Seq[util.Map[String, AnyRef]]) =>
            val roleName = rows.head.get("roleName").asInstanceOf[String]
            val permissions = rows.map(_.get("permissionName").asInstanceOf[String]).flatMap(name => Option(Permission.find(name))).asJava
            new Role(roleId, roleName) -> new util.HashSet(permissions)
        }
    )
  }

  private def getPermissionsForCiAndRole(id: String, role: Role): mutable.Set[String] =
    readPermissions(id, role.getName, null, null).asScala
      .getOrElse(role, new JHashSet[Permission]).asScala.map(_.getPermissionName)

  private def hasPermission(id: String, role: Role, permissionName: String): Boolean =
    getPermissionsForCiAndRole(id, role).contains(permissionName)

  private def getRoles(): List[Role] = {
    val builder = new SelectBuilder(Roles.tableName)
    builder.select(Roles.ID).select(Roles.NAME)
    jdbcTemplate.query(builder.query, Setter(builder.parameters), roleMapper).asScala.toList
  }


  override def grant(role: Role, permission: Permission, id: String): Unit =
    if (!hasPermission(id, role, permission.getPermissionName)) {
      checkApplicablePermissions(id, List(permission).asJava)
      val roleName = role.getName
      logger.info(s"Granting $permission on $id for ${roleName}")
      val resolvedCi = getCiResolverFromId(id)
      transactional(mainTransactionManager) {
        resolvedCi match {
          case ci: ResolvedCi =>
            ciResolver.setSecuredCiId(ci)
            ciResolver.setSecuredDirectoryReference(ci)
          case _ =>
        }
        val roles = getRoles()
        val roleNameToIdMap = roles.map(role => role.getName -> role.getId).toMap
        val roleId = roleNameToIdMap.getOrElse(roleName, role.getId)
        val permissionList = List(permission.getPermissionName)
        val permissions = mutable.Map(roleId -> permissionList)
        jdbcTemplate.batchUpdate(INSERT, new ListBatchPreparedStatementSetter(roleId, resolvedCi.pk, permissionList))
        securityArchive.updatePermissions(resolvedCi.pk, permissions, mutable.Map.empty)
        val originalRoles =  mutable.Map(role.getName -> role)
        resolvedCi match {
          case ci: ResolvedCi =>
            updatePermissionService(
              Set(role),
              roleNameToIdMap,
              originalRoles,
              ci.directoryUuid.map(UUID.fromString)
                .orElse(ci.directoryRef.map(UUID.fromString).orElse(None)),
              permissions,
              mutable.Map.empty
            )
          case _ =>
            updatePermissionService(
              Set(role),
              roleNameToIdMap,
              originalRoles,
              None,
              permissions,
              mutable.Map.empty
            )
        }
      }
    }

  override def revoke(role: Role, permission: Permission, id: String): Unit = {
    val currentPermissions = getPermissionsForCiAndRole(id, role)
    if (!currentPermissions.contains(permission.getPermissionName)) {
      val roleName = role.getName
      logger.info(s"Revoking $permission on $id for ${role.getName}")
      val resolvedCi = getCiResolverFromId(id)
      val parentSecuredCiPk = resolvedCi match {
        case ci: ResolvedCi =>
          ci.securedCiPk match {
            case Some(securedCi) if securedCi == ci.pk =>
              ci.parentPk.flatMap(ciResolver.getSecuredCiId)
            case _ => None
          }
        case _ => None
      }
      val parentSecuredDir = resolvedCi match {
        case ci: ResolvedCi =>
          ci.securedDirectoryRef match {
            case Some(securedDirRef) if securedDirRef == ci.directoryUuid.orNull =>
              ci.parentPk.flatMap(ciResolver.getSecuredDirectoryReference)
            case _ => None
          }
        case _ => None
      }

      transactional(mainTransactionManager) {
        resolvedCi match {
          case ci: ResolvedCi if currentPermissions.size == 1 =>
            ciResolver.removeSecuredCi(ci, parentSecuredCiPk)
            ciResolver.removeSecuredDirectoryReference(ci, parentSecuredDir)
          case _ =>
        }
        val roles = getRoles()
        val roleNameToIdMap = roles.map(role => role.getName -> role.getId).toMap
        val roleId = roleNameToIdMap.getOrElse(roleName, role.getId)
        val permissionList = List(permission.getPermissionName)
        val permissions = mutable.Map(roleId -> permissionList)
        jdbcTemplate.batchUpdate(DELETE, new ListBatchPreparedStatementSetter(roleId, resolvedCi.pk, permissionList))
        securityArchive.updatePermissions(resolvedCi.pk, mutable.Map.empty, permissions)
        val originalRoles =  mutable.Map(role.getName -> role)
        resolvedCi match {
          case ci: ResolvedCi =>
            updatePermissionService(
              Set(role),
              roleNameToIdMap,
              originalRoles,
              ci.directoryUuid.map(UUID.fromString)
                .orElse(ci.directoryRef.map(UUID.fromString).orElse(None)),
              mutable.Map.empty,
              permissions
            )
          case _ =>
            updatePermissionService(
              Set(role),
              roleNameToIdMap,
              originalRoles,
              None,
              mutable.Map.empty,
              permissions
            )
        }
      }
    }
  }

  override def editPermissions(onConfigurationItem: String, permissions: util.Map[Role, util.Set[Permission]]): Unit = {
    logger.info(s"Edit ci $onConfigurationItem to assign next permissions: $permissions")
    val convertedPermissions = permissions.asScala
    val permissionValues: util.Collection[Permission] = convertedPermissions.flatMap(_._2.asScala).asJavaCollection
    logger.info(s"Permission values are $permissionValues")
    checkApplicablePermissions(onConfigurationItem, permissionValues)

    val permissionNamesByRoleName = convertedPermissions.map { case (role, ps) => role.getName -> ps.asScala.map(_.getPermissionName) }
    val roleMap = convertedPermissions.map { case (role, _) => role.getName -> role }
    val resolvedCi = getCiResolverFromId(onConfigurationItem)
    val parentSecuredCiPk = resolvedCi match {
      case ci: ResolvedCi if permissionValues.isEmpty  =>
        ci.securedCiPk match {
          case Some(securedCi) if securedCi == ci.pk =>
            ci.parentPk.flatMap(ciResolver.getSecuredCiId)
          case _ => None
        }
      case _ => None
    }

    val parentSecuredDir = resolvedCi match {
      case ci: ResolvedCi if permissionValues.isEmpty =>
        ci.securedDirectoryRef match {
          case Some(securedDirRef) if securedDirRef == ci.directoryUuid.orNull =>
            ci.parentPk.flatMap(ciResolver.getSecuredDirectoryReference)
          case _ => None
        }
      case _ => None
    }

    transactional(mainTransactionManager) {

      resolvedCi match {
        case ci: ResolvedCi =>
          if (permissionValues.isEmpty) {
            ciResolver.removeSecuredCi(ci, parentSecuredCiPk)
            ciResolver.removeSecuredDirectoryReference(ci, parentSecuredDir)
          } else {
            ciResolver.setSecuredCiId(ci)
            ciResolver.setSecuredDirectoryReference(ci)
          }
        case _ =>
      }

      val recordsByRoleName = jdbcTemplate.queryForList(SELECT, resolvedCi.pk).asScala.groupBy(_.get("roleName").asInstanceOf[String])
      val roles = getRoles()

      val roleNameToIdMap = roles.map(role => role.getName -> role.getId).toMap
      val toCreateRoleNames = permissionNamesByRoleName.keySet.toSet -- recordsByRoleName.keySet

      val createdPermissions: mutable.Map[String, List[String]] = mutable.Map[String, List[String]]()
      toCreateRoleNames.foreach { roleName =>
        val listOfPermissionNames = permissionNamesByRoleName(roleName).toList
        val roleId = roleNameToIdMap.getOrElse(roleName,
          roleMap.getOrElse(roleName, throw MissingPermissionServiceDataException(s"Cannot find $roleName to map to ID")).getId
        )
        jdbcTemplate.batchUpdate(INSERT, new ListBatchPreparedStatementSetter(roleId, resolvedCi.pk, listOfPermissionNames))
        createdPermissions += (roleId -> listOfPermissionNames)
      }

      val deletedPermissions: mutable.Map[String, List[String]] = mutable.Map()
      recordsByRoleName.map { case (roleName, rows) =>
        val originalPermissionNames: Set[String] = rows.map(_.get("permissionName").asInstanceOf[String]).toSet
        val updatedPermissionsNames: Set[String] = permissionNamesByRoleName.getOrElse(roleName, Set()).toSet

        val toCreate = (updatedPermissionsNames -- originalPermissionNames).toList
        val toDelete = (originalPermissionNames -- updatedPermissionsNames).toList

        val roleId = roleNameToIdMap.getOrElse(roleName,
          roleMap.getOrElse(roleName, throw MissingPermissionServiceDataException(s"Cannot find $roleName to map to ID")).getId
        )
        if (toCreate.nonEmpty) {
          jdbcTemplate.batchUpdate(INSERT, new ListBatchPreparedStatementSetter(roleId, resolvedCi.pk, toCreate))
          createdPermissions += (roleId -> toCreate)
        }
        if (toDelete.nonEmpty) {
          jdbcTemplate.batchUpdate(DELETE, new ListBatchPreparedStatementSetter(roleId, resolvedCi.pk, toDelete))
          deletedPermissions += (roleId -> toDelete)
        }
      }
      securityArchive.updatePermissions(resolvedCi.pk, createdPermissions, deletedPermissions)

      val recordsByRole = recordsByRoleName.flatMap {
        case (roleName, _) => roleMap.get(roleName)
      }
      val toCreateByRole = toCreateRoleNames.flatMap {
        roleName => roleMap.get(roleName)
      }
      resolvedCi match {
        case ci: ResolvedCi if !permissionValues.isEmpty =>
          updatePermissionService(
            toCreateByRole ++ recordsByRole,
            roleNameToIdMap,
            roleMap,
            ci.directoryUuid.map(UUID.fromString)
              .orElse(ci.directoryRef.map(UUID.fromString).orElse(None)),
            createdPermissions,
            deletedPermissions
          )
        case ci: ResolvedCi if permissionValues.isEmpty =>
          updatePermissionService(
            toCreateByRole ++ recordsByRole,
            roleNameToIdMap,
            roleMap,
            parentSecuredDir.map(UUID.fromString).orElse(None),
            createdPermissions,
            deletedPermissions
          )
          ci.securedDirectoryRef.map(UUID.fromString).map(removePermissionsForReference)
        case _ =>
          updatePermissionService(
            toCreateByRole ++ recordsByRole,
            roleNameToIdMap,
            roleMap,
            None,
            createdPermissions,
            deletedPermissions
          )
      }

    }
  }

  private def checkApplicablePermissions(onConfigurationItem: String, permissionValues: util.Collection[Permission]) = {
    val notApplicableTo = Permissions.isApplicableTo(permissionValues, onConfigurationItem)
    checkArgument(notApplicableTo.isEmpty, "The permissions %s are not applicable to [%s]", notApplicableTo, onConfigurationItem)
  }

  private def updatePermissionService(recordsByRole: Set[Role],
                                      roleNameToIdMap: Map[String, String],
                                      originalRoleMap: mutable.Map[String, Role],
                                      directoryRef: Option[UUID],
                                      createdPermissions: mutable.Map[String, List[String]],
                                      deletedPermissions: mutable.Map[String, List[String]]): Unit =
    recordsByRole.foreach {
      role =>
        val roleId = roleNameToIdMap.getOrElse(role.getName,
          originalRoleMap.getOrElse(role.getName, throw MissingPermissionServiceDataException(s"Cannot find ${role.getName} to map to ID")).getId
        )
        wrapPermissionServiceCall(() => permissionServiceClient.createOrUpdate(
          directoryRef,
          role.getName,
          createdPermissions.getOrElse(roleId, List()),
          deletedPermissions.getOrElse(roleId, List())
        ))
    }

  private def removePermissionsForReference(reference: UUID): Unit = {
    permissionServiceClient.removeForReference(reference)
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def listGlobalPermissions(roles: util.List[Role], paging: Paging): util.Map[String, util.List[String]] =
    withNonEmpty(roles, EMPTY_PERMISSIONS) {
      val builder = new SelectBuilder(tableName)
        .select(CI_ID)
        .select(PERMISSION_NAME)
        .where(cond.in(ROLE_ID, roles.asScala.map(_.getId)))
        .where(cond.equals(CI_ID, GLOBAL_ID))
      listPermissions(builder)
    }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def listPermissions(role: Role, paging: Paging): util.Map[String, util.List[String]] = {
    val builder = new SelectBuilder(tableName)
      .select(CI_ID)
      .select(PERMISSION_NAME)
      .where(cond.equals(ROLE_ID, role.getId))
    listPermissions(builder)
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def listPermissions(roles: util.List[Role], paging: Paging): util.Map[String, util.List[String]] =
    withNonEmpty(roles, EMPTY_PERMISSIONS) {
      val builder = new SelectBuilder(tableName)
        .select(CI_ID)
        .select(PERMISSION_NAME)
        .where(cond.in(ROLE_ID, roles.asScala.map(_.getId)))
      listPermissions(builder)
    }

  private def listPermissions(builder: SelectBuilder): util.Map[String, util.List[String]] = asMutableJavaMap(
    jdbcTemplate.query(builder.query, Setter(builder.parameters), MapRowMapper).asScala.groupBy(_.get(CI_ID.name).asInstanceOf[Number])
      .map { case (ciId, records) =>
        getIdFromPK(ciId) -> asMutableJavaList(records.map(_.get(PERMISSION_NAME.name).asInstanceOf[String]).toList)
      }
  )

  final class ListBatchPreparedStatementSetter(roleId: String, onConfigurationItem: Number, permissions: List[String]) extends BatchPreparedStatementSetter {
    override def getBatchSize: Int = permissions.size

    override def setValues(ps: PreparedStatement, i: Int): Unit = {
      ps.setString(1, roleId)
      Setter.setString(ps, 2, permissions(i))
      ps.setInt(3, onConfigurationItem.intValue())
    }
  }

  private def getCiResolverFromId(onConfigurationItem: String): CiResolvedType =
    onConfigurationItem match {
      case GLOBAL_SECURITY_ALIAS | "" | null => GlobalResolvedCi
      case _ =>
        val resolvedCi = ciResolver.getResolvedCiFromId(onConfigurationItem)
        checkArgument(
          resolvedCi.ciType.instanceOf(SECURABLE),
          "The type [%s] does not support security permissions, because it doesn't implement com.xebialabs.deployit.repository.core.Securable",
          resolvedCi.ciType)
        resolvedCi
    }

  private def getSecuredCiId(onConfigurationItem: String): Number =
    onConfigurationItem match {
      case GLOBAL_SECURITY_ALIAS | "" | null => GLOBAL_ID
      case _ => ciResolver.getSecuredCiId(onConfigurationItem).getOrElse(-1)
    }

  private def getIdFromPK(pk: Number): String =
    pk match {
      case GLOBAL_ID => GLOBAL_SECURITY_ALIAS
      case bd: BigDecimal if bd.intValue().equals(GLOBAL_ID) => GLOBAL_SECURITY_ALIAS
      case _ => ciResolver.getIdFromPk(pk)
    }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, allRoles: util.List[Role]): Boolean =
    checkPermission(permissions, onConfigurationItem, allRoles, Permissions.getAuthentication)

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, roles: util.List[Role], auth: Authentication): Boolean = {
    if (permissions.isEmpty || roles.isEmpty) return false

    val permissionsNames = permissions.asScala.map(_.getPermissionName).toList.asJava
    val roleIds = roles.asScala.map(_.getId).toList
    if (Strings.isEmpty(onConfigurationItem)) {
      checkPermission(permissionsNames, cond.in(ROLE_ID, roleIds), GLOBAL_ID)
    } else {
      ciResolver.getSecuredCiId(onConfigurationItem).exists { ciPK =>
        checkPermission(permissionsNames, cond.or(mutable.Seq(
          cond.in(ROLE_ID, roleIds),
          SqlPermissionConditions.roleByCIPrincipalAndParent(cond.equals(RolesSchema.Roles.CI_ID, ciPK), roleIds, auth)
        ).toSeq), ciPK)
      }
    }
  }


  override def checkPermission(permissions: util.List[Permission], onConfigurationItems: util.List[String], allRoles: util.List[Role], auth: Authentication): util.Map[String, lang.Boolean] =
    checkPermission(permissions, onConfigurationItems, allRoles)


  override def checkPermission(permissions: util.List[Permission], cis: util.List[String], allRoles: util.List[Role]): util.Map[String, lang.Boolean] =
    checkPermission(permissions.asScala.toList, cis.asScala.toList, allRoles.asScala.toList).view.mapValues(Boolean.box).toMap.asJava

  private def checkPermission(permissions: List[Permission], cis: List[String], allRoles: List[Role]): Map[String, Boolean] = {
    if(cis.isEmpty) {
      return Map.empty[String, Boolean]
    }
    if (permissions.isEmpty || allRoles.isEmpty) {
      return cis.foldLeft(mutable.Map.empty[String, Boolean]) { (acc, ci) =>
        acc.put(ci, false)
        acc
      }.toMap
    }
    val roleIds = allRoles.map(_.getId)
    val _cis = cis
      .toSet[String]
      .map { id =>
        if(id.charAt(0) != '/') s"/$id" else id
      }
      .grouped(schemaInfo.sqlDialect.inClauseLimit)
    val permissionsBuilder = new SelectBuilder(tableName)
      .select(new SqlLiteral(s"count(${PERMISSION_NAME.build(Option(permissionsAlias))})"))
      .where(cond.or(Seq(
        cond.in(PERMISSION_NAME, permissions.map(_.getPermissionName)),
        cond.equalsNull(PERMISSION_NAME)
      )))
      .as(permissionsAlias)
    val roleBuilder = new SelectBuilder(Roles.tableName)
      .where(cond.or(Seq(
        cond.in(Roles.ID.tableAlias(rolesAlias), roleIds),
        cond.equalsNull(Roles.ID.tableAlias(rolesAlias))
      )))
      .as(rolesAlias)
    _cis
      .flatMap { partialCis =>
        val ciBuilder = new SelectBuilder(CIS.tableName)
          .select(CIS.path.tableAlias(ciAlias))
          .where(cond.in(CIS.path.tableAlias(ciAlias), partialCis))
          .groupBy(CIS.path.tableAlias(ciAlias))
          .as(ciAlias)
        val builder = new JoinBuilder(permissionsBuilder)
          .join(roleBuilder, cond.equals(ROLE_ID.tableAlias(permissionsAlias), Roles.ID.tableAlias(rolesAlias)))
          .join(ciBuilder, cond.equals(CI_ID.tableAlias(permissionsAlias), CIS.secured_ci.tableAlias(ciAlias)), JoinType.Right)
        jdbcTemplate.query(builder.query, Setter(builder.parameters), permissionsPerCi)
      }
      .toMap
  }

  val permissionsPerCi: ResultSetExtractor[Map[String, Boolean]] = rs => {
    val map = mutable.Map.empty[String, Boolean]
    while (rs.next()) {
      val hasPermission = rs.getInt(1) > 0
      val ciId = rs.getString(2).substring(1)
      map.put(ciId, hasPermission)
    }
    map.toMap
  }

  val exists: ResultSetExtractor[Boolean] = rs => rs.next() && rs.getInt(1) > 0

  private val permissionsAlias = "xrp"
  private val rolesAlias = "xr"
  private val ciAlias = "xc"

  private def checkPermission(permissions: util.List[String], roleCondition: cond, ciPk: Number) = {
    val builder = new SelectBuilder(tableName)
      .select(func.countAll)
      .where(cond.equals(CI_ID, ciPk))
      .where(cond.in(PERMISSION_NAME, permissions.asScala))
      .where(roleCondition)
    jdbcTemplate.query(builder.query, Setter(builder.parameters), exists)
  }

  private def withNonEmpty[T](collection: util.Collection[_], empty: T)(block: => T) =
    if (collection.isEmpty) {
      empty
    } else {
      block
    }

  private val EMPTY_PERMISSIONS: util.Map[String, util.List[String]] = new util.HashMap[String, util.List[String]]()
}

object PermissionsSchema {
  val tableName: TableName = TableName("XL_ROLE_PERMISSIONS")

  val ROLE_ID: ColumnName = ColumnName("ROLE_ID")
  val PERMISSION_NAME: ColumnName = ColumnName("PERMISSION_NAME")
  val CI_ID: ColumnName = ColumnName("CI_ID")
}

trait PermissionsQueries extends Queries {

  lazy val SELECT_FRAGMENT: String = {
    import RolesSchema._
    sqlb"""role.${Roles.ID} roleId, role.${Roles.NAME} roleName, permission.${PermissionsSchema.PERMISSION_NAME} permissionName
          |from ${Roles.tableName} role inner join ${PermissionsSchema.tableName} permission on (role.${Roles.ID} = permission.${PermissionsSchema.ROLE_ID})"""
  }

  lazy val SELECT: String = {
    sqlb"""select $SELECT_FRAGMENT where permission.${PermissionsSchema.CI_ID} = ?"""
  }

  lazy val INSERT: String = {
    import PermissionsSchema._
    sqlb"insert into $tableName ($ROLE_ID, $PERMISSION_NAME, $CI_ID) values (?,?,?)"
  }

  lazy val DELETE: String = {
    import PermissionsSchema._
    sqlb"delete from $tableName where $ROLE_ID = ? and $PERMISSION_NAME = ? and $CI_ID = ?"
  }

  lazy val DELETE_CI: String = {
    import PermissionsSchema._
    sqlb"delete from $tableName where $CI_ID = ?"
  }
}
