package com.xebialabs.deployit.security.sql

import com.xebialabs.deployit.core.sql.spring.{ExtendedArgumentPreparedStatementSetter, Setter}
import com.xebialabs.deployit.core.sql.{ColumnName, JoinBuilder, JoinType, Queries, SchemaInfo, SelectBuilder, TableName, SqlCondition => cond}
import com.xebialabs.deployit.security.model.XldUserCredentials
import com.xebialabs.deployit.security.repository.{XldUserCredentialsRepository, XldUserDefaultCredentialsRepository}
import com.xebialabs.deployit.security.sql.SqlXldUserCredentialsRepository.{createRowMapper, userCredentialsTableAlias, userDefaultCredentialsTableAlias}
import com.xebialabs.deployit.security.sql.XldUserCredentialsSchema._
import com.xebialabs.deployit.util.PasswordEncrypter
import org.apache.commons.lang3.StringUtils
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.jdbc.support.GeneratedKeyHolder
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.{Connection, ResultSet}
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlXldUserCredentialsRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                      @Autowired val xldUserDefaultCredentialsService: XldUserDefaultCredentialsRepository,
                                      @Autowired val passwordEncrypter: PasswordEncrypter)
                                     (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends XldUserCredentialsRepository with XldUserCredentialsQueries {
  private val defaultCredBuilder = new SelectBuilder(XldUserDefaultCredentialsSchema.tableName)
    .as(userDefaultCredentialsTableAlias)

  private def getLeftJoinBuilder(selectBuilder: SelectBuilder) =
    new JoinBuilder(selectBuilder).join(
      defaultCredBuilder,
      cond.equals(
        ID,
        XldUserDefaultCredentialsSchema.USER_CREDENTIALS_ID
      ),
      JoinType.Left
    )

  override def findOne(id: Long): Option[XldUserCredentials] = {
    val selectBuilder = new SelectBuilder(tableName).where(cond.equals(ID, id)).as(userCredentialsTableAlias)
    val joinBuilder = getLeftJoinBuilder(selectBuilder)
    jdbcTemplate.query(joinBuilder.query, Setter(joinBuilder.parameters), mapRowToXldUserCredentials)
      .asScala.headOption
  }

  override def findDefaultCredentialsByProfileUsername(username: String): Option[XldUserCredentials] = {
    val selectBuilder = new SelectBuilder(tableName)
      .where(cond.equals(PROFILE_USERNAME, username.toLowerCase()))
      .as(userCredentialsTableAlias)
    val joinBuilder = new JoinBuilder(selectBuilder).join(
      defaultCredBuilder,
      cond.and(Seq(
        cond.equals(
          PROFILE_USERNAME.tableAlias(userCredentialsTableAlias),
          XldUserDefaultCredentialsSchema.PROFILE_USERNAME.tableAlias(userDefaultCredentialsTableAlias)
        ),
        cond.equals(ID, XldUserDefaultCredentialsSchema.USER_CREDENTIALS_ID)
      ))
    )
    jdbcTemplate.query(joinBuilder.query, Setter(selectBuilder.parameters), mapRowToXldUserCredentials)
      .asScala.headOption
  }

  override def findByProfileUsername(username: String): Set[XldUserCredentials] = {
    val selectBuilder = new SelectBuilder(tableName)
      .where(cond.equals(PROFILE_USERNAME, username.toLowerCase()))
      .as(userCredentialsTableAlias)
    val joinBuilder = getLeftJoinBuilder(selectBuilder)
    jdbcTemplate.query(joinBuilder.query, Setter(selectBuilder.parameters), mapRowToXldUserCredentials).asScala.toSet
  }


  override def saveCredentials(xldUserCredentials: XldUserCredentials): Long = withUpdateDefaults(xldUserCredentials) {
    validateCredentials(xldUserCredentials)
    val keyHolder = new GeneratedKeyHolder()
    jdbcTemplate.update((conn: Connection) => {
      val ps = conn.prepareStatement(INSERT, Array[String](ID.name))
      val psSetter = new ExtendedArgumentPreparedStatementSetter(getInsertParams(xldUserCredentials).toArray, true)
      psSetter.setValues(ps)
      ps
    }, keyHolder)
    keyHolder.getKey.longValue()
  }

  override def updateCredentials(xldUserCredentials: XldUserCredentials): Long = withUpdateDefaults(xldUserCredentials) {
    validateCredentials(xldUserCredentials)
    jdbcTemplate.update(UPDATE, Setter(getInsertParams(xldUserCredentials) ++ Seq[AnyRef](xldUserCredentials.id)))
    xldUserCredentials.id
  }

  override def deleteById(id: Long): Unit = {
    jdbcTemplate.update(DELETE_BY_ID, Setter(Seq(id)))
  }

  override def deleteByProfileUsername(username: String): Unit = {
    jdbcTemplate.update(DELETE_BY_USERNAME, Setter(Seq(username.toLowerCase())))
  }

  private def withUpdateDefaults(xldUserCredentials: XldUserCredentials)(id: Long): Long = {
    val profileUsername = xldUserCredentials.profileUsername.toLowerCase()
    xldUserDefaultCredentialsService.findDefaultCredentialsId(profileUsername) match {
      case Some(savedId) => if (xldUserCredentials.default) {
        xldUserDefaultCredentialsService.updateDefaultCredentials(profileUsername, id)
      } else if(!xldUserCredentials.default && id == savedId) {
        xldUserDefaultCredentialsService.deleteDefaultCredentials(profileUsername)
      }
      case None => if (xldUserCredentials.default) {
        xldUserDefaultCredentialsService.saveDefaultCredentials(profileUsername, id)
      }
    }
    id
  }

  private def getInsertParams(xldUserCredentials: XldUserCredentials): Seq[AnyRef] = Seq[AnyRef](
    xldUserCredentials.profileUsername.toLowerCase(),
    xldUserCredentials.label,
    xldUserCredentials.username,
    xldUserCredentials.email,
    xldUserCredentials.password.map(passwordEncrypter.ensureEncrypted).orNull,
    xldUserCredentials.passphrase.map(passwordEncrypter.ensureEncrypted).orNull,
    xldUserCredentials.privateKey.map(passwordEncrypter.ensureEncrypted).orNull
  )

  private def validateCredentials(xldUserCredentials: XldUserCredentials): Unit = {
    if (xldUserCredentials == null)
      throw new IllegalArgumentException("User credentials can not be null.")

    def getNilMessage(prop: String) = s"$prop can not be null or empty"

    val errors = new ListBuffer[String]

    if (StringUtils.isBlank(xldUserCredentials.profileUsername)) {
      errors += getNilMessage("Profile username")
    }
    if (StringUtils.isBlank(xldUserCredentials.label)) {
      errors += getNilMessage("Label")
    }
    if (StringUtils.isBlank(xldUserCredentials.username)) {
      errors += getNilMessage("Username")
    }
    if (StringUtils.isBlank(xldUserCredentials.email)) {
      errors += getNilMessage("Email")
    }
    if (StringUtils.isAllBlank(xldUserCredentials.password.getOrElse(""), xldUserCredentials.privateKey.getOrElse(""))) {
      errors += "Password or private key must be filled out."
    }
    if (errors.nonEmpty) {
      throw new IllegalArgumentException(errors.mkString("\n"))
    }
  }

  private def mapRowToXldUserCredentials: RowMapper[XldUserCredentials] =
    createRowMapper()
}

object SqlXldUserCredentialsRepository {
  def createRowMapper()(implicit schemaInfo: SchemaInfo): RowMapper[XldUserCredentials] = (rs: ResultSet, _) => XldUserCredentials(
    id = rs.getLong(ID.name),
    profileUsername = rs.getString(PROFILE_USERNAME.name),
    label = rs.getString(LABEL.name),
    username = rs.getString(USERNAME.name),
    email = rs.getString(EMAIL.name),
    default = Option(rs.getLong(XldUserDefaultCredentialsSchema.USER_CREDENTIALS_ID.name)) match {
      case Some(0L) => false
      case Some(_) => true
      case None => false
    },
    password = Option(rs.getString(PASSWORD.name)),
    passphrase = Option(rs.getString(PASSPHRASE.name)),
    privateKey = Option(rs.getString(PRIVATE_KEY.name))
  )

  val userCredentialsTableAlias = "xuc"
  val userDefaultCredentialsTableAlias = "xudc"
}

object XldUserCredentialsSchema {
  val tableName: TableName = TableName("XLD_USER_CREDENTIALS")
  val ID: ColumnName = ColumnName("ID")
  val PROFILE_USERNAME: ColumnName = ColumnName("PROFILE_USERNAME")
  val LABEL: ColumnName = ColumnName("LABEL")
  val USERNAME: ColumnName = ColumnName("CREDENTIALS_USERNAME")
  val EMAIL: ColumnName = ColumnName("EMAIL")
  val PASSWORD: ColumnName = ColumnName("PASSWORD")
  val PASSPHRASE: ColumnName = ColumnName("PASSPHRASE")
  val PRIVATE_KEY: ColumnName = ColumnName("PRIVATE_KEY")
}

trait XldUserCredentialsQueries extends Queries {
  val INSERT =
    sqlb"insert into $tableName ($PROFILE_USERNAME, $LABEL, $USERNAME, $EMAIL, $PASSWORD, $PASSPHRASE, $PRIVATE_KEY) values (?,?,?,?,?,?,?)"
  val UPDATE =
    sqlb"update $tableName set $PROFILE_USERNAME = ?, $LABEL = ?, $USERNAME = ?, $EMAIL = ?, $PASSWORD = ?, $PASSPHRASE = ?, $PRIVATE_KEY = ? where $ID = ?"
  val DELETE_BY_ID =
    sqlb"delete from $tableName where $ID = ?"
  val DELETE_BY_USERNAME =
    sqlb"delete from $tableName where $PROFILE_USERNAME = ?"
}
