package com.xebialabs.deployit.security.service.impl

import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.deployit.security.model.XldUserProfile
import com.xebialabs.deployit.security.repository.XldUserProfileRepository
import com.xebialabs.deployit.security.service.UserProfileService
import com.xebialabs.license.LicenseProperty
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import com.xebialabs.license.service.LicenseService

import java.util
import java.util.Date

@Service
class UserProfileServiceImpl(@Autowired xldUserProfileRepository: XldUserProfileRepository,
                             @Autowired roleService: RoleService,
                             @Autowired licenseService: LicenseService) extends UserProfileService{
  override def findOne(username: String, loadCredentials: Boolean): XldUserProfile = {
    xldUserProfileRepository.findOne(username, loadCredentials)
      .getOrElse(throw new NotFoundException(s"No user profile found for user: $username"))
  }

  override def findOne(username: String): XldUserProfile = findOne(username, loadCredentials = false)

  override def createOrUpdate(username: String, analyticsEnabled: Boolean): Unit = {
    xldUserProfileRepository.findOne(username) match {
      case Some(_) => xldUserProfileRepository.updateProfile(username, analyticsEnabled)
      case None => xldUserProfileRepository.createProfile(username, analyticsEnabled)
    }
  }

 override def createOrUpdateUserProfile(profile: XldUserProfile): Unit = {
    val existingProfile = xldUserProfileRepository.findOne(profile.username)
    val updatedProfile = existingProfile match {
      case Some(db_profile) if profile.loginAllowed && !profile.username.toLowerCase.equals("admin") =>
        revokeLoginAllowedAccordingTo(profile, db_profile)
      case Some(_) =>
        profile
      case None =>
        revokeLoginAllowedAccordingTo(profile)
    }
   if (existingProfile.isDefined) {
     xldUserProfileRepository.updateUserProfile(updatedProfile)
   } else {
     xldUserProfileRepository.createUserProfile(updatedProfile)
   }
  }

  private def revokeLoginAllowedAccordingTo(profile: XldUserProfile, db_profile: XldUserProfile = null): XldUserProfile = {
    if (!profile.loginAllowed)
      return profile
    val count = if (db_profile !=null && db_profile.loginAllowed) 1 else 0 // to exclude the current user record from the countUserWithLoginAllowed allowed list.
    if (!hasLicenseToCreateNewUser(count))
       return XldUserProfile(profile.username, profile.analyticsEnabled, profile.fullName, profile.email, false, profile.lastActive, profile.isInternal)
    profile
  }

  private def hasLicenseToCreateNewUser(count: Integer): Boolean = {
    val maxNumberOfUsers = licenseService.getLicense.getStringValue(LicenseProperty.MAX_NUMBER_OF_USERS)
    if (maxNumberOfUsers != null) {
      val countUserWithLoginAllowed = xldUserProfileRepository.countUserWithLoginAllowed() - count
      Integer.parseInt(maxNumberOfUsers) > countUserWithLoginAllowed
    } else {
      true
    }
  }

  override def countUserWithLoginAllowed(): Int = xldUserProfileRepository.countUserWithLoginAllowed()

  override def updateLastActive(username: String, date: Date): Int = xldUserProfileRepository.updateLastActive(username,date)

  override def countUsers(username: String, fullName: String, email: String): Long = {
    xldUserProfileRepository.countUsers(username, fullName, email)
  }

  override def listAllUserProfiles(username: String, fullName: String, email: String, paging: Paging, order: dto.Ordering): util.List[XldUserProfile] = {
    xldUserProfileRepository.listAllUserProfiles(username, fullName, email, paging, order)
  }
}