package com.xebialabs.deployit.task.archive.sql.reports

import com.xebialabs.deployit.core.sql.{SqlCondition => cond, SqlFunction => func, _}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.task.TaskType
import com.xebialabs.deployit.task.archive.DeploymentResult
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks._
import org.joda.time.DateTime

trait DeploymentTaskAnalyticsQuery {
  val begin: DateTime
  val end: DateTime

  implicit val schemaInfo: SchemaInfo

  def throughputByDeploymentResult(status: DeploymentResult): AbstractQueryBuilder

  def countByDeploymentResult(status: DeploymentResult): AbstractQueryBuilder

  def totalsByAppAndEnv(applicationName: String, environmentId: String): AbstractQueryBuilder

  def topNLongestRunningDeployments(topN: Int): AbstractQueryBuilder

  def topNDeploymentsByState(status: DeploymentResult, topN: Int): AbstractQueryBuilder

  def averageDurationDeploymentsOverTime(): AbstractQueryBuilder

  def totalSuccessfulDeploymentDuration(): AbstractQueryBuilder

  def bins(binStart: Option[Int], binEnd: Option[Int]): AbstractQueryBuilder
}

class SqlDeploymentTaskAnalyticsQuery(override val begin: DateTime, override val end: DateTime)(override implicit val schemaInfo: SchemaInfo)
  extends DeploymentTaskAnalyticsQuery {

  protected def createSelectBuilder: SelectBuilder = new SelectBuilder(tableName).where(cond.between(start_date, begin, end))

  override def throughputByDeploymentResult(status: DeploymentResult): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    groupByYearAndMonth(selectBuilder)
    selectCount(selectBuilder)
    matchAddConstraint(selectBuilder, status)
    selectBuilder
  }

  override def countByDeploymentResult(status: DeploymentResult): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    selectCount(selectBuilder)
    matchAddConstraint(selectBuilder, status)
    selectBuilder
  }

  override def totalsByAppAndEnv(applicationName: String, environmentId: String): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    selectCount(selectBuilder)
    deployTaskTypesOtherThanRollback(selectBuilder)
    selectBuilder.where(cond.equals(main_application, applicationName)).where(cond.equals(environment, environmentId))
    selectBuilder
  }

  override def topNLongestRunningDeployments(topN: Int): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    selectBuilder.select(func.average(duration)).orderBy(OrderBy.desc(func.average(duration)))
    selectBuilder.limit(topN)
    notRetried(selectBuilder)
    deployTaskTypesOtherThanRollback(selectBuilder)
    groupByAppAndEnv(selectBuilder)
    selectBuilder
  }

  override def topNDeploymentsByState(status: DeploymentResult, topN: Int): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    groupByAppAndEnv(selectBuilder)
    selectCount(selectBuilder)
    orderByCountDesc(selectBuilder)
    selectBuilder.limit(topN)

    status match {
      case DeploymentResult.SUCCESS =>
        addSuccessConstraint(selectBuilder)
      case DeploymentResult.FAILED =>
        sumFailureCounts(selectBuilder)
        addFailedConstraint(selectBuilder)
        orderByFailureCountSumDesc(selectBuilder)
      case _ => throw new IllegalArgumentException("topNDeploymentsByState called with invalid deploymentResult")
    }
    selectBuilder
  }

  override def averageDurationDeploymentsOverTime(): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    groupByYearAndMonth(selectBuilder)
    selectCount(selectBuilder)
    selectBuilder.select(func.average(duration))
    deployTaskTypesOtherThanRollback(selectBuilder)
    done(selectBuilder)
    selectBuilder
  }

  override def totalSuccessfulDeploymentDuration(): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    selectCount(selectBuilder)
    selectBuilder.select(func.average(duration)).select(schemaInfo.sqlDialect.standardDeviation(duration))
    addSuccessConstraint(selectBuilder)
    selectBuilder
  }


  // TODO what the did bins mean?!
  override def bins(binStart: Option[Int], binEnd: Option[Int]): AbstractQueryBuilder = {
    val selectBuilder = createSelectBuilder
    selectCount(selectBuilder)
    deployTaskTypesOtherThanRollback(selectBuilder)
    done(selectBuilder)

    (binStart, binEnd) match {
      case (binStart: Some[Int], None) => selectBuilder
        .where(cond.largerThen(duration, binStart.get))
      case (binStart: Some[Int], binEnd: Some[Int]) => selectBuilder
        .where(cond.largerThen(duration, binStart.get))
        .where(cond.smallerOrEqual(duration, binEnd.get))
      case (None, binEnd: Some[Int]) => selectBuilder
        .where(cond.smallerOrEqual(duration, binEnd.get))
      case (None, None) =>
    }
    selectBuilder
  }

  private[reports] def matchAddConstraint(selectBuilder: SelectBuilder, status: DeploymentResult): SelectBuilder = {
    status match {
      case DeploymentResult.ROLLBACK => addRollbackConstraint(selectBuilder)
      case DeploymentResult.SUCCESS => addSuccessConstraint(selectBuilder)
      case DeploymentResult.FAILED => addFailedConstraint(selectBuilder)
      case DeploymentResult.ABORTED => addAbortedConstraint(selectBuilder)
      case DeploymentResult.TOTAL => selectBuilder
    }
  }

  private[reports] def selectCount(selectBuilder: SelectBuilder) = selectBuilder.select(new SqlLiteral("count(*)"))

  private[reports] def orderByCountDesc(selectBuilder: SelectBuilder) = selectBuilder.orderBy(OrderBy.desc(func.countAll))

  private[reports] def groupByAppAndEnv(selectBuilder: SelectBuilder) = selectBuilder.select(main_application).select(environment)
    .groupBy(main_application).groupBy(environment)

  private[reports] def groupByYearAndMonth(selectBuilder: SelectBuilder): SelectBuilder =
    selectBuilder.select(schemaInfo.sqlDialect.year(start_date)).select(schemaInfo.sqlDialect.month(start_date))
    .groupBy(schemaInfo.sqlDialect.year(start_date)).groupBy(schemaInfo.sqlDialect.month(start_date))

  private[reports] def deployTaskTypesOtherThanRollback(selectBuilder: SelectBuilder) =
    selectBuilder.where(cond.in(task_type, List(TaskType.INITIAL, TaskType.UPGRADE, TaskType.UNDEPLOY)))

  private[reports] def doneWithRetry(selectBuilder: SelectBuilder) = selectBuilder.where(cond.equals(status, TaskExecutionState.DONE.name())).where(cond.largerThen(failure_count, 0))

  private[reports] def done(selectBuilder: SelectBuilder) = selectBuilder.where(cond.equals(status, TaskExecutionState.DONE.name())).where(cond.equals(failure_count, 0))

  private[reports] def notRetried(selectBuilder: SelectBuilder) = selectBuilder.where(cond.equals(failure_count, 0))

  private[reports] def sumFailureCounts(selectBuilder: SelectBuilder) = selectBuilder.select(func.sum(failure_count))

  private[reports] def orderByFailureCountSumDesc(selectBuilder: SelectBuilder) = selectBuilder.orderBy(OrderBy.desc(func.sum(failure_count)))

  private def addSuccessConstraint(selectBuilder: SelectBuilder) = {
    deployTaskTypesOtherThanRollback(selectBuilder)
    done(selectBuilder)
  }

  private def addFailedConstraint(selectBuilder: SelectBuilder) = {
    deployTaskTypesOtherThanRollback(selectBuilder)
    doneWithRetry(selectBuilder)
  }

  private def addAbortedConstraint(selectBuilder: SelectBuilder) = {
    deployTaskTypesOtherThanRollback(selectBuilder)
    selectBuilder.where(cond.equals(status, TaskExecutionState.CANCELLED.name()))
  }

  private def addRollbackConstraint(selectBuilder: SelectBuilder) = selectBuilder.where(cond.equals(task_type, TaskType.ROLLBACK.name()))
}
