package com.xebialabs.deployit.task.archive.sql

import com.google.common.base.Strings
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.security.archive.sql.schema.ArchivePermissionsSchema
import com.xebialabs.deployit.security.archive.sql.schema.ArchiveRolesSchema.{ArchiveRoleRoles, ArchiveRoles}
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters.DateRange._
import com.xebialabs.deployit.task.archive.{ControlTask, DeploymentTask}
import com.xebialabs.deployit.task.archive.sql.SqlArchivedTaskQuery._
import com.xebialabs.deployit.task.archive.sql.schema._
import com.xebialabs.deployit.task.{ArchivedTaskSearchParameters, TaskMetadata}
import org.joda.time.DateTime

import scala.jdk.CollectionConverters._
import scala.language.existentials
import scala.collection.mutable

object SqlArchivedTaskQuery {
  implicit class ExtendedSelectBuilder(builder: SelectBuilder) {
    def addStringLikeConstraint(field: Selectable, value: String): Unit = {
      if (!Strings.isNullOrEmpty(value))
        builder.where(cond.like(SqlFunction.lower(field), s"%${value.toLowerCase}%"))
    }
  }
}

abstract class SqlArchivedTaskQuery(searchParameters: ArchivedTaskSearchParameters)(implicit val schemaInfo: SchemaInfo)
  extends ArchivedTaskSearchParameters(searchParameters) {
  protected val aliasArchivedDeploymentTasks = "adt"
  protected val aliasControlTasks = "act"
  val selectBuilder: AbstractQueryBuilder =
    if (taskTypes.equals(ControlTask.CONTROL_TASK_TYPES))
      controlTaskQuery
    else if (!taskTypes.isEmpty && DeploymentTask.DEPLOYMENT_TASK_TYPES.containsAll(taskTypes))
      deploymentTaskQuery
    else
      new UnionBuilder(controlTaskQuery, deploymentTaskQuery, unknownTaskQuery)

  protected def columns(builder: SelectBuilder): Unit

  protected def deploymentTaskQuery: AbstractQueryBuilder = {
    val builder = new SelectBuilder(ArchivedDeploymentTasks.tableName)
    columns(builder)
    addTaskTypeConstraint(builder)
    addEnvironmentConstraint(builder)
    addDateRangeConstraint(builder)
    addUserConstraint(builder)
    addTaskIdConstraint(builder)
    addTaskStateConstraint(builder)
    addOnlySuccessful(builder)
    addApplicationConstraint(builder)
    addWorkerNameConstraint(builder)
    builder
  }

  protected def controlTaskQuery: AbstractQueryBuilder = {
    val builder = new SelectBuilder(ArchivedControlTasks.tableName)
    columns(builder)
    addDateRangeConstraint(builder)
    addTaskStateConstraint(builder)
    addTargetsConstraint(builder)
    addControlTaskNameConstraint(builder)
    addUserConstraint(builder)
    addWorkerNameConstraint(builder)
    addControlTaskIdConstraint(builder)
    builder
  }

  private[this] def unknownTaskQuery: SelectBuilder = {
    val builder = new SelectBuilder(ArchivedUnknownTasks.tableName)
    columns(builder)
    addDateRangeConstraint(builder)
    builder
  }

  protected def addTaskTypeConstraint(selectBuilder: SelectBuilder): Unit = {
    val tts = taskTypes.asScala
    if (tts.nonEmpty && !taskTypes.equals(DeploymentTask.DEPLOYMENT_TASK_TYPES)) {
      selectBuilder.where(cond.or(tts.map { taskType => cond.equals(ArchivedDeploymentTasks.task_type, taskType.toString) }.toSeq))
    }
  }

  protected def addEnvironmentConstraint(selectBuilder: SelectBuilder): Unit = {
    val envs = environments.asScala
    if (envs.nonEmpty) selectBuilder.where(cond.in(ArchivedDeploymentTasks.environment, envs))
  }

  protected def addUserConstraint(selectBuilder: SelectBuilder): Unit = {
    val usrs = users.asScala
    if (usrs.nonEmpty)
      selectBuilder.where(cond.in(ArchivedDeploymentTasks.owner, usrs))
  }

  protected def addTaskIdConstraint(selectBuilder: SelectBuilder): Unit =
    selectBuilder.addStringLikeConstraint(ArchivedDeploymentTasks.task_id, taskUuid)

  protected def addTaskStateConstraint(selectBuilder: SelectBuilder): Unit = {
    val tes = taskExecutionStates.asScala
    if (tes.nonEmpty)
      selectBuilder.where(cond.or(tes.map { taskState => cond.equals(ArchivedDeploymentTasks.status, taskState.toString) }.toSeq))
  }

  protected def addTargetsConstraint(selectBuilder: SelectBuilder): Unit = {
    Option(targets).map(_.asScala).filter(_.nonEmpty).foreach { targetCis =>
      selectBuilder.where(cond.in(ArchivedControlTasks.target_ci, targetCis))
    }
  }

  protected def addControlTaskIdConstraint(selectBuilder: SelectBuilder): Unit =
    selectBuilder.addStringLikeConstraint(ArchivedControlTasks.task_id, taskUuid)

  protected def addControlTaskNameConstraint(selectBuilder: SelectBuilder): Unit =
    selectBuilder.addStringLikeConstraint(ArchivedControlTasks.control_task_name, taskName)

  protected def addWorkerNameConstraint(selectBuilder: SelectBuilder): Unit =
    selectBuilder.addStringLikeConstraint(ArchivedControlTasks.worker_name, workerName)

  protected def addApplicationConstraint(selectBuilder: SelectBuilder): Unit = {
    val apps = applications.asScala
    if (apps.nonEmpty) {
      val subBuilder =
        new SelectBuilder(ArchivedDeploymentTasks.Applications.tableName)
          .select(ArchivedDeploymentTasks.Applications.task_id)
          .where(getApplicationsByNameCondition(apps))
      selectBuilder.where(cond.subselect(ArchivedDeploymentTasks.Applications.task_id, subBuilder))
    }
  }

  private[sql] def getApplicationsByNameCondition(apps: mutable.Set[ArchivedTaskSearchParameters.Application]): cond =
    cond.in(ArchivedDeploymentTasks.Applications.application, apps.map(_.name))

  protected def addDateRangeConstraint(selectBuilder: SelectBuilder): Unit = {
    dateRangeSearch match {
      case AFTER =>
        selectBuilder.where(cond.after(ArchivedTasksShared.start_date, startDate))
      case BEFORE =>
        selectBuilder.where(cond.before(ArchivedTasksShared.start_date, endDate))
      case BETWEEN =>
        selectBuilder.where(cond.and(Seq(
          cond.after(ArchivedTasksShared.start_date, startDate),
          cond.before(ArchivedTasksShared.start_date, endDate)
        )))
      case NONE =>
    }
  }

  protected def addOnlySuccessful(selectBuilder: SelectBuilder): Unit = {
    if (onlySuccessful) {
      selectBuilder.where(
        cond.or(Seq(cond.equals(ArchivedDeploymentTasks.rolled_back, value = false),
          cond.equalsNull(ArchivedDeploymentTasks.rolled_back)
        ))
      )
    }
  }

  protected def startOfDay(date: DateTime): DateTime = date.withTime(0, 0, 0, 0)

  protected def startOfNextDay(date: DateTime): DateTime = startOfDay(date.plusDays(1))

  private[sql] def addPaging(builder: AbstractQueryBuilder): Unit =
    builder.showPage(page, resultsPerPage)

  private[sql] def addOrderBy(selectBuilder: AbstractQueryBuilder): Unit =
    orderBy.forEach { t =>
      val f: ColumnName = SqlArchivedTaskSelectQuery.orderFieldMapping(t.a)
      selectBuilder.orderBy(if (t.b) OrderBy.asc(f) else OrderBy.desc(f))
    }
}

object SqlArchivedTaskSelectQuery {

  import com.xebialabs.deployit.task.archive.sql.schema._

  val orderFieldMapping: Map[String, ColumnName] = Map(
    "id" -> ArchivedTasksShared.task_id,
    "failures" -> ArchivedDeploymentTasks.failure_count,
    "description" -> ArchivedControlTasks.description,
    "owner" -> ArchivedDeploymentTasks.owner,
    "state" -> ArchivedDeploymentTasks.status,
    "startDate" -> ArchivedTasksShared.start_date,
    "completionDate" -> ArchivedDeploymentTasks.end_date,
    TaskMetadata.APPLICATION -> ArchivedDeploymentTasks.main_application,
    TaskMetadata.ENVIRONMENT_ID -> ArchivedDeploymentTasks.environment,
    TaskMetadata.TASK_TYPE -> ArchivedDeploymentTasks.task_type,
    TaskMetadata.CONTROL_TASK_TARGET_CI -> ArchivedControlTasks.target_ci,
    TaskMetadata.TASK_NAME -> ArchivedControlTasks.control_task_name,
    "worker_name" -> ArchivedDeploymentTasks.worker_name
  )
}

class SqlArchivedTaskSelectQuery(searchParameters: ArchivedTaskSearchParameters, columns: Seq[Selectable] = Seq())
                                (override implicit val schemaInfo: SchemaInfo)
  extends SqlArchivedTaskQuery(searchParameters) {

  override protected def columns(builder: SelectBuilder): Unit = columns.foreach(c => builder.select(c))

  addOrderBy(selectBuilder)
  addPaging(selectBuilder)

}

abstract class AbstractSqlArchivedTaskSelectWithPermissionsQuery(searchParameters: ArchivedTaskSearchParameters)
                                                                (override implicit val schemaInfo: SchemaInfo)
  extends SqlArchivedTaskQuery(searchParameters) {

  override val selectBuilder: AbstractQueryBuilder =
    if (taskTypes.equals(ControlTask.CONTROL_TASK_TYPES)) {
      controlTaskQuery
    }
    else if (!taskTypes.isEmpty && DeploymentTask.DEPLOYMENT_TASK_TYPES.containsAll(taskTypes)) {
      deploymentTaskQuery
    }
    else {
      throw new IllegalStateException("Non-admin user cannot access full task search")
    }

  override protected def columns(builder: SelectBuilder): Unit =
    if (taskTypes.equals(ControlTask.CONTROL_TASK_TYPES)) {
      builder.select(new SqlLiteral(s"$aliasControlTasks.*"))
    }
    else {
      archivedDeploymentTaskColumns.foreach(column => builder.select(column))
      // all this because CLOB cannot be in distinct and the below column is abused as CLOB
      archivedDeploymentTaskLobColumns.foreach{
        case (column, alias) => builder.select(new SqlLiteral(schemaInfo.sqlDialect.castToString(column).build()), alias)
      }
    }

  override protected def deploymentTaskQuery: AbstractQueryBuilder = {
    val taskSelectBuilder = new SelectBuilder(ArchivedDeploymentTasks.tableName)
    columns(taskSelectBuilder)
    addTaskTypeConstraint(taskSelectBuilder)
    addEnvironmentConstraint(taskSelectBuilder)
    addDateRangeConstraint(taskSelectBuilder)
    addUserConstraint(taskSelectBuilder)
    addTaskIdConstraint(taskSelectBuilder)
    addTaskStateConstraint(taskSelectBuilder)
    addOnlySuccessful(taskSelectBuilder)
    addWorkerNameConstraint(taskSelectBuilder)
    addOrderBy(taskSelectBuilder)

    val aliasArchivedApplications = "adtapp"
    val aliasArchivedDeploymentTaskPermissions = "adtperm"
    val aliasArchivedApplicationPermissions = "adtappperm"
    val aliasArchivedDeploymentTaskRoles = "adtrole"
    val aliasArchivedApplicationRoles = "adtapprole"

    val applicationSelectBulder = new SelectBuilder(ArchivedDeploymentTasks.Applications.tableName).as(aliasArchivedApplications)
    addApplicationConstraint(applicationSelectBulder)

    val builder = new JoinBuilder(taskSelectBuilder.as(aliasArchivedDeploymentTasks).distinct()).join(
      applicationSelectBulder,
      cond.equals(
        ArchivedDeploymentTasks.Applications.task_id.tableAlias(aliasArchivedApplications),
        ArchivedDeploymentTasks.task_id.tableAlias(aliasArchivedDeploymentTasks))
    )
      .join(
        permissionsQuery(aliasArchivedDeploymentTaskPermissions, aliasArchivedDeploymentTaskRoles),
        permissionCondition(aliasArchivedDeploymentTaskPermissions,
          ArchivedDeploymentTasks.environment_secured_ci.tableAlias(aliasArchivedDeploymentTasks))
      )
      .join(
        permissionsQuery(aliasArchivedApplicationPermissions, aliasArchivedApplicationRoles),
        permissionCondition(aliasArchivedApplicationPermissions,
          ArchivedDeploymentTasks.Applications.application_secured_ci.tableAlias(aliasArchivedApplications))
      )
      .join(
        new SelectBuilder(ArchiveRoles.tableName).as(aliasArchivedDeploymentTaskRoles),
        rolesCondition(aliasArchivedDeploymentTaskRoles, aliasArchivedDeploymentTaskPermissions,
          ArchivedDeploymentTasks.environment_secured_ci.tableAlias(aliasArchivedDeploymentTasks)),
        JoinType.Left
      )
      .join(
        new SelectBuilder(ArchiveRoles.tableName).as(aliasArchivedApplicationRoles),
        rolesCondition(aliasArchivedApplicationRoles, aliasArchivedApplicationPermissions,
          ArchivedDeploymentTasks.Applications.application_secured_ci.tableAlias(aliasArchivedApplications)),
        JoinType.Left
      )
    builder

  }

  override protected def controlTaskQuery: AbstractQueryBuilder = {
    val taskSelectBuilder = new SelectBuilder(ArchivedControlTasks.tableName)
    columns(taskSelectBuilder)
    addDateRangeConstraint(taskSelectBuilder)
    addTaskStateConstraint(taskSelectBuilder)
    addTargetsConstraint(taskSelectBuilder)
    addControlTaskNameConstraint(taskSelectBuilder)
    addUserConstraint(taskSelectBuilder)
    addWorkerNameConstraint(taskSelectBuilder)
    addOrderBy(taskSelectBuilder)

    val aliasControlTaskPermissions = "actperm"
    val aliasControlTaskRoles = "acttrole"
    val builder = new JoinBuilder(taskSelectBuilder.as(aliasControlTasks).distinct())
      .join(
        permissionsQuery(aliasControlTaskPermissions, aliasControlTaskRoles),
        permissionCondition(aliasControlTaskPermissions,
          ArchivedControlTasks.target_secured_ci.tableAlias(aliasControlTasks))
      )
      .join(
        new SelectBuilder(ArchiveRoles.tableName).as(aliasControlTaskRoles),
        rolesCondition(aliasControlTaskRoles, aliasControlTaskPermissions,
          ArchivedControlTasks.target_secured_ci.tableAlias(aliasControlTasks)),
        JoinType.Left
      )
    builder
  }

  private def permissionsQuery(aliasPermissions: String, aliasRoles: String): SelectBuilder = {
    new SelectBuilder(ArchivePermissionsSchema.tableName).as(aliasPermissions).where(
      cond.or(Seq(
        cond.in(ArchivePermissionsSchema.ROLE_ID.tableAlias(aliasPermissions), getRoles.asScala),
        cond.in(ArchiveRoles.ID.tableAlias(aliasRoles), getRoles.asScala),
        cond.subselect(ArchiveRoles.ID.tableAlias(aliasRoles),
          new SelectBuilder(ArchiveRoleRoles.tableName).select(ArchiveRoleRoles.ROLE_ID).where(cond.in(ArchiveRoleRoles.MEMBER_ROLE_ID, getRoles.asScala)))
      ))
    )
  }

  private def permissionCondition(aliasPermissions: String, conditionColumn: Selectable): cond = {
    cond.and(Seq(cond.equals(
      conditionColumn,
      ArchivePermissionsSchema.CI_ID.tableAlias(aliasPermissions)),
      cond.in(ArchivePermissionsSchema.PERMISSION_NAME.tableAlias(aliasPermissions), getPermissions.asScala)
    ))
  }

  private def rolesCondition(aliasRoles: String, aliasPermissions: String, onColumnCondition: Selectable): cond = {
    cond.and(Seq(
      cond.equals(
        onColumnCondition,
        ArchiveRoles.CI_ID.tableAlias(aliasRoles)),
      cond.equals(
        ArchivePermissionsSchema.ROLE_ID.tableAlias(aliasPermissions),
        ArchiveRoles.ID.tableAlias(aliasRoles))
    ))
  }

  override protected def addApplicationConstraint(selectBuilder: SelectBuilder): Unit = {
    val apps = applications.asScala
    if (apps.nonEmpty) {
      selectBuilder
        .where(getApplicationsByNameCondition(apps))
    }
  }
}

class SqlArchivedTaskSelectWithPermissionsQuery(searchParameters: ArchivedTaskSearchParameters)
                                               (override implicit val schemaInfo: SchemaInfo)
  extends AbstractSqlArchivedTaskSelectWithPermissionsQuery(searchParameters) {

  override private[sql] def addOrderBy(selectBuilder: AbstractQueryBuilder): Unit =
    orderBy.forEach { t =>
      val f: ColumnName = SqlArchivedTaskSelectQuery.orderFieldMapping(t.a)
      selectBuilder.orderBy(if (t.b) OrderBy.asc(f) else OrderBy.desc(f))
    }

  addPaging(selectBuilder)
}

class SqlArchivedTaskCountQuery(searchParameters: ArchivedTaskSearchParameters)
                               (override implicit val schemaInfo: SchemaInfo)
  extends SqlArchivedTaskQuery(searchParameters) {

  override protected def columns(builder: SelectBuilder): Unit = builder.select(new SqlLiteral("count(*)"))
}

class SqlArchivedTaskCountWithPermissionsQuery(searchParameters: ArchivedTaskSearchParameters)
                                              (override implicit val schemaInfo: SchemaInfo)
  extends AbstractSqlArchivedTaskSelectWithPermissionsQuery(searchParameters) {

  override protected def deploymentTaskQuery: AbstractQueryBuilder =
    new WrappedCountJoinBuilder(super.deploymentTaskQuery.asInstanceOf[JoinBuilder])

  override protected def controlTaskQuery: AbstractQueryBuilder =
    new WrappedCountJoinBuilder(super.controlTaskQuery.asInstanceOf[JoinBuilder])
}
