package com.xebialabs.deployit.engine.tasker.repository.sql

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.engine.api.dto.ActiveTaskStatus
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.engine.tasker.repository.ActiveTaskStatusRepository
import com.xebialabs.deployit.sql.base.schema.ActiveTaskStatusSchema._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Component
@Transactional("mainTransactionManager")
class SqlActiveTaskStatusRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                   (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ActiveTaskStatusRepository with ActiveTaskStatusQueries {

  override def create(taskId: TaskId, path: String, status: String): Unit = {
    jdbcTemplate.update(INSERT_TASK_STATUS, taskId, path, status)
  }

  override def findOne(taskId: TaskId, path: String): Option[ActiveTaskStatus] = {
    jdbcTemplate.query(SELECT_TASK_STATUS_BY_PATH, mapToRow, taskId, path).asScala.headOption
  }

  override def delete(taskId: TaskId): Unit = {
    jdbcTemplate.update(DELETE_TASK_STATUS, taskId)
  }

  override def update(taskId: TaskId, path: String, status: String): Unit = {
    jdbcTemplate.update(UPDATE_TASK_STATUS, status, taskId, path)
  }

  override def findAll(taskId: TaskId): List[ActiveTaskStatus] = {
    jdbcTemplate.query(SELECT_TASK_STATUS, mapToList, taskId)
  }

  private def mapToRow: RowMapper[ActiveTaskStatus] = (rs: ResultSet, _) =>
    new ActiveTaskStatus(rs.getString(taskId.name), rs.getString(path.name), rs.getString(status.name))

  private def mapToList: ResultSetExtractor[List[ActiveTaskStatus]] = rs => {
    var taskStatusList = List[ActiveTaskStatus]()
    while (rs.next) {
      val row = new ActiveTaskStatus(rs.getString(taskId.name), rs.getString(path.name), rs.getString(status.name))
      taskStatusList ::= row
    }
    taskStatusList
  }
}

trait ActiveTaskStatusQueries extends Queries {

  lazy val INSERT_TASK_STATUS: String = sqlb"insert into $tableName ($taskId, $path, $status) values (?,?,?)"
  lazy val UPDATE_TASK_STATUS: String = sqlb"update $tableName set $status = ? where $taskId = ? and $path = ?"
  lazy val SELECT_TASK_STATUS_BY_PATH: String = sqlb"select $taskId, $path, $status from $tableName where $taskId = ? and $path = ?"
  lazy val DELETE_TASK_STATUS: String = sqlb"delete from $tableName where $taskId = ?"
  lazy val SELECT_TASK_STATUS: String = sqlb"select $taskId, $path, $status from $tableName where $taskId = ?"

}


