package com.xebialabs.deployit.engine.tasker.repository.sql

import ai.digital.deploy.task.status.TaskPathStatusStore
import ai.digital.deploy.task.steplog.TaskStepLogStore
import akka.actor.ActorPath
import com.xebialabs.deployit.core.events.{TaskPathStatusDeleteEvent, TaskStepLogDeleteEvent}
import ai.digital.configuration.central.deploy.TaskerSystemProperties
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.util.queryStringWithInClause
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTask, ActiveTaskRepository, CrudTaskRepository}
import com.xebialabs.deployit.engine.tasker.{TaskId, TaskSpecification}
import com.xebialabs.deployit.sql.base.schema.ActiveTaskMetadataSchema.{metadata_key, metadata_value}
import com.xebialabs.deployit.sql.base.schema.ActiveTaskSchema.{description, state, task_owner, worker_id}
import com.xebialabs.deployit.sql.base.schema.{ActiveTaskMetadataSchema, ActiveTaskSchema, ActiveTaskMetadataKey => metadataKey}
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional
import java.sql.{ResultSet, Timestamp}
import java.util

import ai.digital.deploy.task.serdes.TaskAkkaSerializer
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
@Transactional("mainTransactionManager")
class SqlActiveTaskRepository(@Autowired taskPathStatusStore: TaskPathStatusStore,
                              @Autowired taskStepLogStore: TaskStepLogStore,
                              @Autowired val workerRepository: TaskExecutionWorkerRepository,
                              @Autowired taskerSystemProperties: TaskerSystemProperties,
                              @Autowired @Qualifier("mainJdbcTemplate") override val jdbcTemplate: JdbcTemplate)
                             (@Autowired @Qualifier("mainSchema") override implicit val schemaInfo: SchemaInfo)
  extends SqlCrudTaskRepository(taskPathStatusStore, taskStepLogStore, jdbcTemplate, taskerSystemProperties) with ActiveTaskRepository with ActiveTaskQueries with ActiveTaskMetadataQueries with Logging {

  override def store(taskId: TaskId, spec: TaskSpecification, path: ActorPath): Unit = {
    val workerAddress = path.address.toString
    workerRepository.getWorkerByAddress(workerAddress)
      .orElse(throw new IllegalStateException(s"Worker with $workerAddress not found in worker repository"))
      .foreach(worker => {
        spec.getMetadata.put("worker_name", worker.name)
        store(taskId, spec.getDescription, spec.getOwner.getName, worker.id, spec.getMetadata, spec)
      })
  }

  override def changeOwner(taskId: TaskId, newOwner: String): Unit = {
    jdbcTemplate.update(UPDATE_OWNER, newOwner, taskId)
  }

  override def workerAddress(taskId: TaskId): Option[String] = {
    firstOrNone(jdbcTemplate.query(SELECT_WORKER_ADDRESS, (rs: ResultSet, _: Int) => {
      rs.getString(1)
    }, taskId))
  }

  private def executeQuery(query: String, taskIds: Iterable[TaskId]): Map[String, List[String]] = {
    import ActiveTaskMetadataSchema._
    queryStringWithInClause(taskIds)(query, sqlb"task.$task_id", "taskIdsInClause") {
      jdbcTemplate.query(_, (rs: ResultSet, rowNum: Int) => (rs.getString(1), rs.getString(2)))
        .asScala.toList
    }.groupBy(_._1).view.mapValues(_.map(_._2)).toMap
  }

  override def ciPathsByDeployments(taskIds: List[TaskId]): Map[String, List[String]] = {
    schemaInfo.sqlDialect match {
      case DerbyDialect | OracleDialect | Db2Dialect =>
        executeQuery(ORACLE_OR_DERBY_SELECT_CI_PATHS_BY_DEPL_TASK_ID, taskIds)
      case _ =>
        executeQuery(SELECT_CI_PATHS_BY_DEPL_TASK_ID, taskIds)
    }
  }

  override def ciPathsByControlTasks(taskIds: List[TaskId]): Map[String, List[String]] = {
    schemaInfo.sqlDialect match {
      case DerbyDialect | OracleDialect | Db2Dialect =>
        executeQuery(ORACLE_OR_DERBY_SELECT_CI_PATHS_BY_CONTROL_TASK_ID, taskIds)
      case _ =>
        executeQuery(SELECT_CI_PATHS_BY_CONTROL_TASK_ID, taskIds)
    }
  }

  override def tasksWithStatus(): List[ActiveTask] = {
    jdbcTemplate.queryForList(SELECT_ALL_TASKS_WITH_SUMMARY).asScala.groupBy {
      _.get(ActiveTaskSchema.task_id.name).asInstanceOf[String]
    }.map { case (taskId, list) =>
      val firstElement = list.headOption.getOrElse(throw new NoSuchElementException(s"No active task with taskId $taskId"))
      val taskDescription = firstElement.get(description.name).asInstanceOf[String]
      val owner = firstElement.get(task_owner.name).asInstanceOf[String]
      val workerId = asInteger(firstElement.get(worker_id.name))
      val status = TaskExecutionState.valueOf(firstElement.get(state.name).asInstanceOf[String])
      val startDate = Option(firstElement.get(ActiveTaskSchema.startDate.name)).map(t => toDateTime(t.asInstanceOf[Timestamp])).orNull
      val completionDate = Option(firstElement.get(ActiveTaskSchema.completionDate.name)).map(t => toDateTime(t.asInstanceOf[Timestamp])).orNull
      val metadata = list.map { m => m.get(metadata_key.name).asInstanceOf[String] ->
        m.get(metadata_value.name).asInstanceOf[String] }.toMap.asJava
      new ActiveTask(taskId, taskDescription, owner, metadata, workerId, status, startDate, completionDate)
    }.toList
  }

  override def recordStartDate(taskId: TaskId): Unit = {
    jdbcTemplate.update(SET_START_DATE, toTimestamp(DateTime.now()), taskId)
  }

  override def recordCompletionDate(taskId: TaskId): Unit = {
    jdbcTemplate.update(SET_COMPLETION_DATE, toTimestamp(DateTime.now()), taskId)
  }

  override def updateState(taskId: TaskId, state:String): Unit = {
    jdbcTemplate.update(UPDATE_STATE, state, taskId)
  }
}

@Transactional("mainTransactionManager")
class SqlCrudTaskRepository(@Autowired val taskPathStatusStore: TaskPathStatusStore,
                            @Autowired val taskStepLogStore: TaskStepLogStore,
                            @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                            @Autowired taskerSystemProperties: TaskerSystemProperties
                           )
                           (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends CrudTaskRepository with ActiveTaskQueries with ActiveTaskMetadataQueries with Logging {

  import ActiveTaskMetadataSchema._
  import ActiveTaskSchema._

  private lazy val isResilient = taskerSystemProperties.resilient

  private lazy val serializer = {
    if(taskerSystemProperties.serialization.`type`.equals("jackson-yaml")) {
      TaskAkkaSerializer.getTaskAkkaSerializer(classOf[ai.digital.deploy.task.serdes.yaml.TaskerSerializableYaml])
    } else {
      TaskAkkaSerializer.getTaskAkkaSerializer(classOf[ai.digital.deploy.task.serdes.kryo.TaskerSerializableKryo])
    }
  }

  private def specToBinary(spec: TaskSpecification): Array[Byte] = {
    Try(serializer.toBinary(spec)) match {
      case Success(value) => value
      case Failure(exception) =>
        logger.error(exception.getMessage, exception)
        throw exception
    }
  }

  override def store(taskId: TaskId, description: String, owner: String, workerId: Integer, metadata: util.Map[String, String]): Unit = {
    jdbcTemplate.update(INSERT, taskId, description, owner, workerId)
    metadata.forEach { (k, v) =>
      jdbcTemplate.update(INSERT_METADATA, taskId, k, v)
    }
  }

  override def store(taskId: TaskId, description: String, owner: String, workerId: Integer, metadata: util.Map[String, String], specification: TaskSpecification): Unit = {
    if(isResilient) {
      jdbcTemplate.update(INSERT_SERIALIZATION_OBJECT, taskId, description, owner, workerId, taskerSystemProperties.serialization.`type`, specToBinary(specification))
    } else {
      jdbcTemplate.update(INSERT, taskId, description, owner, workerId)
    }
    metadata.forEach { (k, v) =>
      jdbcTemplate.update(INSERT_METADATA, taskId, k, v)
    }
  }

  private def getDeserializer(serializationType: String) : TaskAkkaSerializer = {
      if("jackson-yaml".equals(serializationType)) {
        TaskAkkaSerializer.getTaskAkkaSerializer(classOf[ai.digital.deploy.task.serdes.yaml.TaskerSerializableYaml])
      } else {
        TaskAkkaSerializer.getTaskAkkaSerializer(classOf[ai.digital.deploy.task.serdes.kryo.TaskerSerializableKryo])
      }
  }

  private def binaryToSpec(serializationType: String, taskSpecBytes: Array[Byte], taskId: String): TaskSpecification = {
   val deserializer = getDeserializer(serializationType)
   val taskSpecification = deserializer.fromBinary(taskSpecBytes, classOf[TaskSpecification]) match {
      case spec: TaskSpecification =>
        spec
      case _ =>
        throw new RuntimeException(s"Could not deserialize TaskSpecification [$taskId]")
    }
    taskSpecification
  }

  override def task(taskId: TaskId): Option[ActiveTask] = {
    case class ActiveTaskDetail (description: String, owner: String, workerId: Integer, state: TaskExecutionState = null, startDate: DateTime = null, completionDate: DateTime = null, serializationType: String = null, spec: TaskSpecification = null)

    def getTask: util.List[ActiveTaskDetail] = {
      jdbcTemplate.query(SELECT, (rs: ResultSet, _: Int) => {
        val taskDescription = rs.getString(description.name)
        val owner = rs.getString(task_owner.name)
        val workerId = asInteger(rs.getInt(worker_id.name))
        if (isResilient) {
          val serializationType = rs.getString(serialization_type.name)
          val taskSpecBytes = rs.getBytes(specification.name)
          val status =  TaskExecutionState.valueOf(rs.getString(state.name))
          val startDate =  Option(rs.getTimestamp(ActiveTaskSchema.startDate.name)).map(t => toDateTime(t)).orNull
          val completionDate = Option(rs.getTimestamp(ActiveTaskSchema.completionDate.name)).map(t => toDateTime(t)).orNull
          Option(serializationType, taskSpecBytes) match {
            case Some((seriaType: String, taskBytes: Array[Byte])) =>
              val taskSpec = binaryToSpec(seriaType, taskBytes, taskId)
              ActiveTaskDetail(taskDescription, owner, workerId, status, startDate, completionDate, serializationType, taskSpec)
            case _ =>
              ActiveTaskDetail(taskDescription, owner, workerId, status, startDate, completionDate)
          }
        } else {
          ActiveTaskDetail(taskDescription, owner, workerId)
        }
      }, taskId)
    }

    def getTaskMetaData: util.HashMap[String, String] = {
      jdbcTemplate.query(SELECT_METADATA, (rs: ResultSet) => {
        val map = new util.HashMap[String, String]()
        while (rs.next()) {
          map.put(
            rs.getString(metadata_key.name),
            rs.getString(metadata_value.name)
          )
        }
        map
      }, taskId)
    }

   firstOrNone(getTask).map {
      case activeTaskDetails: ActiveTaskDetail =>
        new ActiveTask(taskId, activeTaskDetails.description, activeTaskDetails.owner, getTaskMetaData,
          activeTaskDetails.workerId, activeTaskDetails.state, activeTaskDetails.startDate,
          activeTaskDetails.completionDate,activeTaskDetails.serializationType, Some(activeTaskDetails.spec))
    }
  }

  override def tasks(): List[ActiveTask] = {
    jdbcTemplate.queryForList(SELECT_ALL).asScala.groupBy {
      _.get(ActiveTaskSchema.task_id.name).asInstanceOf[String]
    }.map { case (taskId, list) =>
      val firstElement = list.headOption.getOrElse(throw new NoSuchElementException(s"No active task with taskId $taskId"))
      val taskDescription = firstElement.get(description.name).asInstanceOf[String]
      val owner = firstElement.get(task_owner.name).asInstanceOf[String]
      val workerId = asInteger(firstElement.get(worker_id.name))
      val metadata = list.map { m =>
        m.get(metadata_key.name).asInstanceOf[String] ->
          m.get(metadata_value.name).asInstanceOf[String]
      }.toMap.asJava
      if (isResilient) {
        val serializationType = firstElement.get(serialization_type.name).asInstanceOf[String]
        val taskSpecBytes = firstElement.get(specification.name).asInstanceOf[Array[Byte]]
        val status = TaskExecutionState.valueOf(firstElement.get(state.name).asInstanceOf[String])
        val startDate = Option(firstElement.get(ActiveTaskSchema.startDate.name)).map(t => toDateTime(t.asInstanceOf[Timestamp])).orNull
        val completionDate = Option(firstElement.get(ActiveTaskSchema.completionDate.name)).map(t => toDateTime(t.asInstanceOf[Timestamp])).orNull
        Option(serializationType, taskSpecBytes) match {
          case Some((seriaType: String, taskBytes: Array[Byte])) =>
            val taskSpec = binaryToSpec(seriaType, taskBytes, taskId)
            new ActiveTask(taskId, taskDescription, owner, metadata, workerId, status, startDate, completionDate, serializationType, Some(taskSpec))
          case _ =>
            new ActiveTask(taskId, taskDescription, owner, metadata, workerId, status, startDate, completionDate)
        }
      } else {
        new ActiveTask(taskId, taskDescription, owner, metadata, workerId)
      }

    }.toList
  }

  override def delete(taskId: TaskId): Unit = {
    jdbcTemplate.update(DELETE_METADATA, taskId)
    jdbcTemplate.update(DELETE, taskId)
    taskPathStatusStore.sendDeleteEvent(TaskPathStatusDeleteEvent(taskId))
    taskStepLogStore.sendStepLogDeleteEvent(TaskStepLogDeleteEvent(taskId))
  }

}

trait ActiveTaskQueries extends Queries {
  import ActiveTaskSchema._

  lazy val INSERT = sqlb"insert into $tableName ($task_id, $description, $task_owner, $worker_id) values (?,?,?,?)"
  lazy val INSERT_SERIALIZATION_OBJECT = sqlb"insert into $tableName ($task_id, $description, $task_owner, $worker_id, $serialization_type, $specification) values (?,?,?,?,?,?)"
  lazy val UPDATE_OWNER = sqlb"update $tableName set $task_owner = ? where $task_id = ?"
  lazy val SELECT: String = sqlb"select $task_id, $description, $task_owner, $worker_id, $state, $startDate, $completionDate, $serialization_type, $specification from $tableName where $task_id = ?"
  lazy val SELECT_WORKER_ADDRESS: String = {
    import com.xebialabs.deployit.engine.tasker.repository.sql.{WorkersSchema => ws}
    sqlb"select ${ws.address} from $tableName t left join ${ws.tableName} w on t.$worker_id = w.${ws.worker_id} where $task_id = ?"
  }
  lazy val SELECT_ALL: String = {
    import com.xebialabs.deployit.sql.base.schema.{ActiveTaskMetadataSchema => md}
    sqlb"select t.$task_id, t.$description, t.$task_owner, t.$worker_id, t.$state, t.$startDate, t.$completionDate, t.$serialization_type, t.$specification, md.${md.metadata_key}, md.${md.metadata_value} from $tableName t left join ${md.tableName} md on t.$task_id = md.${md.task_id}"
  }
  lazy val SELECT_ALL_TASKS_WITH_SUMMARY: String = {
    import com.xebialabs.deployit.sql.base.schema.{ActiveTaskMetadataSchema => md}
    sqlb"select t.$task_id, t.$description, t.$task_owner, t.$worker_id, t.$state, t.$startDate, t.$completionDate, md.${md.metadata_key}, md.${md.metadata_value} from $tableName t left join ${md.tableName} md on t.$task_id = md.${md.task_id}"
  }
  lazy val SET_START_DATE = sqlb"update $tableName set $startDate = ? where $task_id = ?"
  lazy val SET_COMPLETION_DATE = sqlb"update $tableName set $completionDate = ? where $task_id = ?"
  lazy val UPDATE_STATE = sqlb"update $tableName set $state = ? where $task_id = ?"

  lazy val DELETE = sqlb"delete from $tableName where $task_id = ?"
}

trait ActiveTaskMetadataQueries extends Queries {

  import ActiveTaskMetadataSchema._

  lazy val INSERT_METADATA = sqlb"insert into $tableName ($task_id, $metadata_key, $metadata_value) values (?,?,?)"
  lazy val SELECT_METADATA = sqlb"select $metadata_key, $metadata_value from $tableName where $task_id = ?"
  private lazy val SELECT_CI_PATHS_FRAGMENT: String = {
    import com.xebialabs.deployit.sql.base.schema.{CIS => ci}
    sqlb"  select task.${task_id}, ci.${ci.path} " +
      sqlb"from $tableName task inner join ${ci.tableName} ci on ci.${ci.ID} = ${schemaInfo.sqlDialect.castToInt(metadata_value.tableAlias("task"))} " +
      sqlb"where :taskIdsInClause "
  }
  lazy val SELECT_CI_PATHS_BY_DEPL_TASK_ID: String = {
    sqlb"$SELECT_CI_PATHS_FRAGMENT and (task.$metadata_key = '${metadataKey.ENVIRONMENT_INTERNAL_ID}' or task.$metadata_key = '${metadataKey.APPLICATION_INTERNAL_ID}')"
  }
  lazy val SELECT_CI_PATHS_BY_CONTROL_TASK_ID: String = {
    sqlb" $SELECT_CI_PATHS_FRAGMENT and task.$metadata_key = '${metadataKey.CONTROL_TASK_TARGET_INTERNAL_CI}'"
  }

  private lazy val ORACLE_OR_DERBY_SELECT_CI_PATHS_FRAGMENT: String = {
    import com.xebialabs.deployit.sql.base.schema.{CIS => ci}
    sqlb"  select task.${task_id}, ci.${ci.path} " +
      sqlb"from $tableName task inner join ${ci.tableName} ci on cast(ci.${ci.ID} as char(254)) = cast(task.$metadata_value as char(254)) " +
      sqlb"where :taskIdsInClause "
  }

  lazy val ORACLE_OR_DERBY_SELECT_CI_PATHS_BY_DEPL_TASK_ID: String = {
    sqlb"$ORACLE_OR_DERBY_SELECT_CI_PATHS_FRAGMENT and (task.$metadata_key = '${metadataKey.ENVIRONMENT_INTERNAL_ID}' or task.$metadata_key = '${metadataKey.APPLICATION_INTERNAL_ID}')"
  }
  lazy val ORACLE_OR_DERBY_SELECT_CI_PATHS_BY_CONTROL_TASK_ID: String = {
    sqlb" $ORACLE_OR_DERBY_SELECT_CI_PATHS_FRAGMENT and task.$metadata_key = '${metadataKey.CONTROL_TASK_TARGET_INTERNAL_CI}'"
  }

  lazy val DELETE_METADATA = sqlb"delete from $tableName where $task_id = ?"
}
