package com.xebialabs.deployit.engine.tasker.repository.sql

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.engine.api.dto.TaskStepLog
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.engine.tasker.repository.TaskStepLogRepository
import com.xebialabs.deployit.sql.base.schema.ActiveTaskStepLogSchema._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.{Isolation, Transactional}

import com.xebialabs.deployit.core.events.TaskStepLogEvent
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs, BatchExecutorRepository}

import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
class SqlActiveTaskStepLogRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                     @Autowired @Qualifier("mainBatchExecutorRepository") val batchExecutorRepository: BatchExecutorRepository)
                                    (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)

  extends TaskStepLogRepository with ActiveTaskStepLogQueries with Logging {

  @Transactional(value = "mainTransactionManager", readOnly = true)
  override def findOne(taskId: TaskId, stepPath: String): Option[TaskStepLog] = {
    val result = jdbcTemplate.query(SELECT_TASK_STEP_LOG_BY_PATH, mapToList(taskId), taskId, stepPath).headOption
    result
  }

  @Transactional("mainTransactionManager")
  override def delete(taskId: TaskId): Unit =
    jdbcTemplate.update(DELETE_TASK_STEP_LOG, taskId)

  @Transactional(value = "mainTransactionManager", isolation = Isolation.SERIALIZABLE)
  override def create(taskStepLog: TaskStepLogEvent, updateTime: Long): Unit = {
    jdbcTemplate.update(INSERT_TASK_STEP_LOG, taskStepLog.taskId, taskStepLog.stepPath, taskStepLog.log, taskStepLog.lineNumber, taskStepLog.logLevel, taskStepLog.attempt, updateTime)
  }

  @Transactional(value = "mainTransactionManager", readOnly = true)
  override def findAll(taskId: TaskId): List[TaskStepLog] =
    jdbcTemplate.query(SELECT_TASK_STEP_LOG, mapToList(taskId), taskId)

  @Transactional(value = "mainTransactionManager")
  override def batchCreate(taskStepLogs: Seq[TaskStepLogEvent], updateTime: Long): Unit = {
    val commands: Seq[BatchCommandWithArgs] = taskStepLogs.map(taskStepLog =>
      BatchCommand(INSERT_TASK_STEP_LOG, taskStepLog.taskId, taskStepLog.stepPath, taskStepLog.log, taskStepLog.lineNumber, taskStepLog.logLevel, taskStepLog.attempt, updateTime)
    )
    batchExecutorRepository.execute(commands)
  }

  private def mapToList(taskId: TaskId): ResultSetExtractor[List[TaskStepLog]] = rs => {
    var taskStepLogList = List[TaskStepLog]()
    val logMap = mutable.Map[String, String]()
    while (rs.next) {
      val stepPathKey = rs.getString(stepPath.name)
      val stepPathLog = rs.getString(log.name).stripSuffix("\n")
      if (logMap.contains(stepPathKey)) {
        val value = logMap(stepPathKey).appendedAll("\n" + stepPathLog)
        logMap.asJava.put(stepPathKey, value)
      } else {
        logMap.asJava.put(stepPathKey, stepPathLog)
      }
    }
    logMap.foreach { x =>
      val taskStepLog = TaskStepLog(taskId, x._1, x._2)
      taskStepLogList ::= taskStepLog
    }
    taskStepLogList
  }

}

private[sql] trait ActiveTaskStepLogQueries extends Queries {
  lazy val INSERT_TASK_STEP_LOG: String = sqlb"insert into $tableName ($taskId, $stepPath, $log, $lineNumber, $logLevel, ${attempt},$updateTime) values (?,?,?,?,?,?,?)"
  lazy val SELECT_TASK_STEP_LOG_BY_PATH: String = sqlb"select $taskId, $stepPath, $log from $tableName where $taskId = ? and $stepPath = ? order by $attempt, $lineNumber"
  lazy val DELETE_TASK_STEP_LOG: String = sqlb"delete from $tableName where $taskId = ?"
  lazy val SELECT_TASK_STEP_LOG: String = sqlb"select $taskId, $stepPath, $log, $lineNumber, $attempt from $tableName where $taskId = ? order by $stepPath, $attempt, $lineNumber"

}


