package ai.digital.deploy.task.status.queue

import java.util

import ai.digital.configuration.central.deploy.TaskerSystemProperties
import ai.digital.deploy.task.status._
import com.xebialabs.deployit.core.events.{TaskPathStatusDeleteEvent, TaskPathStatusEvent}
import com.xebialabs.xldeploy.jms.adapter.TaskPathStatusQueueNameResolver
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.jms.core.JmsTemplate
import org.springframework.scheduling.annotation.Scheduled
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

case class TaskPathStatusEventKey(taskId: String, path: String) {
  def this(event: TaskPathStatusEvent) = this(event.taskId, event.path)
}

@ConditionalOnProperty(prefix = "deploy.task", name = Array("resilient"), havingValue = "true", matchIfMissing = false)
@Component
@Autowired
class DefaultTaskPathStatusProducer(@Qualifier("taskPathStatusJmsTemplate") jmsTemplate: JmsTemplate,
                                    taskPathStatusQueueNameResolver: TaskPathStatusQueueNameResolver,
                                    taskerSystemProperties: TaskerSystemProperties) extends TaskPathStatusStore with Logging {

  TaskPathStatusStoreHolder.taskPathStatusStore.set(this)

  private val taskPathStatusEvents = new util.LinkedHashMap[TaskPathStatusEventKey, TaskPathStatusEvent]()

  private def enabled: Boolean = taskerSystemProperties.resilient

  private def flushThreshold: Int = taskerSystemProperties.events.taskPathStatus.queue.eventFlushThreshold

  override def sendChangeSetEvent(event: TaskPathStatusEvent): Unit =
    if (enabled) {
      taskPathStatusEvents.synchronized {
        taskPathStatusEvents.put(new TaskPathStatusEventKey(event), event)

        if (taskPathStatusEvents.size() >= flushThreshold) {
          logger.debug(s"Flushing change set events ${taskPathStatusEvents.size()}")
          flushChangeSetEvents()
        }
      }
    }

  override def sendDeleteEvent(event: TaskPathStatusDeleteEvent): Unit =
    if (enabled) {
      taskPathStatusEvents.synchronized {
        val entries = taskPathStatusEvents.entrySet().iterator()
        while (entries.hasNext) {
          val entry = entries.next()
          if (event.taskId == entry.getValue.taskId)
            entries.remove()
        }
      }
      jmsTemplate.convertAndSend(taskPathStatusQueueNameResolver.getTaskPathStatusQueueName, event)
    }

  @Scheduled(fixedDelayString = "#{@taskPathStatusQueueFlushResolver.getEventFlushPeriodInMillis()}")
  def sendPeriodicallyChangeSetEvents(): Unit =
    if (!taskPathStatusEvents.isEmpty) {
      taskPathStatusEvents.synchronized {
        logger.debug(s"Flushing change set events ${taskPathStatusEvents.size()}")
        flushChangeSetEvents()
      }
    }

  private def flushChangeSetEvents(): Unit = {
    val entries = taskPathStatusEvents.entrySet()
    val messageEntries = entries.asScala
      .map(_.getValue)
      .toSeq
    jmsTemplate.convertAndSend(taskPathStatusQueueNameResolver.getTaskPathStatusQueueName, TaskPathStatusEventMessage(messageEntries))
    entries.clear()
  }
}

@ConditionalOnProperty(prefix = "deploy.task", name = Array("resilient"), havingValue = "false", matchIfMissing = true)
@Component
class DummyTaskPathStatusProducer extends TaskPathStatusStore {
  override def sendChangeSetEvent(event: TaskPathStatusEvent): Unit = {}

  override def sendDeleteEvent(event: TaskPathStatusDeleteEvent): Unit = {}
}
