package com.xebialabs.deployit.provision
package steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Preview, PreviewStep, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Container, Environment}
import com.xebialabs.deployit.provision.resolver.TemplateResolver
import com.xebialabs.deployit.provision.resolver.TemplateResolver.ResolutionResult
import com.xebialabs.deployit.repository.ChangeSet

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.reflect.ClassTag
import scala.util.{Failure, Success}

@StepMetadata(name = "create-all-provisioned-cis")
class CreateAllProvisionedItems extends ProvisionedItemsStep with PreviewStep {

  @transient
  private lazy val templateResolver = TemplateResolver(deployedApplication)

  override def validScope: Scope = Scope.POST_PLAN

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext): Unit = {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = s"Register containers and dictionaries in ${correctDeployedApplication(ctx).getEnvironment.getName}"
    }
    order = if (order != null) order else 75
    setParameters(ctx)
  }

  override def getPreview: Preview = {
    val deployedItems = resolveTemplatesOnPackage() ++ resolvedTemplatesPerDeployed().values.flatten
    val previewMessages = deployedItems.map {
      case ResolutionResult(_, Success(id), _, _) =>
        s"Create or update $id"
      case ResolutionResult(templateId, Failure(ex), _, _) =>
        s"FAILED on $templateId with message: ${ex.getMessage}"
    }
    Preview.withContents(s"Register containers and dictionaries in ${deployedApplication.getEnvironment.getName}\n" + previewMessages.mkString("\n"))
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    implicit val changeSet: ChangeSet = new ChangeSet
    implicit val theCtx: ExecutionContext = ctx
    ctx.logOutput("The environment is being provisioned")
    resolveEnvironmentAndTemplates
    repositoryService.execute(changeSet)
    StepExitCode.SUCCESS
  }

  private def resolveEnvironmentAndTemplates(implicit ctx: ExecutionContext, changeSet: ChangeSet): Unit = {
    val cisOnPackage = resolveTemplatesOnPackage()
    val cisPerDeployed = resolvedTemplatesPerDeployed()
    val cache: mutable.Map[String, Environment] = mutable.Map[String, Environment]()
    val itemsToCreateOrUpdate: mutable.Map[String, ConfigurationItem] = mutable.Map[String, ConfigurationItem]()

    cisPerDeployed.foreach {
      case (deployed: DeployedType, cis: Set[ResolutionResult]) =>
        val resolvedResults = cisOnPackage ++ cis
        resolvedResults.foreach(ci => ctx.logOutput(s"Resolving ${ci.templateId}"))
        throwErrorOnFailedResolution(resolvedResults)
        setBoundConfigurationItemsOnDeployeds(itemsToCreateOrUpdate, deployed, cis)
        setBoundConfigurationItemsOnBlueprint(itemsToCreateOrUpdate, cisOnPackage)
        val allConfigurationItems = resolvedResults.map(_.configurationItem.getOrElse(throw new IllegalArgumentException("Cannot get CIs from template")))
        val members: JSet[Container] = new JHashSet[Container](filterConfigurationItems[Container](allConfigurationItems).toSet.asJava)

        val environment: Environment = getResolvedEnvironment(cache, deployed)
        environment.setMembers((environment.getMembers.asScala ++ members.asScala).asJava)

        val dictionaries = filterConfigurationItems[UdmDictionaryType](allConfigurationItems).toList
        val allDicts: JList[UdmDictionaryType] = (environment.getDictionaries.asScala ++ dictionaries).asJava
        environment.setDictionaries(allDicts)
    }
    changeSet.createOrUpdate((cache.values.toList ++ itemsToCreateOrUpdate.values.toList).asJava)
  }

  private def throwErrorOnFailedResolution(resolvedItems: Set[ResolutionResult])(implicit ctx: ExecutionContext): Unit = {
    val failed = resolvedItems.filter(_.configurationItem.isFailure)
    failed.foreach { result =>
      ctx.logError(s"Could not resolve ${result.templateId}. Cause: ${result.configurationItem.failed.get.getMessage}")
    }
    if (failed.nonEmpty) throw new RuntimeException("Not all templates have been resolved")
  }

  private[steps] def resolveTemplatesOnPackage(): Set[ResolutionResult] = {
    templateResolver.resolveBoundTemplate(deploymentPackage)
  }

  private def setBoundConfigurationItemsOnBlueprint(items: mutable.Map[String, ConfigurationItem], resolvedCIs: Set[ResolutionResult]): Unit = {
    val configurationItems = resolvedCIs.map(_.configurationItem.getOrElse(throw new IllegalArgumentException("Cannot get CIs for resolved CIs")))
    configurationItems.foreach(item => items.put(item.getId, item))
    val resolvedConfigurationItems : java.util.Set[ConfigurationItem] = new JHashSet(configurationItems.asJava)
    if (deployedApplication.getBoundConfigurationItems != null) {
      deployedApplication.getBoundConfigurationItems.addAll(resolvedConfigurationItems)
    } else {
      deployedApplication.setBoundConfigurationItems(resolvedConfigurationItems)
    }
  }

  private def setBoundConfigurationItemsOnDeployeds(items: mutable.Map[String, ConfigurationItem], deployed: DeployedType, cis: Set[ResolutionResult]): Unit = {
    val configurationItems = cis.map(_.configurationItem.getOrElse(throw new IllegalArgumentException("Cannot get CIs for resolved deployeds")))
    configurationItems.foreach(item => items.put(item.getId, item))
    val resolvedConfigurationItems : java.util.Set[ConfigurationItem] = new JHashSet(configurationItems.asJava)
    if (deployed.getBoundConfigurationItems != null) {
      deployed.getBoundConfigurationItems.addAll(resolvedConfigurationItems)
    } else {
      deployed.setBoundConfigurationItems(resolvedConfigurationItems)
    }
  }

  private[steps] def resolvedTemplatesPerDeployed(): Map[DeployedType, Set[ResolutionResult]] = {
    deployeds.map(p => p -> resolveTemplatesOnDeployed(p)).toMap
  }

  private[steps] def resolveTemplatesOnDeployed(deployed: DeployedType): Set[ResolutionResult] = {
    templateResolver.resolveBoundTemplate(deployed)
  }

  private[this] def filterConfigurationItems[C <: ConfigurationItem : ClassTag](boundCis: Iterable[ConfigurationItem]): Iterable[C] = {
    boundCis.withFilter(ci => ci.getType.getDescriptor.isAssignableTo(implicitly[ClassTag[C]].runtimeClass)).map(_.asInstanceOf[C])
  }
}
