package com.xebialabs.deployit.provision
package steps

import com.xebialabs.deployit.plugin.api.deployment.specification.{DeltaSpecification, Operation}
import com.xebialabs.deployit.plugin.api.rules.{Scope, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.provision.ProvisionHelper.getProvisionEnvironmentId
import com.xebialabs.deployit.repository.{RepositoryService, RepositoryServiceHolder}

import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait ProvisionedItemsStep extends BaseStep with ScopeValidator {
  private[steps] var deployedApplication: DeployedApplication = _
  private[steps] var provisioneds: Set[ProvisionedType] = _
  private[steps] var deployeds: Set[DeployedType] = _
  private[steps] var deploymentPackage: DeploymentPackage = _
  private[steps] var application: Application = _
  private[steps] var provisioningEnvironment: Environment = _

  def validScope: Scope

  private[steps] def validate(ctx: StepPostConstructContext): Unit = {
    if (!correctDeployedApplication(ctx).isInstanceOf[DeployedApplication]) {
      throw new IllegalArgumentException("This step can be applied to provisioned blueprint only")
    }
    validateScope(ctx.getScope, validScope)
  }

  private[steps] def correctDeployedApplication(ctx: StepPostConstructContext) = {
    val specification: DeltaSpecification = ctx.getSpecification
    if (specification.getOperation == Operation.DESTROY) {
      specification.getPreviousDeployedApplication
    } else {
      specification.getDeployedApplication
    }
  }

  private[steps] def setParameters(ctx: StepPostConstructContext): Unit = {
    deployedApplication = correctDeployedApplication(ctx)
    deploymentPackage = deployedApplication.getVersion.asInstanceOf[DeploymentPackage]
    provisioneds = deployedApplication.getDeployeds.asScala.collect { case p: ProvisionedType => p }.toSet
    deployeds = deployedApplication.getDeployeds.asScala.toSet
    provisioningEnvironment = deployedApplication.getEnvironment
    application = deploymentPackage.getApplication
  }

  protected def repositoryService: RepositoryService = {
    RepositoryServiceHolder.getRepositoryService
  }

  protected def getEnvironment: Environment = {
    val envId = deployedApplication.getEnvironment.getId
    val environment: Environment = repositoryService.read(envId)
    deployedApplication.setEnvironment(environment)
    deployedApplication.getEnvironment
  }

  protected def getResolvedEnvironment(cache: mutable.Map[String, Environment], deployed: DeployedType): Environment = {
    if (isInfraAsCodeDeployment(deployed)) {
      val envPath = deployed.asInstanceOf[BaseDeployedInfrastructureAsCodeType].getEnvironmentPath
      val envId = getProvisionEnvironmentId(envPath, deployedApplication.getEnvironment.getId)

      if (cache.contains(envId))
        cache(envId)
      else {
        updateCache(cache, if (envId.nonEmpty) repositoryService.read(envId) else getEnvironmentAndUpdateCache(cache, envId))
      }
    } else {
      getEnvironmentForOtherDeployeds(cache)
    }
  }

  private def getEnvironmentForOtherDeployeds(cache: mutable.Map[String, Environment]): Environment = {
    val envId = deployedApplication.getEnvironment.getId
    if (cache.contains(envId))
      cache(envId)
    else {
      updateCache(cache, getEnvironmentAndUpdateCache(cache, envId))
    }
  }

  private def updateCache(cache: mutable.Map[String, Environment], env: Environment): Environment = {
    cache.put(env.getId, env)
    env
  }

  private def getEnvironmentAndUpdateCache(cache: mutable.Map[String, Environment], envId: String): Environment = {
    val environment = getEnvironment
    cache += envId -> environment
    environment
  }

  protected def isInfraAsCodeDeployment(deployed: DeployedType): Boolean =
    deployed.isInstanceOf[BaseDeployedInfrastructureAsCodeType]

  protected def allCIsToDelete: Set[String] =
    resolvedTemplateOnDeployedApplication ++ resolvedTemplatesOnDeployeds ++ generatedConfigurationItems ++ resolvedEnvironmentsOnDeployeds

  private def resolvedTemplateOnDeployedApplication: Set[String] =
    deployedApplication.getBoundConfigurationItems.asScala.map(_.getId).toSet

  private def resolvedTemplatesOnDeployeds: Set[String] =
    deployeds.flatMap(_.getBoundConfigurationItems.asScala).map(_.getId)

  private def generatedConfigurationItems: Set[String] =
    deployedInfrastructureAsCode
      .flatMap(_.getGeneratedConfigurationItems.asScala)
      .map(_.getId)

  private def resolvedEnvironmentsOnDeployeds: Set[String] =
    deployeds.filter(deployed => isInfraAsCodeDeployment(deployed))
      .map(ci => ci.asInstanceOf[BaseDeployedInfrastructureAsCodeType].getEnvironmentPath)
      .filter(_.nonEmpty)
      .map(envPath => getProvisionEnvironmentId(envPath, deployedApplication.getEnvironment.getId))

  private def deployedInfrastructureAsCode: Set[BaseDeployedInfrastructureAsCodeType] =
    deployeds
      .filter(_.isInstanceOf[BaseDeployedInfrastructureAsCodeType])
      .map(_.asInstanceOf[BaseDeployedInfrastructureAsCodeType])
}
