package com.xebialabs.deployit.provision

import com.xebialabs.deployit.booter.local.generator.TypeGenerator
import com.xebialabs.deployit.booter.local.{LocalDescriptor, LocalPropertyDescriptor, TypeDefinition, TypeDefinitions}
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, DescriptorRegistryId, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.validation.Validator
import com.xebialabs.deployit.repository.core.Directory
import grizzled.slf4j.Logging

import java.util.Optional
import scala.jdk.CollectionConverters._

class TemplateTypeGenerator extends TypeGenerator with Logging {

  override def getName: String = "provision-templates"

  override def dependsOn(): JList[String] = Nil.asJava

  override def generateAndRegister(typeDefinitions: TypeDefinitions): Unit = {
    logger.debug(s"Found types for template generator: ${typeDefinitions.getDefinitions.asScala.map(d => s"${d.getType}")}")

    typeDefinitions.getDefinitions.asScala.filter(td => shouldHaveATemplate(td.getType)).foreach { td =>
      val descriptorRegistryId = td.getType.getTypeSource
      val descriptorRegistry = DescriptorRegistry.getDescriptorRegistry(descriptorRegistryId)
      val templateTypeDefinition: TypeDefinition = new TypeDefinition(descriptorRegistry) {
        setType(toTemplateType(td.getType))
        setSuperType(Type.valueOf(classOf[Template]))
        val superTypes: Set[Type] = (td.getType.getDescriptor.getSuperClasses.asScala ++ td.getType.getDescriptor.getInterfaces.asScala)
          .filter(shouldHaveATemplate).map(toTemplateType).toSet
        interfaces = superTypes.toList.asJava

        override def createDescriptor(): LocalDescriptor = {
          val localDescriptor = td.getType.getDescriptor.asInstanceOf[LocalDescriptor]
          new TemplateLocalDescriptor(toTemplateType(localDescriptor.getType), localDescriptor, superTypes)
        }
      }
      logger.debug(s"Adding new template type: ${templateTypeDefinition.getType}")
      typeDefinitions.addTypeDef(templateTypeDefinition)
    }
  }

  class TemplateLocalDescriptor(registeredType: Type, localDescriptor: LocalDescriptor, interfaceTypes: Set[Type]) extends LocalDescriptor(registeredType) {
    addSuperClass(Type.valueOf(classOf[Template]))
    setRootName(Optional.empty())
    setClazz(classOf[Template])
    setDescription(s"Template of ${localDescriptor.getType}")
    setVirtual(localDescriptor.isVirtual)
    initHierarchy()
    interfaceTypes.foreach(addInterface)
    localDescriptor.getInterfaces.asScala.filter(shouldHaveATemplate).map(toTemplateType).foreach(addInterface)
    localDescriptor.getSuperClasses.asScala.filter(shouldHaveATemplate).map(toTemplateType).foreach(addSuperClass)
    addPropertyDescriptorsFrom(localDescriptor)
    resolveIcon()

    private def addPropertyDescriptorsFrom(descriptor: LocalDescriptor): Unit =
      descriptor.getPropertyDescriptors.asScala.foreach(pd =>
        addPropertyDescriptor(new TemplatePropertyDescriptor(descriptorRegistry().getId, this, pd.asInstanceOf[LocalPropertyDescriptor]))
      )

    class TemplatePropertyDescriptor(typeSource: DescriptorRegistryId,
                                     templateLocalDescriptor: TemplateLocalDescriptor,
                                     propertyDescriptor: LocalPropertyDescriptor) extends LocalPropertyDescriptor(typeSource) {
      setDeclaringDescriptor(templateLocalDescriptor)
      setFromPropertyDescriptor(propertyDescriptor)
      addDefaultValidationRules()
      registerDefault(propertyDescriptor)
      reInitializeRequired()

      override def doSetValue(item: ConfigurationItem, value: scala.Any): Unit = {
        templateLocalDescriptor.setSyntheticPropertyValue(item, getName, value)
      }

      override def get(item: ConfigurationItem): AnyRef = {
        templateLocalDescriptor.getSyntheticPropertyValue(item, getName)
      }

      override def setEnumValues(enumValues: JList[String]): Unit = super.setEnumValues(null)

      override def setEnumClass(enumClass: Class[_]): Unit = super.setEnumClass(null)

      override def setInspectionProperty(inspectionProperty: Boolean): Unit = super.setInspectionProperty(false)

      override def setRequiredForInspection(requiredForInspection: Boolean): Unit = super.setRequiredForInspection(false)

      override def setKind(kind: PropertyKind): Unit = {
        if (kind.isSimple) {
          super.setKind(PropertyKind.STRING)
        } else {
          super.setKind(kind)
        }
      }

      override def setReferencedType(referencedType: Type): Unit = if (referencedType != null) {
        if (shouldHaveATemplate(referencedType)) {
          super.setReferencedType(toTemplateType(referencedType))
        } else {
          super.setReferencedType(referencedType)
        }
      }

      override def setValidationRules(validationRules: JSet[Validator[_]]): Unit = {
        super.setValidationRules(new JHashSet())
      }
    }

  }

  private def shouldHaveATemplate(ciType: Type): Boolean = {
    ciType.getPrefix != "upm" && !ciType.instanceOf(typeOf[Directory]) &&
      (ciType.isSubTypeOf(typeOf[Container]) || !ciType.isSubTypeOf(typeOf[EmbeddedDeployedContainerType])) &&
      !ciType.isSubTypeOf(typeOf[EmbeddedDeployable]) &&
      !ciType.isSubTypeOf(typeOf[Deployable]) &&
      !ciType.isSubTypeOf(typeOf[Parameters])
  }

  private def toTemplateType(ciType: Type): Type = Type.valueOf("template." + ciType.getPrefix, ciType.getName)

}
