package com.xebialabs.deployit.core.api;

import com.xebialabs.deployit.core.api.dto.DefinedPlaceholderReference;
import com.xebialabs.deployit.core.api.dto.DefinedPlaceholderView;
import com.xebialabs.deployit.core.api.dto.DictionaryWithValueView;
import com.xebialabs.deployit.core.api.dto.ResolvedPlaceholderView;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.xlplatform.documentation.PublicApi;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import java.util.List;

/**
 * Information about placeholders
 */
@Path("/placeholder")
@Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
@Consumes({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
@PublicApi
public interface PlaceholderService {

    /**
     * Find all defined placeholders, optionally filtered by (exact) key, (part of) the application ID and (part of) the
     * application name.
     *
     * @param key (Optional) the exact key to look for
     * @param applicationId (Optional) (part of) the application ID to filter on
     * @param applicationName (Optional) (part of) the application name to filter on
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return a list of {@see com.xebialabs.deployit.core.api.dto.DefinedPlaceholderView} objects
     */
    @GET
    @Path("defined")
    List<DefinedPlaceholderView> definedPlaceholders(
            @QueryParam("key") String key,
            @QueryParam("idPattern") String applicationId,
            @QueryParam("namePattern") String applicationName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

    /**
     * Find placeholders, optionally filtered by (exact) key, or (part of) the value, dictionary ID and/or dictionary name
     * Returns whether it is encrypted; if it isn't and the user has read access, provides its value; and tells which dictionary
     * defines its value
     *
     * @param key (Optional) The exact placeholder key to look for
     * @param value (Optional) (Part of) the placeholder value to look for
     * @param dictionaryId (Optional) (Part of) the dictionary ID to look for
     * @param dictionaryName (Optional) (Part of) the dictionary name to look for
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return a list of {@see com.xebialabs.deployit.core.api.dto.DictionaryWithValueView} objects
     */
    @GET
    @Path("defined/by-dictionary")
    List<DictionaryWithValueView> definedPlaceholdersByDictionary(
            @QueryParam("key") String key,
            @QueryParam("value") String value,
            @QueryParam("idPattern") String dictionaryId,
            @QueryParam("namePattern") String dictionaryName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

    /**
     * Find placeholders, optionally filtered by (exact) key, or (part of) the environment name or ID
     *
     * @param key (Optional) The exact placeholder key to look for
     * @param environmentId (Optional) (Part of) the dictionary ID to look for
     * @param environmentName (Optional) (Part of) the dictionary name to look for
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return A list of {@see com.xebialabs.deployit.core.api.dto.DefinedPlaceholderReference} objects
     */
    @GET
    @Path("defined/by-environment")
    List<DefinedPlaceholderReference> definedPlaceholdersByEnvironment(
            @QueryParam("key") String key,
            @QueryParam("idPattern") String environmentId,
            @QueryParam("namePattern") String environmentName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

    /**
     * Find placeholders by (part of their) key.
     *
     * @param key the key name pattern to match against
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return a list of placeholder key names
     */
    @GET
    @Path("defined/names")
    List<String> definedPlaceholdersNames(
            @QueryParam("namePattern") String key,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

    /**
     * Gets resolved placeholder values, as far as the user has read permissions on them, on an environment, optionally filtered
     * by (parts of) the key name, the placeholder value, and/or deployed application name, dictionary name, and/or host name that
     * the value is restricted to.
     *
     * Returns information about the placeholder key, value (if readable), whether it is an encrypted entry, and restrictions
     *
     * @param environmentId The environment to query the resolved placeholder values against
     * @param key (Optional) (part of) placeholder key to look for
     * @param value (Optional) (part of) the placeholder value to look for
     * @param deployedName (Optional) (part of) the deployed application name that the placeholder should be restricted to
     * @param dictionaryName (Optional) (part of) the dictionary name that defines the placeholder
     * @param hostName (Optional) (part of) the host name that the placeholder is restricted to
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return a list of {@see com.xebialabs.deployit.core.api.dto.ResolvedPlaceholderView} objects
     */
    @GET
    @Path("environment/{ID:.+}")
    List<ResolvedPlaceholderView> findPlaceholdersForEnvironment(
            @PathParam("ID") String environmentId,
            @QueryParam("key") String key,
            @QueryParam("value") String value,
            @QueryParam("deployedName") String deployedName,
            @QueryParam("dictionaryName") String dictionaryName,
            @QueryParam("hostName") String hostName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

    /**
     * Gets resolved placeholder values for placeholders that have been restricted to a host, optionally filtered
     * by (parts of) the key name, the placeholder value, and/or deployed application name, dictionary name, and/or host name that
     * the value is restricted to.
     *
     * Returns information about the placeholder key, value (if readable), whether it is an encrypted entry, and restrictions
     *
     * @param hostId the host ID to query resolved placeholders against
     * @param key (Optional) (part of) placeholder key to look for
     * @param value (Optional) (part of) the placeholder value to look for
     * @param deployedName (Optional) (part of) the deployed application name that the placeholder should be restricted to
     * @param dictionaryName (Optional) (part of) the dictionary name that defines the placeholder
     * @param environmentName (Optional) (part of) the environment name that the placeholder is restricted to
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return a list of {@see com.xebialabs.deployit.core.api.dto.ResolvedPlaceholderView} objects
     */
    @GET
    @Path("host/{ID:.+}")
    List<ResolvedPlaceholderView> findPlaceholdersForHost(
            @PathParam("ID") String hostId,
            @QueryParam("key") String key,
            @QueryParam("value") String value,
            @QueryParam("deployedName") String deployedName,
            @QueryParam("dictionaryName") String dictionaryName,
            @QueryParam("hostName") String environmentName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

    /**
     * Gets historic resolved placeholder values on an environment, optionally restricted to finding those used in a specific deployment
     * task, optionally filtered by (parts of) the key name, the placeholder value, and/or deployed application name, dictionary
     * name, and/or host name that the value is restricted to.
     *
     * Returns information about the placeholder key, value (if readable), whether it is an encrypted entry, and restrictions
     *
     * @param environmentId The environment to query the resolved placeholder values against
     * @param key (Optional) (part of) placeholder key to look for
     * @param value (Optional) (part of) the placeholder value to look for
     * @param deployedName (Optional) (part of) the deployed application name that the placeholder should be restricted to
     * @param dictionaryName (Optional) (part of) the dictionary name that defines the placeholder
     * @param hostName (Optional) (part of) the host name that the placeholder is restricted to
     * @param taskId (Optional) The task ID for which to find used placeholders
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return a list of {@see com.xebialabs.deployit.core.api.dto.ResolvedPlaceholderView} objects
     */
    @GET
    @Path("archived/environment/{ID:.+}")
    List<ResolvedPlaceholderView> findArchivedPlaceholdersForEnvironment(
            @PathParam("ID") String environmentId,
            @QueryParam("key") String key,
            @QueryParam("value") String value,
            @QueryParam("deployedName") String deployedName,
            @QueryParam("dictionaryName") String dictionaryName,
            @QueryParam("hostName") String hostName,
            @QueryParam("taskId") String taskId,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

}
