package com.xebialabs.deployit.engine.tasker.log.elastic

import com.xebialabs.deployit.engine.api.execution._
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.log.StepLogRetriever
import grizzled.slf4j.Logging
import org.joda.time.DateTime

import java.util
import scala.jdk.CollectionConverters._

class ElasticStepLogRetriever(logRetrieverHelper: ElasticLogRetrieverHelper) extends StepLogRetriever with Logging {
  override def retrieveLogs(taskId: TaskId, stepPath: BlockPath, step: StepState): StepState = {
    step match {
      case taskStep: TaskStep => new ElasticLogStepState(taskId, stepPath, taskStep)(logRetrieverHelper)
      case _ => step
    }
  }

  override def retrieveLogs(taskId: TaskId, block: BlockState): BlockState = {
    block match {
      case stepBlock: StepBlock => new ElasticLogStepBlockState(taskId, stepBlock)(logRetrieverHelper)
      case _ => block
    }
  }

  override def fillLogs(task: Task): Unit = {
    try {
      logRetrieverHelper.getHits(task.getId())
        .groupBy(m => m(STEP_PATH).asInstanceOf[String])
        .foreach { case (stepPath, list) =>
          val path = BlockPath(stepPath)
          val logs = toLogs(list)
          if (logs.nonEmpty) {
            task.getBlock(path.init)
              .foreach(_.getStep(BlockPath(path.elems.lastOption.getOrElse(throw new NoSuchElementException("Cannot read last path element")))) match {
                case s: TaskStep => s.setLogs(logs.init.asJava, logs.lastOption.orNull)
                case _ =>
              })
          }
        }
    }
    catch {
      case e: Exception =>
        error("Could not read logs to external storage, logs will be presented as empty.",  e)
    }
  }

  override def removeLogs(taskId: TaskId): Unit = {
    try {
      logRetrieverHelper.removeLogs(taskId)
    }
    catch {
      case e: Exception =>
        error("Could not remove logs from external storage, logs will be left behind - check the error below.",  e)
    }
  }
}

class ElasticLogStepBlockState(taskId: TaskId, stepBlock: StepBlock)(implicit logRetrieverHelper: ElasticLogRetrieverHelper) extends StepBlockState with Logging {

  override def getSteps: util.List[StepState] = {
    val list = stepBlock.getSteps()
    val result = new util.ArrayList[StepState](list.size())
    for (i <- 0 until list.size()) {
      result.add(list.get(i) match {
        case taskStep : TaskStep => new ElasticLogStepState(taskId, stepBlock.id / (i + 1), taskStep)
        case o => o
      })
    }
    result
  }

  //Delegates
  override def getCurrentStep: Int = stepBlock.getCurrentStep
  override def getSatelliteId: String = stepBlock.getSatelliteId
  override def getId: String = stepBlock.getId
  override def getDescription: String = stepBlock.getDescription
  override def getState: BlockExecutionState = stepBlock.getState()
  override def getSatelliteConnectionState: SatelliteConnectionState = stepBlock.getSatelliteConnectionState()
}

class ElasticLogStepState(taskId: TaskId, stepPath: BlockPath, taskStep: TaskStep)(implicit logRetrieverHelper: ElasticLogRetrieverHelper) extends StepState with Logging {

  private lazy val logs: List[String] = fetchLogs()

  override def getLog: String = checkMemory(taskStep.getLog) {
    logs.lastOption.getOrElse("")
  }

  override def getPreviousAttemptsLogs: util.List[String] = checkMemory(taskStep.getPreviousAttemptsLogs) {
    if (logs.isEmpty) new util.ArrayList[String]() else logs.init.asJava
  }

  private def checkMemory[T](inmemory: T)(b: => T): T = {
    if (!isEmpty(inmemory)) {
      warn(s"Found logs in memory for $taskId / ${stepPath.toBlockId} while expecting logs in ELK.")
      inmemory
    } else {
      b
    }
  }

  private def isEmpty[T](inmemory: T) = Option(inmemory).isEmpty || (inmemory match {
    case s: String => s.isEmpty
    case l: util.List[_] => l.isEmpty
    case _ => false
  })

  private def fetchLogs(): List[String] = {
    try {
      toLogs(logRetrieverHelper.getHits(taskId, stepPath.toBlockId))
    }
    catch {
      case e: Exception =>
        error("Could not read logs to external storage, logs could be presented as empty.",  e)
        List[String]()
    }

  }

  //Delegates
  override def getDescription: String = taskStep.getDescription
  override def getState: StepExecutionState = taskStep.getState
  override def getStartDate: DateTime = taskStep.getStartDate
  override def getCompletionDate: DateTime = taskStep.getCompletionDate
  override def getFailureCount: Int = taskStep.getFailureCount
  override def getMetadata: util.Map[String, String] = taskStep.getMetadata
  override def getSkippable: Boolean = taskStep.getSkippable
}
