package com.xebialabs.deployit.task.archive.queue

import com.xebialabs.deployit.task.archive.compression.CompressionUtils
import com.xebialabs.deployit.task.archive.{BatchSecureCiMessage, BatchTaskArchiveRepository, SecureCiMessage, SecureDirectoryReferenceMessage}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.jms.annotation.JmsListener
import org.springframework.jms.support.converter.MessageConverter
import org.springframework.stereotype.Component

import jakarta.jms.TextMessage

@Component
@Autowired
class DeploymentArchiveMessageConsumer(taskArchiveRepository: BatchTaskArchiveRepository,
                                       jacksonJmsMessageConverter: MessageConverter) extends Logging {
  @JmsListener(destination = ArchiveTaskQueueing.ArchiveQueueSpELExpression,
    containerFactory = "xlJmsListenerContainerFactory")
  def receiveTextMessage(message: TextMessage): Unit = {

    logger.debug(s"Received message ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp}")

    jacksonJmsMessageConverter.fromMessage(message) match {
      case message: Array[Byte] =>
        val taskMessage = CompressionUtils.decompressTaskMessage(message)
        logger.debug(s"Consuming Archive Message for taskId [${taskMessage.taskId}]")
        taskArchiveRepository.store(taskMessage)
      case secureCiMessage: SecureCiMessage if message.getBooleanProperty(ArchiveTaskQueueing.IsBatchProperty) =>
        taskArchiveRepository.updateSecureCi(secureCiMessage.securedCi,
          secureCiMessage.environmentIds,
          secureCiMessage.applicationIds,
          secureCiMessage.otherCiIds)
      case secureCiMessage: SecureCiMessage =>
        taskArchiveRepository.updateSecureCi(secureCiMessage.primaryKey.orNull, secureCiMessage.securedCi)
      case secureDirectoryRefMessage: SecureDirectoryReferenceMessage if message.getBooleanProperty(ArchiveTaskQueueing.IsBatchProperty) =>
        taskArchiveRepository.updateSecuredDirectoryRef(
          secureDirectoryRefMessage.securedDirectoryRef,
          secureDirectoryRefMessage.environmentIds,
          secureDirectoryRefMessage.applicationIds,
          secureDirectoryRefMessage.otherCiIds
        )
      case secureDirectoryReferenceMessage: SecureDirectoryReferenceMessage =>
        taskArchiveRepository.updateSecuredDirectoryRef(
          secureDirectoryReferenceMessage.primaryKey.orNull,
          secureDirectoryReferenceMessage.securedDirectoryRef
        )
      case batch: BatchSecureCiMessage =>
        taskArchiveRepository.batchUpdateSecureCi(batch)
    }
  }
}
