package com.xebialabs.deployit.pluginmanager

import ai.digital.configuration.central.deploy.db.{DatabaseProperties, MainDatabase}
import com.tqdev.metrics.core.MetricRegistry
import com.tqdev.metrics.jdbc.InstrumentedDataSource
import com.xebialabs.deployit.TaskEngineLaunchOptions
import com.xebialabs.deployit.core.config.SqlConfiguration.hikariDataSource
import com.xebialabs.deployit.core.metrics.XldDbMetricsTrackerFactory
import com.xebialabs.plugin.manager.startup.{PluginSynchronizer, SourceOfTruth}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.datasource.DataSourceTransactionManager
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.interceptor.TransactionProxyFactoryBean

import java.beans.BeanProperty
import java.util.Properties
import javax.sql.DataSource

@Configuration
class PluginManagerConfig {

  @Bean
  def pluginSynchronizer(repository: SqlPluginRepository): PluginSynchronizer = {
    val pluginSource = TaskEngineLaunchOptions.getInstance.pluginSource
    val sourceOfTruth = if (pluginSource != null) SourceOfTruth.withName(pluginSource) else SourceOfTruth.DATABASE
    new PluginSynchronizer(repository, "xl-deploy", sourceOfTruth)
  }

  @Bean
  def sqlPluginRepository(@Qualifier("pluginManagerJdbcTemplate") jdbcTemplate: JdbcTemplate,
                          @Qualifier("pluginManagerTransactionManager") tm: PlatformTransactionManager): TransactionProxyFactoryBean = {
    val target = new SqlPluginRepository(jdbcTemplate)
    val factoryBean = new TransactionProxyFactoryBean()
    factoryBean.setTransactionManager(tm)
    factoryBean.setTarget(target)
    factoryBean.setProxyTargetClass(true)
    val txProperties = new Properties()
    txProperties.setProperty("*", "PROPAGATION_REQUIRED,ISOLATION_READ_COMMITTED")
    factoryBean.setTransactionAttributes(txProperties)
    factoryBean
  }

  @BeanProperty
  var database: DatabaseProperties = new DatabaseProperties()

  @Autowired(required = true)
  var mainDatabase: MainDatabase = _

  @Autowired
  var xldDbMetricsTrackerFactory: XldDbMetricsTrackerFactory = _

  @Bean(destroyMethod = "close")
  def pluginManagerDataSource: DataSource = {
    val database = if (this.database.hasConfigured) this.database else mainDatabase.database
    instrumentedDataSource(hikariDataSource(database, "PluginManagerPool", xldDbMetricsTrackerFactory))
  }

  @Bean
  def pluginManagerJdbcTemplate: JdbcTemplate = new JdbcTemplate(pluginManagerDataSource, true)

  @Bean
  def pluginManagerTransactionManager: PlatformTransactionManager =
    new DataSourceTransactionManager(pluginManagerDataSource)

  private def instrumentedDataSource(dataSource: DataSource) = new InstrumentedDataSource(dataSource, MetricRegistry.getInstance())

}