package com.xebialabs.deployit

import ai.digital.configuration.central.deploy.CommandWhitelistProperties
import ai.digital.deploy.core.common.XldServerPaths
import ai.digital.deploy.core.common.XldServerPaths._
import com.xebialabs.deployit.booter.local.LocalBooter
import com.xebialabs.deployit.booter.remote.resteasy.InternalServerErrorClientResponseInterceptor
import com.xebialabs.deployit.bootstrap.TaskEnginePekkoBootstrapper
import com.xebialabs.deployit.config.TaskEngineHoconConfigLoader
import com.xebialabs.deployit.configuration.XldProductConfiguration
import com.xebialabs.deployit.configuration.system.SystemValues
import com.xebialabs.deployit.core.config.hash.{ConfigurationHashConfiguration, InMemoryConfigurationHashProvider}
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.spi.services.RepositoryFactory
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.log.{StepLogFactory, StepLogRetriever}
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import com.xebialabs.deployit.env.TaskEngineActiveProfileRegistry
import com.xebialabs.deployit.hocon.HoconPropertySourceLoader
import com.xebialabs.deployit.inspection.service.discovery.DiscoveryWorker
import com.xebialabs.deployit.log.LogbackHelper.{disableConsoleLoggerIfNeeded, enableJavaUtilLoggingBridge}
import com.xebialabs.deployit.pluginmanager.PluginManagerConfig
import com.xebialabs.deployit.resolver.XLPluginAwareResourcePatternResolver
import com.xebialabs.deployit.resteasy.DeployitExceptionRethrower
import com.xebialabs.deployit.util.{DeployitKeys, PasswordEncrypter}
import com.xebialabs.plugin.Xlp
import com.xebialabs.xlplatform.config.ProductConfiguration
import com.xebialabs.xlplatform.settings.shared.TaskerSettings
import com.xebialabs.xltype.serialization.rest.LocalDateParamConverter
import grizzled.slf4j.Logging
import org.apache.pekko.actor.ActorSystem
import org.bouncycastle.jce.provider.BouncyCastleProvider
import org.jboss.resteasy.core.ResteasyDeploymentImpl
import org.kohsuke.args4j.{CmdLineException, CmdLineParser, ParserProperties}
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.boot.autoconfigure.EnableAutoConfiguration
import org.springframework.boot.autoconfigure.security.oauth2.client.reactive.ReactiveOAuth2ClientAutoConfiguration
import org.springframework.boot.autoconfigure.security.servlet.SecurityAutoConfiguration
import org.springframework.boot.autoconfigure.web.servlet.MultipartAutoConfiguration
import org.springframework.boot.{Banner, CommandLineRunner, SpringApplication, WebApplicationType}
import org.springframework.context.annotation.{Configuration, Import, ImportResource}
import org.springframework.context.support.GenericApplicationContext
import org.springframework.jms.config.JmsListenerContainerFactory
import org.springframework.jms.listener.DefaultMessageListenerContainer

import java.io.PrintStream
import java.security._
import java.util.Arrays.asList
import java.util.function.Supplier
import scala.jdk.CollectionConverters._

object TaskExecutionEngineBootstrapper {
  private val logger = LoggerFactory.getLogger(classOf[TaskExecutionEngineBootstrapper])

  /**
   * Use currently server configuration with defaults.
   * In future this will be replaced with reading configuration from the xl-worker.yaml.
   */
  val serverConfig: ServerConfiguration = {
    val serverConfigFile = new ServerConfigFile(DEFAULT_CONFIGURATION_FILE);
    if (serverConfigFile.exists()) {
      val serverConfiguration = serverConfigFile.loadConfig(false, false, true)
      serverConfiguration
    } else {
      val serverConfiguration = new ServerConfiguration(true)
      serverConfiguration.loadDefault()
      serverConfiguration
    }
  }

  Xlp.init(XL_PLUGIN_EXTENSION, asList(XL_HOTFIX_PLUGINS_FOLDER, XL_PLUGINS_LOCAL_FOLDER, XL_PLUGINS_OFFICIAL_FOLDER), asList(EXT_FOLDER))
  LocalBooter.boot(XldServerPaths.TYPE_DEFAULTS)

  SystemValues
    .apply()
    .apply(TaskEngineActiveProfileRegistry.getActiveProfiles(serverConfig))
    .applyConfigClientOnlySystemValues()

  disableConsoleLoggerIfNeeded("STDOUT")
  enableJavaUtilLoggingBridge()
  PasswordEncrypter.init(DeployitKeys.getPasswordEncryptionKey(serverConfig.getRepositoryKeyStorePassword))
  Security.addProvider(new BouncyCastleProvider)
  ServerConfiguration.setInstance(serverConfig)

  SystemValues.apply(serverConfig)

  def main(args: Array[String]): Unit = {
    val launchOptions = parseCmdLineArgs(args)
    logCmdLineArgs(launchOptions)
    TaskEngineLaunchOptions.setInstance(launchOptions)
    HoconPropertySourceLoader.setInstance(new TaskEngineHoconConfigLoader(launchOptions))

    val app = new SpringApplication(classOf[TaskExecutionEngineBootstrapper])
    app.setResourceLoader(new XLPluginAwareResourcePatternResolver())
    app.setWebApplicationType(WebApplicationType.NONE)
    app.setAllowBeanDefinitionOverriding(true)
    app.setBannerMode(Banner.Mode.OFF)
    app.run(args: _*)
  }

  private def parseCmdLineArgs(args: Array[String]): TaskEngineLaunchOptions = {
    val launchOptions = new TaskEngineLaunchOptions
    val parser = new CmdLineParser(launchOptions, ParserProperties.defaults().withUsageWidth(120))
    try parser.parseArgument(args.toList.asJava) catch {
      case e: CmdLineException =>
        System.err.println(e.getMessage)
        System.err.print("run(.sh|.cmd)")
        parser.printSingleLineUsage(System.err)
        printUsageMessage(System.err)
        parser.printUsage(System.err)
        System.exit(0);
    }
    launchOptions
  }

  private def logCmdLineArgs(launchOptions: TaskEngineLaunchOptions): Unit = {
    logger.info("Starting worker with the following command line options. {}", launchOptions.toString)
  }

  private def printUsageMessage(out: PrintStream): Unit = {
    out.println(
      """
        |Usage: Start XL Deploy in worker mode.
      """.stripMargin)
  }
}

@Configuration
@EnableAutoConfiguration(
  exclude = Array(
    classOf[SecurityAutoConfiguration],
    classOf[MultipartAutoConfiguration],
    classOf[ReactiveOAuth2ClientAutoConfiguration]
  )
)
@Import(Array(
  classOf[TaskEngineConfig],
  classOf[XldProductConfiguration],
  classOf[CommandWhitelistProperties],
  classOf[PluginManagerConfig]
))
@ImportResource(
  Array(
    "classpath:spring/task-engine-context.xml"
  )
)
class TaskExecutionEngineBootstrapper(archive: Archive,
                                      repositoryFactory: RepositoryFactory,
                                      stepLogFactory: StepLogFactory,
                                      stepLogRetriever: StepLogRetriever,
                                      taskRepository: ActiveTaskRepository,
                                      pendingTaskRepository: PendingTaskRepository,
                                      workerRepository: TaskExecutionWorkerRepository,
                                      @Qualifier("xlJmsListenerContainerFactory")
                                      containerFactory: JmsListenerContainerFactory[DefaultMessageListenerContainer],
                                      taskerSettings: TaskerSettings,
                                      xldProductConfiguration: ProductConfiguration,
                                      context: GenericApplicationContext) extends CommandLineRunner
  with Logging {
  TaskEngine.setInstance(new TaskEngine(xldProductConfiguration))

  private def initWorkers()(implicit worker: TaskEngine, system: ActorSystem): Unit = {
    TaskExecutionWorker.initialize(
      masters = worker.masters,
      hostnameToActorPathTransformer = worker.hostnameToActorPath,
      repositoryFactory = repositoryFactory,
      stepLogFactory = stepLogFactory,
      stepLogRetriever = stepLogRetriever,
      archive = archive,
      workerName = worker.settings.name,
      workerPublicKey = worker.publicKey,
      configurationHashProvider = configurationHashProvider(),
      activeTaskRepository = taskRepository,
      pendingTaskRepository = pendingTaskRepository,
      workerRepository = workerRepository,
      containerFactory = containerFactory,
      taskerSettings = taskerSettings,
      shouldNotChunk = false
    ).recoverTasks()
    DiscoveryWorker.initialize(system)
  }

  private def registerListeners()(implicit worker: TaskEngine, system: ActorSystem): Unit = {
    new ChangeSetEventPublisher(system, worker.masters)
  }

  private def setupRestEasyDeployment(): Unit = {
    val deployment = new ResteasyDeploymentImpl
    deployment.getProviders.addAll(List(
      new AcceptHeaderSetter("application/xml;xl-exception=binary"),
      new LocalDateParamConverter,
      new DeployitExceptionRethrower,
      new InternalServerErrorClientResponseInterceptor).asJava)
    deployment.setAddCharset(true)
    deployment.start()
  }

  private def configurationHashProvider() =
    ConfigurationHashConfiguration.hashDefaults(InMemoryConfigurationHashProvider)

  override def run(args: String*): Unit = {
    implicit val worker: TaskEngine = TaskEngine.getInstance
    implicit val system: ActorSystem = TaskEnginePekkoBootstrapper.init(worker.config, context)

    setupRestEasyDeployment()
    initWorkers()
    registerListeners()
  }
}

class BeanSupplier[T](bean: T) extends Supplier[T] {
  override def get(): T = bean
}
