package org.apache.pekko

import ai.digital.deploy.task.serdes.TaskPekkoSerializer
import ai.digital.deploy.task.serdes.yaml.DeployStringQuotingChecker
import com.fasterxml.jackson.databind.SerializationFeature
import com.fasterxml.jackson.dataformat.yaml.{YAMLFactory, YAMLGenerator}
import org.apache.pekko.actor.ExtendedActorSystem
import org.apache.pekko.event.Logging
import org.apache.pekko.serialization.jackson.{JacksonObjectMapperProvider, JacksonSerializer}

final class JacksonYamlSerializer(system: ExtendedActorSystem, bindingName: String)
  extends JacksonSerializer(
    system,
    bindingName: String,
    JacksonObjectMapperProvider(system).getOrCreate(
      bindingName,
      Some(
        new YAMLFactory()
          .disable(YAMLGenerator.Feature.USE_NATIVE_TYPE_ID)
          .disable(YAMLGenerator.Feature.USE_NATIVE_OBJECT_ID)
          .disable(YAMLGenerator.Feature.WRITE_DOC_START_MARKER)
          .enable(YAMLGenerator.Feature.MINIMIZE_QUOTES)
          .rebuild()
          .stringQuotingChecker(DeployStringQuotingChecker)
          .build()
      )
    )
      .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
      .deactivateDefaultTyping()
  ) with TaskPekkoSerializer {

  private val log = Logging.withMarker(system, classOf[JacksonSerializer])
  private val conf = JacksonObjectMapperProvider.configForBinding(bindingName, system.settings.config)
  private val isDebugEnabled = conf.getBoolean("verbose-debug-logging") && log.isDebugEnabled

  // TODO recheck ENG-7355
  private val serializationBindingsCheckedOkField = classOf[JacksonSerializer].getDeclaredField("serializationBindingsCheckedOk")
  serializationBindingsCheckedOkField.setAccessible(true)
  serializationBindingsCheckedOkField.setBoolean(this, true)
  serializationBindingsCheckedOkField.setAccessible(false)

  override def initExternalWorker(conf: TaskPekkoSerializer.Conf, createWorkDir: Boolean = true): Unit = {}

  override def toBinary(obj: AnyRef): Array[Byte] = {
    val bytes = super.toBinary(obj)
    if (isDebugEnabled) {
      log.debug("Serialization of {} to {}", obj.getClass, new String(bytes))
    }
    bytes
  }

  override def fromBinary(bytes: Array[Byte], manifest: String): AnyRef = {
    if (log.isDebugEnabled) {
      log.debug("Deserialization of {} to {}", manifest, new String(bytes))
    }
    super.fromBinary(bytes, manifest)
  }
}

