/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.camera2.internal;

import android.hardware.camera2.CameraCharacteristics;
import android.os.Build;
import android.text.TextUtils;
import androidx.annotation.DoNotInline;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.camera.camera2.internal.compat.CameraCharacteristicsCompat;
import androidx.camera.camera2.internal.compat.params.DynamicRangeConversions;
import androidx.camera.camera2.internal.compat.params.DynamicRangesCompat;
import androidx.camera.core.DynamicRange;
import androidx.camera.core.Logger;
import androidx.camera.core.impl.AttachedSurfaceInfo;
import androidx.camera.core.impl.UseCaseConfig;
import androidx.core.util.Preconditions;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

@RequiresApi(value=21)
final class DynamicRangeResolver {
    private static final String TAG = "DynamicRangeResolver";
    private final CameraCharacteristicsCompat mCharacteristics;
    private final DynamicRangesCompat mDynamicRangesInfo;
    private final boolean mIs10BitSupported;

    DynamicRangeResolver(@NonNull CameraCharacteristicsCompat characteristics) {
        this.mCharacteristics = characteristics;
        this.mDynamicRangesInfo = DynamicRangesCompat.fromCameraCharacteristics(characteristics);
        int[] availableCapabilities = (int[])this.mCharacteristics.get(CameraCharacteristics.REQUEST_AVAILABLE_CAPABILITIES);
        boolean is10BitSupported = false;
        if (availableCapabilities != null) {
            for (int capability : availableCapabilities) {
                if (capability != 18) continue;
                is10BitSupported = true;
                break;
            }
        }
        this.mIs10BitSupported = is10BitSupported;
    }

    boolean is10BitDynamicRangeSupported() {
        return this.mIs10BitSupported;
    }

    Map<UseCaseConfig<?>, DynamicRange> resolveAndValidateDynamicRanges(@NonNull List<AttachedSurfaceInfo> existingSurfaces, @NonNull List<UseCaseConfig<?>> newUseCaseConfigs, @NonNull List<Integer> useCasePriorityOrder) {
        LinkedHashSet<DynamicRange> orderedExistingDynamicRanges = new LinkedHashSet<DynamicRange>();
        for (AttachedSurfaceInfo asi : existingSurfaces) {
            orderedExistingDynamicRanges.add(asi.getDynamicRange());
        }
        Set<DynamicRange> supportedDynamicRanges = this.mDynamicRangesInfo.getSupportedDynamicRanges();
        HashSet<DynamicRange> combinedConstraints = new HashSet<DynamicRange>(supportedDynamicRanges);
        for (DynamicRange dynamicRange : orderedExistingDynamicRanges) {
            DynamicRangeResolver.updateConstraints(combinedConstraints, dynamicRange, this.mDynamicRangesInfo);
        }
        ArrayList orderedFullyDefinedUseCaseConfigs = new ArrayList();
        ArrayList orderedPartiallyDefinedUseCaseConfigs = new ArrayList();
        ArrayList orderedUndefinedUseCaseConfigs = new ArrayList();
        for (int priorityIdx : useCasePriorityOrder) {
            UseCaseConfig<?> config = newUseCaseConfigs.get(priorityIdx);
            DynamicRange requestedDynamicRange = config.getDynamicRange();
            if (DynamicRangeResolver.isFullyUnspecified(requestedDynamicRange)) {
                orderedUndefinedUseCaseConfigs.add(config);
                continue;
            }
            if (DynamicRangeResolver.isPartiallySpecified(requestedDynamicRange)) {
                orderedPartiallyDefinedUseCaseConfigs.add(config);
                continue;
            }
            orderedFullyDefinedUseCaseConfigs.add(config);
        }
        HashMap resolvedDynamicRanges = new HashMap();
        LinkedHashSet<DynamicRange> orderedNewDynamicRanges = new LinkedHashSet<DynamicRange>();
        ArrayList orderedUseCaseConfigs = new ArrayList();
        orderedUseCaseConfigs.addAll(orderedFullyDefinedUseCaseConfigs);
        orderedUseCaseConfigs.addAll(orderedPartiallyDefinedUseCaseConfigs);
        orderedUseCaseConfigs.addAll(orderedUndefinedUseCaseConfigs);
        for (UseCaseConfig useCaseConfig : orderedUseCaseConfigs) {
            DynamicRange resolvedDynamicRange = this.resolveDynamicRangeAndUpdateConstraints(supportedDynamicRanges, orderedExistingDynamicRanges, orderedNewDynamicRanges, useCaseConfig, combinedConstraints);
            resolvedDynamicRanges.put(useCaseConfig, resolvedDynamicRange);
            if (orderedExistingDynamicRanges.contains(resolvedDynamicRange)) continue;
            orderedNewDynamicRanges.add(resolvedDynamicRange);
        }
        return resolvedDynamicRanges;
    }

    private DynamicRange resolveDynamicRangeAndUpdateConstraints(@NonNull Set<DynamicRange> supportedDynamicRanges, @NonNull Set<DynamicRange> orderedExistingDynamicRanges, @NonNull Set<DynamicRange> orderedNewDynamicRanges, @NonNull UseCaseConfig<?> config, @NonNull Set<DynamicRange> outCombinedConstraints) {
        DynamicRange requestedDynamicRange = config.getDynamicRange();
        DynamicRange resolvedDynamicRange = this.resolveDynamicRange(requestedDynamicRange, outCombinedConstraints, orderedExistingDynamicRanges, orderedNewDynamicRanges, config.getTargetName());
        if (resolvedDynamicRange == null) {
            throw new IllegalArgumentException(String.format("Unable to resolve supported dynamic range. The dynamic range may not be supported on the device or may not be allowed concurrently with other attached use cases.\nUse case:\n  %s\nRequested dynamic range:\n  %s\nSupported dynamic ranges:\n  %s\nConstrained set of concurrent dynamic ranges:\n  %s", config.getTargetName(), requestedDynamicRange, TextUtils.join((CharSequence)"\n  ", supportedDynamicRanges), TextUtils.join((CharSequence)"\n  ", outCombinedConstraints)));
        }
        DynamicRangeResolver.updateConstraints(outCombinedConstraints, resolvedDynamicRange, this.mDynamicRangesInfo);
        return resolvedDynamicRange;
    }

    @Nullable
    private DynamicRange resolveDynamicRange(@NonNull DynamicRange requestedDynamicRange, @NonNull Set<DynamicRange> combinedConstraints, @NonNull Set<DynamicRange> orderedExistingDynamicRanges, @NonNull Set<DynamicRange> orderedNewDynamicRanges, @NonNull String rangeOwnerLabel) {
        if (requestedDynamicRange.isFullySpecified()) {
            if (combinedConstraints.contains(requestedDynamicRange)) {
                return requestedDynamicRange;
            }
            return null;
        }
        int requestedEncoding = requestedDynamicRange.getEncoding();
        int requestedBitDepth = requestedDynamicRange.getBitDepth();
        if (requestedEncoding == 1 && requestedBitDepth == 0) {
            if (combinedConstraints.contains(DynamicRange.SDR)) {
                return DynamicRange.SDR;
            }
            return null;
        }
        DynamicRange resolvedDynamicRange = DynamicRangeResolver.findSupportedHdrMatch(requestedDynamicRange, orderedExistingDynamicRanges, combinedConstraints);
        if (resolvedDynamicRange != null) {
            Logger.d((String)TAG, (String)String.format("Resolved dynamic range for use case %s from existing attached surface.\n%s\n->\n%s", rangeOwnerLabel, requestedDynamicRange, resolvedDynamicRange));
            return resolvedDynamicRange;
        }
        resolvedDynamicRange = DynamicRangeResolver.findSupportedHdrMatch(requestedDynamicRange, orderedNewDynamicRanges, combinedConstraints);
        if (resolvedDynamicRange != null) {
            Logger.d((String)TAG, (String)String.format("Resolved dynamic range for use case %s from concurrently bound use case.\n%s\n->\n%s", rangeOwnerLabel, requestedDynamicRange, resolvedDynamicRange));
            return resolvedDynamicRange;
        }
        if (DynamicRangeResolver.canResolveWithinConstraints(requestedDynamicRange, DynamicRange.SDR, combinedConstraints)) {
            Logger.d((String)TAG, (String)String.format("Resolved dynamic range for use case %s to no compatible HDR dynamic ranges.\n%s\n->\n%s", rangeOwnerLabel, requestedDynamicRange, DynamicRange.SDR));
            return DynamicRange.SDR;
        }
        if (requestedEncoding == 2 && (requestedBitDepth == 10 || requestedBitDepth == 0)) {
            LinkedHashSet<DynamicRange> hdrDefaultRanges = new LinkedHashSet<DynamicRange>();
            DynamicRange recommendedRange = null;
            if (Build.VERSION.SDK_INT >= 33 && (recommendedRange = Api33Impl.getRecommended10BitDynamicRange(this.mCharacteristics)) != null) {
                hdrDefaultRanges.add(recommendedRange);
            }
            hdrDefaultRanges.add(DynamicRange.HLG_10_BIT);
            resolvedDynamicRange = DynamicRangeResolver.findSupportedHdrMatch(requestedDynamicRange, hdrDefaultRanges, combinedConstraints);
            if (resolvedDynamicRange != null) {
                Logger.d((String)TAG, (String)String.format("Resolved dynamic range for use case %s from %s 10-bit supported dynamic range.\n%s\n->\n%s", rangeOwnerLabel, Objects.equals(resolvedDynamicRange, recommendedRange) ? "recommended" : "required", requestedDynamicRange, resolvedDynamicRange));
                return resolvedDynamicRange;
            }
        }
        for (DynamicRange candidateRange : combinedConstraints) {
            Preconditions.checkState((boolean)candidateRange.isFullySpecified(), (String)"Candidate dynamic range must be fully specified.");
            if (candidateRange.equals((Object)DynamicRange.SDR) || !DynamicRangeResolver.canResolve(requestedDynamicRange, candidateRange)) continue;
            Logger.d((String)TAG, (String)String.format("Resolved dynamic range for use case %s from validated dynamic range constraints or supported HDR dynamic ranges.\n%s\n->\n%s", rangeOwnerLabel, requestedDynamicRange, candidateRange));
            return candidateRange;
        }
        return null;
    }

    private static void updateConstraints(@NonNull Set<DynamicRange> combinedConstraints, @NonNull DynamicRange newDynamicRange, @NonNull DynamicRangesCompat dynamicRangesInfo) {
        Preconditions.checkState((!combinedConstraints.isEmpty() ? 1 : 0) != 0, (String)"Cannot update already-empty constraints.");
        Set<DynamicRange> newConstraints = dynamicRangesInfo.getDynamicRangeCaptureRequestConstraints(newDynamicRange);
        if (!newConstraints.isEmpty()) {
            HashSet<DynamicRange> previousConstraints = new HashSet<DynamicRange>(combinedConstraints);
            combinedConstraints.retainAll(newConstraints);
            if (combinedConstraints.isEmpty()) {
                throw new IllegalArgumentException(String.format("Constraints of dynamic range cannot be combined with existing constraints.\nDynamic range:\n  %s\nConstraints:\n  %s\nExisting constraints:\n  %s", newDynamicRange, TextUtils.join((CharSequence)"\n  ", newConstraints), TextUtils.join((CharSequence)"\n  ", previousConstraints)));
            }
        }
    }

    @Nullable
    private static DynamicRange findSupportedHdrMatch(@NonNull DynamicRange rangeToMatch, @NonNull Collection<DynamicRange> fullySpecifiedCandidateRanges, @NonNull Set<DynamicRange> constraints) {
        if (rangeToMatch.getEncoding() == 1) {
            return null;
        }
        for (DynamicRange candidateRange : fullySpecifiedCandidateRanges) {
            Preconditions.checkNotNull((Object)candidateRange, (Object)"Fully specified DynamicRange cannot be null.");
            int candidateEncoding = candidateRange.getEncoding();
            Preconditions.checkState((boolean)candidateRange.isFullySpecified(), (String)"Fully specified DynamicRange must have fully defined encoding.");
            if (candidateEncoding == 1 || !DynamicRangeResolver.canResolveWithinConstraints(rangeToMatch, candidateRange, constraints)) continue;
            return candidateRange;
        }
        return null;
    }

    private static boolean isFullyUnspecified(@NonNull DynamicRange dynamicRange) {
        return Objects.equals(dynamicRange, DynamicRange.UNSPECIFIED);
    }

    private static boolean isPartiallySpecified(@NonNull DynamicRange dynamicRange) {
        return dynamicRange.getEncoding() == 2 || dynamicRange.getEncoding() != 0 && dynamicRange.getBitDepth() == 0 || dynamicRange.getEncoding() == 0 && dynamicRange.getBitDepth() != 0;
    }

    private static boolean canResolveWithinConstraints(@NonNull DynamicRange rangeToResolve, @NonNull DynamicRange candidateRange, @NonNull Set<DynamicRange> constraints) {
        if (!constraints.contains(candidateRange)) {
            Logger.d((String)TAG, (String)String.format("Candidate Dynamic range is not within constraints.\nDynamic range to resolve:\n  %s\nCandidate dynamic range:\n  %s", rangeToResolve, candidateRange));
            return false;
        }
        return DynamicRangeResolver.canResolve(rangeToResolve, candidateRange);
    }

    private static boolean canResolve(@NonNull DynamicRange testRange, @NonNull DynamicRange fullySpecifiedRange) {
        Preconditions.checkState((boolean)fullySpecifiedRange.isFullySpecified(), (String)"Fully specified range is not actually fully specified.");
        if (testRange.getEncoding() == 2 && fullySpecifiedRange.getEncoding() == 1) {
            return false;
        }
        if (testRange.getEncoding() != 2 && testRange.getEncoding() != 0 && testRange.getEncoding() != fullySpecifiedRange.getEncoding()) {
            return false;
        }
        return testRange.getBitDepth() == 0 || testRange.getBitDepth() == fullySpecifiedRange.getBitDepth();
    }

    @RequiresApi(value=33)
    static final class Api33Impl {
        private Api33Impl() {
        }

        @DoNotInline
        @Nullable
        static DynamicRange getRecommended10BitDynamicRange(@NonNull CameraCharacteristicsCompat characteristics) {
            Long recommendedProfile = (Long)characteristics.get(CameraCharacteristics.REQUEST_RECOMMENDED_TEN_BIT_DYNAMIC_RANGE_PROFILE);
            if (recommendedProfile != null) {
                return DynamicRangeConversions.profileToDynamicRange(recommendedProfile);
            }
            return null;
        }
    }
}

