/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.core.impl.utils;

import android.location.Location;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.camera.core.ImageProxy;
import androidx.camera.core.Logger;
import androidx.exifinterface.media.ExifInterface;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;

@RequiresApi(value=21)
public final class Exif {
    public static final long INVALID_TIMESTAMP = -1L;
    private static final String TAG = Exif.class.getSimpleName();
    private static final ThreadLocal<SimpleDateFormat> DATE_FORMAT = new ThreadLocal<SimpleDateFormat>(){

        @Override
        public SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yyyy:MM:dd", Locale.US);
        }
    };
    private static final ThreadLocal<SimpleDateFormat> TIME_FORMAT = new ThreadLocal<SimpleDateFormat>(){

        @Override
        public SimpleDateFormat initialValue() {
            return new SimpleDateFormat("HH:mm:ss", Locale.US);
        }
    };
    private static final ThreadLocal<SimpleDateFormat> DATETIME_FORMAT = new ThreadLocal<SimpleDateFormat>(){

        @Override
        public SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yyyy:MM:dd HH:mm:ss", Locale.US);
        }
    };
    private static final String KILOMETERS_PER_HOUR = "K";
    private static final String MILES_PER_HOUR = "M";
    private static final String KNOTS = "N";
    private static final List<String> ALL_EXIF_TAGS = Exif.getAllExifTags();
    private static final List<String> DO_NOT_COPY_EXIF_TAGS = Arrays.asList("ImageWidth", "ImageLength", "PixelXDimension", "PixelYDimension", "Compression", "JPEGInterchangeFormat", "JPEGInterchangeFormatLength", "ThumbnailImageLength", "ThumbnailImageWidth", "ThumbnailOrientation");
    private final ExifInterface mExifInterface;
    private boolean mRemoveTimestamp = false;

    private Exif(ExifInterface exifInterface) {
        this.mExifInterface = exifInterface;
    }

    @NonNull
    public static Exif createFromFile(@NonNull File file) throws IOException {
        return Exif.createFromFileString(file.toString());
    }

    @NonNull
    public static Exif createFromImageProxy(@NonNull ImageProxy imageProxy) throws IOException {
        ByteBuffer buffer = imageProxy.getPlanes()[0].getBuffer();
        buffer.rewind();
        byte[] data = new byte[buffer.capacity()];
        buffer.get(data);
        ByteArrayInputStream inputStream = new ByteArrayInputStream(data);
        return Exif.createFromInputStream(inputStream);
    }

    @NonNull
    public static Exif createFromFileString(@NonNull String filePath) throws IOException {
        return new Exif(new ExifInterface(filePath));
    }

    @NonNull
    public static Exif createFromInputStream(@NonNull InputStream is) throws IOException {
        return new Exif(new ExifInterface(is));
    }

    private static String convertToExifDateTime(long timestamp) {
        return DATETIME_FORMAT.get().format(new Date(timestamp));
    }

    private static Date convertFromExifDateTime(String dateTime) throws ParseException {
        return DATETIME_FORMAT.get().parse(dateTime);
    }

    private static Date convertFromExifDate(String date) throws ParseException {
        return DATE_FORMAT.get().parse(date);
    }

    private static Date convertFromExifTime(String time) throws ParseException {
        return TIME_FORMAT.get().parse(time);
    }

    public void save() throws IOException {
        if (!this.mRemoveTimestamp) {
            this.attachLastModifiedTimestamp();
        }
        this.mExifInterface.saveAttributes();
    }

    public void copyToCroppedImage(@NonNull Exif croppedExif) {
        ArrayList<String> exifTags = new ArrayList<String>(ALL_EXIF_TAGS);
        exifTags.removeAll(DO_NOT_COPY_EXIF_TAGS);
        for (String tag : exifTags) {
            String originalValue = this.mExifInterface.getAttribute(tag);
            if (originalValue == null) continue;
            croppedExif.mExifInterface.setAttribute(tag, originalValue);
        }
    }

    public String toString() {
        return String.format(Locale.ENGLISH, "Exif{width=%s, height=%s, rotation=%d, isFlippedVertically=%s, isFlippedHorizontally=%s, location=%s, timestamp=%s, description=%s}", this.getWidth(), this.getHeight(), this.getRotation(), this.isFlippedVertically(), this.isFlippedHorizontally(), this.getLocation(), this.getTimestamp(), this.getDescription());
    }

    public int getOrientation() {
        return this.mExifInterface.getAttributeInt("Orientation", 0);
    }

    public void setOrientation(int orientation) {
        this.mExifInterface.setAttribute("Orientation", String.valueOf(orientation));
    }

    public int getWidth() {
        return this.mExifInterface.getAttributeInt("ImageWidth", 0);
    }

    public int getHeight() {
        return this.mExifInterface.getAttributeInt("ImageLength", 0);
    }

    @Nullable
    public String getDescription() {
        return this.mExifInterface.getAttribute("ImageDescription");
    }

    public void setDescription(@Nullable String description) {
        this.mExifInterface.setAttribute("ImageDescription", description);
    }

    public int getRotation() {
        switch (this.getOrientation()) {
            case 2: {
                return 0;
            }
            case 3: {
                return 180;
            }
            case 4: {
                return 180;
            }
            case 5: {
                return 270;
            }
            case 6: {
                return 90;
            }
            case 7: {
                return 90;
            }
            case 8: {
                return 270;
            }
        }
        return 0;
    }

    public boolean isFlippedVertically() {
        switch (this.getOrientation()) {
            case 2: {
                return false;
            }
            case 3: {
                return false;
            }
            case 4: {
                return true;
            }
            case 5: {
                return true;
            }
            case 6: {
                return false;
            }
            case 7: {
                return true;
            }
            case 8: {
                return false;
            }
        }
        return false;
    }

    public boolean isFlippedHorizontally() {
        switch (this.getOrientation()) {
            case 2: {
                return true;
            }
            case 3: {
                return false;
            }
            case 4: {
                return false;
            }
            case 5: {
                return false;
            }
            case 6: {
                return false;
            }
            case 7: {
                return false;
            }
            case 8: {
                return false;
            }
        }
        return false;
    }

    private void attachLastModifiedTimestamp() {
        long now = System.currentTimeMillis();
        String datetime = Exif.convertToExifDateTime(now);
        this.mExifInterface.setAttribute("DateTime", datetime);
        try {
            String subsec = Long.toString(now - Exif.convertFromExifDateTime(datetime).getTime());
            this.mExifInterface.setAttribute("SubSecTime", subsec);
        }
        catch (ParseException parseException) {
            // empty catch block
        }
    }

    public long getLastModifiedTimestamp() {
        long timestamp = this.parseTimestamp(this.mExifInterface.getAttribute("DateTime"));
        if (timestamp == -1L) {
            return -1L;
        }
        String subSecs = this.mExifInterface.getAttribute("SubSecTime");
        if (subSecs != null) {
            try {
                long sub;
                for (sub = Long.parseLong(subSecs); sub > 1000L; sub /= 10L) {
                }
                timestamp += sub;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return timestamp;
    }

    public long getTimestamp() {
        long timestamp = this.parseTimestamp(this.mExifInterface.getAttribute("DateTimeOriginal"));
        if (timestamp == -1L) {
            return -1L;
        }
        String subSecs = this.mExifInterface.getAttribute("SubSecTimeOriginal");
        if (subSecs != null) {
            try {
                long sub;
                for (sub = Long.parseLong(subSecs); sub > 1000L; sub /= 10L) {
                }
                timestamp += sub;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return timestamp;
    }

    @Nullable
    public Location getLocation() {
        String provider = this.mExifInterface.getAttribute("GPSProcessingMethod");
        double[] latlng = this.mExifInterface.getLatLong();
        double altitude = this.mExifInterface.getAltitude(0.0);
        double speed = this.mExifInterface.getAttributeDouble("GPSSpeed", 0.0);
        String speedRef = this.mExifInterface.getAttribute("GPSSpeedRef");
        speedRef = speedRef == null ? KILOMETERS_PER_HOUR : speedRef;
        long timestamp = this.parseTimestamp(this.mExifInterface.getAttribute("GPSDateStamp"), this.mExifInterface.getAttribute("GPSTimeStamp"));
        if (latlng == null) {
            return null;
        }
        if (provider == null) {
            provider = TAG;
        }
        Location location = new Location(provider);
        location.setLatitude(latlng[0]);
        location.setLongitude(latlng[1]);
        if (altitude != 0.0) {
            location.setAltitude(altitude);
        }
        if (speed != 0.0) {
            switch (speedRef) {
                case "M": {
                    speed = Speed.fromMilesPerHour(speed).toMetersPerSecond();
                    break;
                }
                case "N": {
                    speed = Speed.fromKnots(speed).toMetersPerSecond();
                    break;
                }
                default: {
                    speed = Speed.fromKilometersPerHour(speed).toMetersPerSecond();
                }
            }
            location.setSpeed((float)speed);
        }
        if (timestamp != -1L) {
            location.setTime(timestamp);
        }
        return location;
    }

    public void rotate(int degrees) {
        if (degrees % 90 != 0) {
            Logger.w(TAG, String.format(Locale.US, "Can only rotate in right angles (eg. 0, 90, 180, 270). %d is unsupported.", degrees));
            this.mExifInterface.setAttribute("Orientation", String.valueOf(0));
            return;
        }
        degrees %= 360;
        int orientation = this.getOrientation();
        block18: while (degrees < 0) {
            degrees += 90;
            switch (orientation) {
                case 2: {
                    orientation = 5;
                    continue block18;
                }
                case 3: {
                    orientation = 6;
                    continue block18;
                }
                case 4: {
                    orientation = 7;
                    continue block18;
                }
                case 5: {
                    orientation = 4;
                    continue block18;
                }
                case 6: {
                    orientation = 1;
                    continue block18;
                }
                case 7: {
                    orientation = 2;
                    continue block18;
                }
                case 8: {
                    orientation = 6;
                    continue block18;
                }
            }
            orientation = 8;
        }
        block19: while (degrees > 0) {
            degrees -= 90;
            switch (orientation) {
                case 2: {
                    orientation = 7;
                    continue block19;
                }
                case 3: {
                    orientation = 8;
                    continue block19;
                }
                case 4: {
                    orientation = 5;
                    continue block19;
                }
                case 5: {
                    orientation = 2;
                    continue block19;
                }
                case 6: {
                    orientation = 3;
                    continue block19;
                }
                case 7: {
                    orientation = 4;
                    continue block19;
                }
                case 8: {
                    orientation = 1;
                    continue block19;
                }
            }
            orientation = 6;
        }
        this.mExifInterface.setAttribute("Orientation", String.valueOf(orientation));
    }

    public void flipVertically() {
        int orientation;
        switch (this.getOrientation()) {
            case 2: {
                orientation = 3;
                break;
            }
            case 3: {
                orientation = 2;
                break;
            }
            case 4: {
                orientation = 1;
                break;
            }
            case 5: {
                orientation = 8;
                break;
            }
            case 6: {
                orientation = 7;
                break;
            }
            case 7: {
                orientation = 6;
                break;
            }
            case 8: {
                orientation = 5;
                break;
            }
            default: {
                orientation = 4;
            }
        }
        this.mExifInterface.setAttribute("Orientation", String.valueOf(orientation));
    }

    public void flipHorizontally() {
        int orientation;
        switch (this.getOrientation()) {
            case 2: {
                orientation = 1;
                break;
            }
            case 3: {
                orientation = 4;
                break;
            }
            case 4: {
                orientation = 3;
                break;
            }
            case 5: {
                orientation = 6;
                break;
            }
            case 6: {
                orientation = 5;
                break;
            }
            case 7: {
                orientation = 8;
                break;
            }
            case 8: {
                orientation = 7;
                break;
            }
            default: {
                orientation = 2;
            }
        }
        this.mExifInterface.setAttribute("Orientation", String.valueOf(orientation));
    }

    public void attachTimestamp() {
        long now = System.currentTimeMillis();
        String datetime = Exif.convertToExifDateTime(now);
        this.mExifInterface.setAttribute("DateTimeOriginal", datetime);
        this.mExifInterface.setAttribute("DateTimeDigitized", datetime);
        try {
            String subsec = Long.toString(now - Exif.convertFromExifDateTime(datetime).getTime());
            this.mExifInterface.setAttribute("SubSecTimeOriginal", subsec);
            this.mExifInterface.setAttribute("SubSecTimeDigitized", subsec);
        }
        catch (ParseException parseException) {
            // empty catch block
        }
        this.mRemoveTimestamp = false;
    }

    public void removeTimestamp() {
        this.mExifInterface.setAttribute("DateTime", null);
        this.mExifInterface.setAttribute("DateTimeOriginal", null);
        this.mExifInterface.setAttribute("DateTimeDigitized", null);
        this.mExifInterface.setAttribute("SubSecTime", null);
        this.mExifInterface.setAttribute("SubSecTimeOriginal", null);
        this.mExifInterface.setAttribute("SubSecTimeDigitized", null);
        this.mRemoveTimestamp = true;
    }

    public void attachLocation(@NonNull Location location) {
        this.mExifInterface.setGpsInfo(location);
    }

    public void removeLocation() {
        this.mExifInterface.setAttribute("GPSProcessingMethod", null);
        this.mExifInterface.setAttribute("GPSLatitude", null);
        this.mExifInterface.setAttribute("GPSLatitudeRef", null);
        this.mExifInterface.setAttribute("GPSLongitude", null);
        this.mExifInterface.setAttribute("GPSLongitudeRef", null);
        this.mExifInterface.setAttribute("GPSAltitude", null);
        this.mExifInterface.setAttribute("GPSAltitudeRef", null);
        this.mExifInterface.setAttribute("GPSSpeed", null);
        this.mExifInterface.setAttribute("GPSSpeedRef", null);
        this.mExifInterface.setAttribute("GPSDateStamp", null);
        this.mExifInterface.setAttribute("GPSTimeStamp", null);
    }

    private long parseTimestamp(@Nullable String date, @Nullable String time) {
        if (date == null && time == null) {
            return -1L;
        }
        if (time == null) {
            try {
                return Exif.convertFromExifDate(date).getTime();
            }
            catch (ParseException e) {
                return -1L;
            }
        }
        if (date == null) {
            try {
                return Exif.convertFromExifTime(time).getTime();
            }
            catch (ParseException e) {
                return -1L;
            }
        }
        return this.parseTimestamp(date + " " + time);
    }

    private long parseTimestamp(@Nullable String datetime) {
        if (datetime == null) {
            return -1L;
        }
        try {
            return Exif.convertFromExifDateTime(datetime).getTime();
        }
        catch (ParseException e) {
            return -1L;
        }
    }

    @NonNull
    public static List<String> getAllExifTags() {
        return Arrays.asList("ImageWidth", "ImageLength", "BitsPerSample", "Compression", "PhotometricInterpretation", "Orientation", "SamplesPerPixel", "PlanarConfiguration", "YCbCrSubSampling", "YCbCrPositioning", "XResolution", "YResolution", "ResolutionUnit", "StripOffsets", "RowsPerStrip", "StripByteCounts", "JPEGInterchangeFormat", "JPEGInterchangeFormatLength", "TransferFunction", "WhitePoint", "PrimaryChromaticities", "YCbCrCoefficients", "ReferenceBlackWhite", "DateTime", "ImageDescription", "Make", "Model", "Software", "Artist", "Copyright", "ExifVersion", "FlashpixVersion", "ColorSpace", "Gamma", "PixelXDimension", "PixelYDimension", "ComponentsConfiguration", "CompressedBitsPerPixel", "MakerNote", "UserComment", "RelatedSoundFile", "DateTimeOriginal", "DateTimeDigitized", "OffsetTime", "OffsetTimeOriginal", "OffsetTimeDigitized", "SubSecTime", "SubSecTimeOriginal", "SubSecTimeDigitized", "ExposureTime", "FNumber", "ExposureProgram", "SpectralSensitivity", "PhotographicSensitivity", "OECF", "SensitivityType", "StandardOutputSensitivity", "RecommendedExposureIndex", "ISOSpeed", "ISOSpeedLatitudeyyy", "ISOSpeedLatitudezzz", "ShutterSpeedValue", "ApertureValue", "BrightnessValue", "ExposureBiasValue", "MaxApertureValue", "SubjectDistance", "MeteringMode", "LightSource", "Flash", "SubjectArea", "FocalLength", "FlashEnergy", "SpatialFrequencyResponse", "FocalPlaneXResolution", "FocalPlaneYResolution", "FocalPlaneResolutionUnit", "SubjectLocation", "ExposureIndex", "SensingMethod", "FileSource", "SceneType", "CFAPattern", "CustomRendered", "ExposureMode", "WhiteBalance", "DigitalZoomRatio", "FocalLengthIn35mmFilm", "SceneCaptureType", "GainControl", "Contrast", "Saturation", "Sharpness", "DeviceSettingDescription", "SubjectDistanceRange", "ImageUniqueID", "CameraOwnerName", "BodySerialNumber", "LensSpecification", "LensMake", "LensModel", "LensSerialNumber", "GPSVersionID", "GPSLatitudeRef", "GPSLatitude", "GPSLongitudeRef", "GPSLongitude", "GPSAltitudeRef", "GPSAltitude", "GPSTimeStamp", "GPSSatellites", "GPSStatus", "GPSMeasureMode", "GPSDOP", "GPSSpeedRef", "GPSSpeed", "GPSTrackRef", "GPSTrack", "GPSImgDirectionRef", "GPSImgDirection", "GPSMapDatum", "GPSDestLatitudeRef", "GPSDestLatitude", "GPSDestLongitudeRef", "GPSDestLongitude", "GPSDestBearingRef", "GPSDestBearing", "GPSDestDistanceRef", "GPSDestDistance", "GPSProcessingMethod", "GPSAreaInformation", "GPSDateStamp", "GPSDifferential", "GPSHPositioningError", "InteroperabilityIndex", "ThumbnailImageLength", "ThumbnailImageWidth", "ThumbnailOrientation", "DNGVersion", "DefaultCropSize", "ThumbnailImage", "PreviewImageStart", "PreviewImageLength", "AspectFrame", "SensorBottomBorder", "SensorLeftBorder", "SensorRightBorder", "SensorTopBorder", "ISO", "JpgFromRaw", "Xmp", "NewSubfileType", "SubfileType");
    }

    private static final class Speed {
        private Speed() {
        }

        static Converter fromKilometersPerHour(double kph) {
            return new Converter(kph * 0.621371);
        }

        static Converter fromMetersPerSecond(double mps) {
            return new Converter(mps * 2.23694);
        }

        static Converter fromMilesPerHour(double mph) {
            return new Converter(mph);
        }

        static Converter fromKnots(double knots) {
            return new Converter(knots * 1.15078);
        }

        static final class Converter {
            final double mMph;

            Converter(double mph) {
                this.mMph = mph;
            }

            double toKilometersPerHour() {
                return this.mMph / 0.621371;
            }

            double toMilesPerHour() {
                return this.mMph;
            }

            double toKnots() {
                return this.mMph / 1.15078;
            }

            double toMetersPerSecond() {
                return this.mMph / 2.23694;
            }
        }
    }
}

