/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.core.processing;

import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.RectF;
import android.util.Size;
import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.camera.core.Logger;
import androidx.camera.core.ProcessingException;
import androidx.camera.core.SurfaceOutput;
import androidx.camera.core.SurfaceRequest;
import androidx.camera.core.impl.CameraInternal;
import androidx.camera.core.impl.StreamSpec;
import androidx.camera.core.impl.utils.Threads;
import androidx.camera.core.impl.utils.TransformUtils;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.core.impl.utils.futures.FutureCallback;
import androidx.camera.core.impl.utils.futures.Futures;
import androidx.camera.core.processing.AutoValue_SurfaceProcessorNode_In;
import androidx.camera.core.processing.AutoValue_SurfaceProcessorNode_OutConfig;
import androidx.camera.core.processing.Node;
import androidx.camera.core.processing.SurfaceEdge;
import androidx.camera.core.processing.SurfaceProcessorInternal;
import androidx.core.util.Preconditions;
import com.google.auto.value.AutoValue;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

@RequiresApi(api=21)
public class SurfaceProcessorNode
implements Node<In, Out> {
    private static final String TAG = "SurfaceProcessorNode";
    @NonNull
    final SurfaceProcessorInternal mSurfaceProcessor;
    @NonNull
    final CameraInternal mCameraInternal;
    @Nullable
    private Out mOutput;
    @Nullable
    private In mInput;

    public SurfaceProcessorNode(@NonNull CameraInternal cameraInternal, @NonNull SurfaceProcessorInternal surfaceProcessor) {
        this.mCameraInternal = cameraInternal;
        this.mSurfaceProcessor = surfaceProcessor;
    }

    @Override
    @NonNull
    @MainThread
    public Out transform(@NonNull In input) {
        Threads.checkMainThread();
        this.mInput = input;
        this.mOutput = new Out();
        SurfaceEdge inputSurface = input.getSurfaceEdge();
        for (OutConfig config : input.getOutConfigs()) {
            this.mOutput.put(config, this.transformSingleOutput(inputSurface, config));
        }
        this.sendSurfaceRequest(inputSurface, this.mOutput);
        this.sendSurfaceOutputs(inputSurface, this.mOutput);
        return this.mOutput;
    }

    @NonNull
    private SurfaceEdge transformSingleOutput(@NonNull SurfaceEdge input, @NonNull OutConfig outConfig) {
        Rect cropRect = outConfig.getCropRect();
        int rotationDegrees = outConfig.getRotationDegrees();
        boolean mirroring = outConfig.getMirroring();
        Matrix sensorToBufferTransform = new Matrix(input.getSensorToBufferTransform());
        Matrix imageTransform = TransformUtils.getRectToRect(new RectF(cropRect), TransformUtils.sizeToRectF(outConfig.getSize()), rotationDegrees, mirroring);
        sensorToBufferTransform.postConcat(imageTransform);
        Size rotatedCropSize = TransformUtils.getRotatedSize(cropRect, rotationDegrees);
        Preconditions.checkArgument((boolean)TransformUtils.isAspectRatioMatchingWithRoundingError(rotatedCropSize, outConfig.getSize()));
        StreamSpec streamSpec = input.getStreamSpec().toBuilder().setResolution(outConfig.getSize()).build();
        SurfaceEdge outputSurface = new SurfaceEdge(outConfig.getTargets(), outConfig.getFormat(), streamSpec, sensorToBufferTransform, false, TransformUtils.sizeToRect(outConfig.getSize()), input.getRotationDegrees() - rotationDegrees, -1, input.getMirroring() != mirroring);
        return outputSurface;
    }

    private void sendSurfaceRequest(@NonNull SurfaceEdge input, @NonNull Map<OutConfig, SurfaceEdge> outputs) {
        SurfaceRequest surfaceRequest = input.createSurfaceRequest(this.mCameraInternal);
        this.setUpRotationUpdates(surfaceRequest, outputs);
        try {
            this.mSurfaceProcessor.onInputSurface(surfaceRequest);
        }
        catch (ProcessingException e) {
            Logger.e(TAG, "Failed to send SurfaceRequest to SurfaceProcessor.", e);
        }
    }

    private void sendSurfaceOutputs(@NonNull SurfaceEdge input, @NonNull Map<OutConfig, SurfaceEdge> outputs) {
        for (Map.Entry<OutConfig, SurfaceEdge> output : outputs.entrySet()) {
            this.createAndSendSurfaceOutput(input, output);
            output.getValue().addOnInvalidatedListener(() -> this.createAndSendSurfaceOutput(input, output));
        }
    }

    private void createAndSendSurfaceOutput(@NonNull SurfaceEdge input, Map.Entry<OutConfig, SurfaceEdge> output) {
        ListenableFuture<SurfaceOutput> future = output.getValue().createSurfaceOutputFuture(input.getStreamSpec().getResolution(), output.getKey().getFormat(), output.getKey().getCropRect(), output.getKey().getRotationDegrees(), output.getKey().getMirroring(), input.hasCameraTransform() ? this.mCameraInternal : null);
        Futures.addCallback(future, new FutureCallback<SurfaceOutput>(){

            @Override
            public void onSuccess(@Nullable SurfaceOutput output) {
                Preconditions.checkNotNull((Object)output);
                try {
                    SurfaceProcessorNode.this.mSurfaceProcessor.onOutputSurface(output);
                }
                catch (ProcessingException e) {
                    Logger.e(SurfaceProcessorNode.TAG, "Failed to send SurfaceOutput to SurfaceProcessor.", e);
                }
            }

            @Override
            public void onFailure(@NonNull Throwable t) {
                Logger.w(SurfaceProcessorNode.TAG, "Downstream node failed to provide Surface.", t);
            }
        }, CameraXExecutors.mainThreadExecutor());
    }

    void setUpRotationUpdates(@NonNull SurfaceRequest inputSurfaceRequest, @NonNull Map<OutConfig, SurfaceEdge> outputs) {
        inputSurfaceRequest.setTransformationInfoListener(CameraXExecutors.mainThreadExecutor(), info -> {
            for (Map.Entry output : outputs.entrySet()) {
                int rotationDegrees = info.getRotationDegrees() - ((OutConfig)output.getKey()).getRotationDegrees();
                if (((OutConfig)output.getKey()).getMirroring()) {
                    rotationDegrees = -rotationDegrees;
                }
                rotationDegrees = TransformUtils.within360(rotationDegrees);
                ((SurfaceEdge)output.getValue()).updateTransformation(rotationDegrees, -1);
            }
        });
    }

    @Override
    public void release() {
        this.mSurfaceProcessor.release();
        SurfaceProcessorNode.runOnMainThread(() -> {
            if (this.mOutput != null) {
                for (SurfaceEdge surface : this.mOutput.values()) {
                    surface.close();
                }
            }
        });
    }

    private static void runOnMainThread(@NonNull Runnable runnable) {
        if (Threads.isMainThread()) {
            runnable.run();
        } else {
            CameraXExecutors.mainThreadExecutor().execute(runnable);
        }
    }

    @NonNull
    public SurfaceProcessorInternal getSurfaceProcessor() {
        return this.mSurfaceProcessor;
    }

    @AutoValue
    public static abstract class In {
        @NonNull
        public abstract SurfaceEdge getSurfaceEdge();

        @NonNull
        public abstract List<OutConfig> getOutConfigs();

        @NonNull
        public static In of(@NonNull SurfaceEdge edge, @NonNull List<OutConfig> configs) {
            return new AutoValue_SurfaceProcessorNode_In(edge, configs);
        }
    }

    public static class Out
    extends HashMap<OutConfig, SurfaceEdge> {
    }

    @AutoValue
    public static abstract class OutConfig {
        @NonNull
        abstract UUID getUuid();

        public abstract int getTargets();

        public abstract int getFormat();

        @NonNull
        public abstract Rect getCropRect();

        @NonNull
        public abstract Size getSize();

        public abstract int getRotationDegrees();

        public abstract boolean getMirroring();

        @NonNull
        public static OutConfig of(@NonNull SurfaceEdge inputEdge) {
            return OutConfig.of(inputEdge.getTargets(), inputEdge.getFormat(), inputEdge.getCropRect(), TransformUtils.getRotatedSize(inputEdge.getCropRect(), inputEdge.getRotationDegrees()), inputEdge.getRotationDegrees(), inputEdge.getMirroring());
        }

        @NonNull
        public static OutConfig of(int targets, int format, @NonNull Rect cropRect, @NonNull Size size, int rotationDegrees, boolean mirroring) {
            return new AutoValue_SurfaceProcessorNode_OutConfig(UUID.randomUUID(), targets, format, cropRect, size, rotationDegrees, mirroring);
        }
    }
}

